//**************************************************************************************************
//                                          ChoUnits.cpp                                           *
//                                         --------------                                          *
// Started     : 2004-03-27                                                                        *
// Last Update : 2020-09-22                                                                        *
// Copyright   : (C) 2004-2020 MSWaters                                                            *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#include "ChoUnits.hpp"

//**************************************************************************************************
// Constructor.

ChoUnits::ChoUnits( void ) : UnitsBase( ), wxChoice( )
{
  bClear( );
}

//**************************************************************************************************
// Destructor.

ChoUnits::~ChoUnits( )
{
}

//**************************************************************************************************
// Create an instance of this object.
//
// Argument List :
//   poWin  - The parent window
//   oWinID - The window identifier
//   iWidth - The width of the choice control in pixels
//
// Return Values :
//   true  - Success
//   false - Failure

bool  ChoUnits::bCreate( wxWindow * poWin, wxWindowID oWinID, int iWidth )
{
  // Check if the object has already been created
  if( bIsCreated( ) )                                                               return( true );

  // Create the object
  if( ! wxChoice::Create( poWin, oWinID, wxDefaultPosition, wxSize( iWidth, GUI_CTRL_HT+2 ) ) )
                                                                                    return( false );
  // Set the units
  if( ! bSetUnitsType( m_eUnitsType ) )                                             return( false );
  if( ! bSetUnits( m_osDefUnits ) )                                                 return( false );

  return( true );
}

//**************************************************************************************************
// Clear object attributes.
//
// Return Values :
//   Success - true
//   Failure - false

bool  ChoUnits::bClear( void )
{
  m_eUnitsType = eUNITS_NONE;
  m_osDefUnits = "Units";

  if( bIsCreated( ) ) ChoUnits::Clear( );

  return( true );
}

//**************************************************************************************************
// Set the type of units to be displayed.
//
// Argument List :
//   eUType - The type of units to be displayed
//
// Return Values :
//   Success - true
//   Failure - false

bool  ChoUnits::bSetUnitsType( eTypeUnits eUType )
{
  int  i1;

  if( ! bIsCreated( ) )                               return( false );
  if( eUType==m_eUnitsType && !ChoUnits::IsEmpty( ) ) return( true );

  // Load the new choice items into the control
  switch( eUType )
  {
    case eUNITS_CAP :   // Capacitance
      ChoUnits::Clear( ); // Delete existing choice items from the list
      ChoUnits::Append( "F"      , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "mF"     , new wxStringClientData(  "-3" ) );
      ChoUnits::Append( "uF"     , new wxStringClientData(  "-6" ) );
      ChoUnits::Append( "nF"     , new wxStringClientData(  "-9" ) );
      ChoUnits::Append( "pF"     , new wxStringClientData( "-12" ) );
      m_osDefUnits = "uF";
      break;

    case eUNITS_IND :   // Inductance
      ChoUnits::Clear( );
      ChoUnits::Append( "H"      , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "mH"     , new wxStringClientData(  "-3" ) );
      ChoUnits::Append( "uH"     , new wxStringClientData(  "-6" ) );
      m_osDefUnits = "mH";
      break;

    case eUNITS_RES :   // Resistance
      ChoUnits::Clear( );
      ChoUnits::Append( "GOhm"   , new wxStringClientData(   "9" ) );
      ChoUnits::Append( "MOhm"   , new wxStringClientData(   "6" ) );
      ChoUnits::Append( "kOhm"   , new wxStringClientData(   "3" ) );
      ChoUnits::Append( "Ohm"    , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "mOhm"   , new wxStringClientData(  "-3" ) );
      ChoUnits::Append( "uOhm"   , new wxStringClientData(  "-6" ) );
      ChoUnits::Append( "nOhm"   , new wxStringClientData(  "-9" ) );
      m_osDefUnits = "kOhm";
      break;

    case eUNITS_COND :  // Conductance
      ChoUnits::Clear( );
      ChoUnits::Append( "kS"     , new wxStringClientData(   "3" ) );
      ChoUnits::Append( "S"      , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "mS"     , new wxStringClientData(  "-3" ) );
      ChoUnits::Append( "uS"     , new wxStringClientData(  "-6" ) );
      ChoUnits::Append( "nS"     , new wxStringClientData(  "-9" ) );
      ChoUnits::Append( "pS"     , new wxStringClientData( "-12" ) );
      ChoUnits::Append( "fS"     , new wxStringClientData( "-15" ) );
      m_osDefUnits = "mS";
      break;

    case eUNITS_VOLT :  // Voltage
      ChoUnits::Clear( );
      ChoUnits::Append( "MV"     , new wxStringClientData(   "6" ) );
      ChoUnits::Append( "kV"     , new wxStringClientData(   "3" ) );
      ChoUnits::Append( "V"      , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "mV"     , new wxStringClientData(  "-3" ) );
      ChoUnits::Append( "uV"     , new wxStringClientData(  "-6" ) );
      ChoUnits::Append( "nV"     , new wxStringClientData(  "-9" ) );
      ChoUnits::Append( "pV"     , new wxStringClientData( "-12" ) );
      ChoUnits::Append( "fV"     , new wxStringClientData( "-15" ) );
      m_osDefUnits = "mV";
      break;

    case eUNITS_CURR :  // Current
      ChoUnits::Clear( );
      ChoUnits::Append( "kA"     , new wxStringClientData(   "3" ) );
      ChoUnits::Append( "A"      , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "mA"     , new wxStringClientData(  "-3" ) );
      ChoUnits::Append( "uA"     , new wxStringClientData(  "-6" ) );
      ChoUnits::Append( "nA"     , new wxStringClientData(  "-9" ) );
      ChoUnits::Append( "pA"     , new wxStringClientData( "-12" ) );
      m_osDefUnits = "mA";
      break;

    case eUNITS_TIME :  // Time
      ChoUnits::Clear( );
      ChoUnits::Append( "Sec"    , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "mSec"   , new wxStringClientData(  "-3" ) );
      ChoUnits::Append( "uSec"   , new wxStringClientData(  "-6" ) );
      ChoUnits::Append( "nSec"   , new wxStringClientData(  "-9" ) );
      ChoUnits::Append( "pSec"   , new wxStringClientData( "-12" ) );
      m_osDefUnits = "mSec";
      break;

    case eUNITS_FREQ :  // Frequency
      ChoUnits::Clear( );
      ChoUnits::Append( "THz"    , new wxStringClientData(  "12" ) );
      ChoUnits::Append( "GHz"    , new wxStringClientData(   "9" ) );
      ChoUnits::Append( "MHz"    , new wxStringClientData(   "6" ) );
      ChoUnits::Append( "kHz"    , new wxStringClientData(   "3" ) );
      ChoUnits::Append( "Hz"     , new wxStringClientData(   "0" ) );
      m_osDefUnits = "kHz";
      break;

    case eUNITS_CHRG :  // Charge
      ChoUnits::Clear( );
      ChoUnits::Append( "C"      , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "mC"     , new wxStringClientData(  "-3" ) );
      ChoUnits::Append( "uC"     , new wxStringClientData(  "-6" ) );
      ChoUnits::Append( "nC"     , new wxStringClientData(  "-9" ) );
      ChoUnits::Append( "pC"     , new wxStringClientData( "-12" ) );
      ChoUnits::Append( "fC"     , new wxStringClientData( "-15" ) );
      m_osDefUnits = "uC";
      break;

    case eUNITS_PHAD :  // Phase / angle
    case eUNITS_PHAR :
      ChoUnits::Clear( );
      ChoUnits::Append( "Degree" , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "Radian" , new wxStringClientData(   "0" ) );
      m_osDefUnits = "Degree";
      break;

    case eUNITS_TMPC :  // Temperature
    case eUNITS_TMPF :
      ChoUnits::Clear( );
      ChoUnits::Append( "Deg.C"  , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "Deg.F"  , new wxStringClientData(   "0" ) );
      m_osDefUnits = "Deg.C";
      break;

    case eUNITS_EXP :  // Dimensionless, append an exponent
      ChoUnits::Clear( );
//    ChoUnits::Append( "x 1E21" , new wxStringClientData(  "21" ) );
//    ChoUnits::Append( "x 1E18" , new wxStringClientData(  "18" ) );
      ChoUnits::Append( "x 1E15" , new wxStringClientData(  "15" ) );
      ChoUnits::Append( "x 1E12" , new wxStringClientData(  "12" ) );
      ChoUnits::Append( "x 1E9"  , new wxStringClientData(   "9" ) );
      ChoUnits::Append( "x 1E6"  , new wxStringClientData(   "6" ) );
      ChoUnits::Append( "x 1E3"  , new wxStringClientData(   "3" ) );
      ChoUnits::Append( "x 1"    , new wxStringClientData(   "0" ) );
      ChoUnits::Append( "x 1E-3" , new wxStringClientData(  "-3" ) );
      ChoUnits::Append( "x 1E-6" , new wxStringClientData(  "-6" ) );
      ChoUnits::Append( "x 1E-9" , new wxStringClientData(  "-9" ) );
      ChoUnits::Append( "x 1E-12", new wxStringClientData( "-12" ) );
      ChoUnits::Append( "x 1E-15", new wxStringClientData( "-15" ) );
//    ChoUnits::Append( "x 1E-18", new wxStringClientData( "-18" ) );
//    ChoUnits::Append( "x 1E-21", new wxStringClientData( "-21" ) );
      m_osDefUnits = "x 1";
      break;

    case eUNITS_NONE :  // No units specified
      ChoUnits::Clear( );
      ChoUnits::Append( "Units"  , new wxStringClientData(   "0" ) );
      m_osDefUnits = "Units";
      break;

    default :
      return( false );
  }

  // Set the new units type
  m_eUnitsType = eUType;

  // Select the default units
  i1 = ChoUnits::FindString( m_osDefUnits, true );  // Do case sensitive string search
  if( i1 != wxNOT_FOUND ) ChoUnits::SetSelection( i1 );

  return( true );
}

//**************************************************************************************************
// Set the units to be displayed.
//
// Note : If a units type is specified in the function argument it must correspond with the type
//        currently set in this class instance.
//
// Argument List:
//   rosUnits - The units to be displayed as a string
//
// Return Values:
//   Success - true
//   Failure - false

bool  ChoUnits::bSetUnits( const wxString & rosUnits )
{
  eTypeUnits  eUType;
  int         i1;

  if( ! bIsCreated( ) )              return( false );

  // Are a different units type specified in the function argument?
  if( UnitsBase::bGetUnitsType( rosUnits, &eUType ) )
    if( eUType != eGetUnitsType( ) ) return( false );

  // Check that the specified units are supported
  i1 = ChoUnits::FindString( rosUnits );
  if( i1 == wxNOT_FOUND )            return( false );

  // Set the units displayed in the control
  ChoUnits::SetSelection( i1 );

  return( true );
}

//**************************************************************************************************
// Set the units to be displayed.
//
// Argument List :
//   iExp - The exponent associated with the units to be displayed
//
// Return Values :
//   Success - true
//   Failure - false

bool  ChoUnits::bSetUnits( int iExp )
{
  wxString  os1;
  long      li1;
  uint      ui1;

  if( ! bIsCreated( ) ) return( false );

  for( ui1=0; ui1<ChoUnits::GetCount( ); ui1++ )
  {
    li1 = 0;
    os1 = ( (wxStringClientData *) ChoUnits::GetClientObject( ui1 ) )->GetData( );
    os1.ToLong( &li1 );
    if( (int) li1 == iExp )
    {
      ChoUnits::SetSelection( ui1 );
      return( true );
    }
  }

  return( false );
}

//**************************************************************************************************
// Set the default units.
//
// Argument List :
//   rosUnits - The default units to be used
//
// Return Values :
//   Success - true
//   Failure - false

bool  ChoUnits::bSetDefUnits( const wxString & rosUnits )
{
  // Don't proceed unless the display object has been created
  if( ! bIsCreated( ) )                                 return( false );

  // Attempt to find the specified units
  if( ChoUnits::FindString( rosUnits ) == wxNOT_FOUND ) return( false );

  // Set the default units
  m_osDefUnits = rosUnits;

  return( true );
}

//**************************************************************************************************
// Get the currently selected units.
//
// Return Values :
//   Success - The selected units
//   Failure - An empty string

const wxString & ChoUnits::rosGetUnits( void )
{
  static  wxString  osUnits;

  osUnits.Empty( );

  if( ! bIsCreated( ) ) return( osUnits );

  osUnits = ChoUnits::GetStringSelection( );

  return( osUnits );
}

//**************************************************************************************************
// Get the currently selected units as an exponent.
//
// Eg. if the units are mV return -3 or if the units are MOhm return 6.
//
// Return Values :
//   Success - The units exponent
//   Failure - 0

int  ChoUnits::iGetUnits( void )
{
  wxString  os1;
  long      i1;

  if( ! bIsCreated( ) )                   return( 0 );

  i1 = ChoUnits::GetSelection( );
  if( i1 == wxNOT_FOUND )                 return( 0 );

  os1 = ( (wxStringClientData *) ChoUnits::GetClientObject( i1 ) )->GetData( );
  if( ! CnvtType::bStrToInt( os1, &i1 ) ) return( 0 );

  return( i1 );
}

//**************************************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  ChoUnits::Print( const wxString & rosPrefix )
{
  std::cout << rosPrefix.mb_str( ) << "m_eUnitsType : ";
  switch( m_eUnitsType )
  {
    case eUNITS_CAP  : std::cout << "eUNITS_CAP";  break;
    case eUNITS_IND  : std::cout << "eUNITS_IND";  break;
    case eUNITS_RES  : std::cout << "eUNITS_RES";  break;
    case eUNITS_COND : std::cout << "eUNITS_COND"; break;
    case eUNITS_VOLT : std::cout << "eUNITS_VOLT"; break;
    case eUNITS_CURR : std::cout << "eUNITS_CURR"; break;
    case eUNITS_TIME : std::cout << "eUNITS_TIME"; break;
    case eUNITS_FREQ : std::cout << "eUNITS_FREQ"; break;
    case eUNITS_CHRG : std::cout << "eUNITS_CHRG"; break;
    case eUNITS_PHAD : std::cout << "eUNITS_PHAD"; break;
    case eUNITS_PHAR : std::cout << "eUNITS_PHAR"; break;
    case eUNITS_TMPC : std::cout << "eUNITS_TMPC"; break;
    case eUNITS_TMPF : std::cout << "eUNITS_TMPF"; break;
    case eUNITS_PCT  : std::cout << "eUNITS_PCT";  break;
    case eUNITS_EXP  : std::cout << "eUNITS_EXP";  break;

    case eUNITS_NONE : std::cout << "eUNITS_NONE"; break;

    default          : std::cout << "Invalid";
  }
  std::cout << '\n';

  std::cout << rosPrefix.mb_str( ) << "m_osDefUnits : " << m_osDefUnits.mb_str( ) << '\n';
}

//**************************************************************************************************
