# GNU Enterprise Application Server - Per-Session Cache
#
# Copyright 2004-2009 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: data.py 9953 2009-10-11 18:50:17Z reinhard $

from gnue.common.apps import errors
from gnue.common.datasources import GDataSource, GConditions, GConnections
from gnue.common.utils.uuid import UUID

class StateChangeError (errors.SystemError):
  def __init__ (self, table, row, oldState):
    msg = u_("Changing state from '%(oldstate)s' to 'initialized' not allowed "
             "in table '%(table)s' row %(row)s") \
          % {'table': table, 'row': row, 'oldstate': repr (oldState)}
    errors.SystemError.__init__ (self, msg)

class InvalidCacheError (errors.SystemError):
  def __init__ (self, table, row, state):
    msg = u_("Row '%(row)s' of table '%(table)s' has an invalid state "
             "'%(state)s'") \
          % {'table': table, 'row': row, 'state': state}
    errors.SystemError.__init__ (self, msg)

class OrderBySequenceError (errors.ApplicationError):
  pass

class CircularReferenceError (errors.ApplicationError):
  def __init__ (self):
    msg = u_("Data contains circular references")
    errors.ApplicationError.__init__ (self, msg)

class DuplicateRecordError (errors.SystemError):
  def __init__ (self, table, row):
    msg = u_("Duplicate instance '%(row)s' in class '%(table)s'") \
          % {'table': table, 'row': row}
    errors.SystemError.__init__ (self, msg)


# =============================================================================
# Cache class
# =============================================================================

class _cache:
  """
  This class is acutally not more than a 3-dimensional array with the
  dimensions called "table", "row", and "field". Any combination of these can
  either have a string value, a value of None, or not be available.

  For any data item, the cache remembers the current value as well as the
  original value.

  This class doesn't do database access. It gets the values to store via the
  "write" method.

  This class is only used internally.
  """

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self):

    self.__old   = {}                     # Original data
    self.__new   = {}                     # Changed (dirty) data
    self.__state = {}                     # State of the data

    # Atomic cache dictionaries
    self.__undo     = {}
    self.__backup   = {}

    # Transaction wide cache dictionaries
    self.inserted = {}
    self.deleted  = {}


  # ---------------------------------------------------------------------------
  # Store data in the clean cache
  # ---------------------------------------------------------------------------

  def write_clean (self, table, row, data):
    """
    Write data to the clean cache.

    A write to the clean cache will be dropped silently if the given field is
    already available in a 'clean version'. This provides transaction integrity
    for backends without transaction support. If a clean write succeeds, the
    state of the given record will be set to 'clean'

    @param table: name of the table
    @param row: id of the row
    @param data: (field, value) pairs to write
    """

    # But be aware not to overwrite an already cached clean value. This is
    # necessary to keep transaction integrity for backends without
    # transaction support.
    r = self.__old.setdefault(table, {}).setdefault(row, {})
    for (field, value) in data:
        r.setdefault(field, value)
    self.__state.setdefault (table + u'-' + row, 'clean')


  # ---------------------------------------------------------------------------
  # Store data in the cache
  # ---------------------------------------------------------------------------

  def write_dirty (self, table, row, field, value):
    """
    Write data to the dirty cache. If no atomic operation is started already,
    this function will start a new atomic operation. 

    A write to the dirty cache will change the record-state to 'commitable', if
    no state information was available before or the state was 'clean'.
    
    @param table: name of the table
    @param row: id of the row
    @param field: name of the field to write data for
    @param value: the value of the field
    """

    key = table + u'-' + row

    # If there is no backup of the record right now, we have to create one
    # before applying any changes.
    if not self.__backup.has_key (key):
      rec  = self.__new.get (table) and self.__new [table].get (row)
      self.__backup [key] = (self.__state.get (key), rec and rec.items ())

    self.__new.setdefault (table, {}).setdefault (row, {}) [field] = value

    # After modifying a record update it's state information. If a record is
    # already initialized or it was a clean record, this modification made it
    # 'commitable'.
    state = self.__state.setdefault (key, 'clean')
    if state in ['initialized', 'clean']:
      self.__state [key] = 'commitable'


  # ---------------------------------------------------------------------------
  # Read data from the cache
  # ---------------------------------------------------------------------------

  def read (self, table, row, field, dirty = True):
    """
    Read data from the cache. If a dirty value is requested, this function
    returns the current version of the field, no matter if it's dirty or not.

    @param table: name of the table
    @param row: id of the row
    @param field: name of the field
    @param dirty: if True, return the current version of the field, otherwise
        return only the clean value of the field.

    @raises KeyError: if no value for the given field is available at all

    @return: The current or clean value of the given field
    """

    # Make sure to use the proper dictionary; use the dirty tables only if a
    # dirty value is available, otherwise always use the clean tables.
    # This construction with the exception is much faster than checking first
    # if the key exists in the __new dictionary construct.

    if dirty:
      try:
        return self.__new[table][row][field]
      except KeyError:
        pass
    return self.__old[table][row][field]


  # ---------------------------------------------------------------------------
  # Return whether a certain value is stored in the cache or not
  # ---------------------------------------------------------------------------

  def has (self, table, row, field, dirty = None):
    """
    Check wether a given table/row has a clean or dirty version of a given
    field.

    @param table: the table to be checked
    @param row: id of the row to be checked
    @param field: the name of the field to be checked
    @param dirty: Could be True, False or None with the following implication:
        True : search for a 'dirty' version
        False: search for a 'clean' version
        None : search for a 'clean' or a 'dirty' version

    @return: True if a value for field is available, False otherwise
    """

    try:
      # If we do not search for a given kind (clean/dirty) try both
      if dirty is None:
        try:
          self.__old [table] [row] [field]

        except KeyError:
          self.__new [table] [row] [field]

      elif dirty:
        self.__new [table] [row] [field]

      else:
        self.__old [table] [row] [field]

    except KeyError:
      return False

    else:
      return True


  # ---------------------------------------------------------------------------
  # Update the state information of a given row
  # ---------------------------------------------------------------------------

  def initialized (self, table, row):
    """
    Finish initialization of a given record (table/row). The next call of the
    write () method will change the state to 'commitable'.

    @param table: name of table
    @param row: name of the row

    @raises StateChangeError: if the former state is not 'initializing'.
    """

    key = "%s-%s" % (table, row)

    # Calling initialized () is allowed only if the former state was
    # 'initializing'. Otherwise it must be a bug !
    state = self.__state.get (key)
    if state != 'initializing':
      raise StateChangeError, (table, row, state)

    self.__state [key] = 'initialized'



  # ---------------------------------------------------------------------------
  # Insert a new record
  # ---------------------------------------------------------------------------

  def insertRecord (self, table, row):
    """
    Insert a new record to the cache. This adds the field 'gnue_id' to the
    dirty cache and sets the record-state to 'initializing'. Make sure to call
    'initialized ()' in order to make a record 'commitable' by further writes.

    @param table: name of the table
    @param row: id of the new record

    @raises DuplicateRecordError: If a record (table, row) should be inserted
         for which we have state information alredy!
    """

    key = "%s-%s" % (table, row)

    # There must not be state information for the new record yet!
    if key in self.__state:
      raise DuplicateRecordError, (table, row)

    # A new record always starts with a state of 'initializing', which means 
    # one has to call the initialized () method in order to let further changes
    # bring the record into a 'commitable' state.
    self.__state [key] = 'initializing'

    # Create a backup of (None, None) for the record and initialize state. This
    # also checks the type of table and row arguments.
    self.write_dirty (table, row, u'gnue_id', row)

    # Mark this record as 'inserted' and add it to the undo-sequence
    self.inserted [key] = (table, row)
    self.__undo [key] = 'insert'


  # ---------------------------------------------------------------------------
  # Delete a record from the cache
  # ---------------------------------------------------------------------------

  def deleteRecord (self, table, row):
    """
    Delete a given record from the cache. The state of the given record changes
    to 'deleted' and the record will be added to the dirty tables.

    @param table: name of the table
    @param row: id of the row
    """

    key = "%s-%s" % (table, row)
    if not self.__backup.has_key (key):
      rec  = self.__new.get (table) and self.__new [table].get (row)
      self.__backup [key] = (self.__state.get (key), rec and rec.items ())

    # If the record is added within the same transaction, drop it from the
    # insertion queue, but make sure to keep a note regarding it's state within
    # the atomic operation.
    if self.inserted.has_key (key):
      # If the record has been created within the same atomic operation, it's
      # safe to remove it from the insertion queue. This is because a cancel
      # does not restore that inserted record. If the record has been created
      # by another atomic operation, we have to re-add it to the inserted list
      # on canceling this deletion.
      if not self.__undo.has_key (key):
        self.__undo [key] = 'remove'
          
      del self.inserted [key]

    # The record is not added within the same transaction, so just add it to
    # the deletion queue
    else:
      self.deleted [key] = (table, row)
      self.__undo [key]  = 'delete'

    # Finally remove the current 'gnue_id' and set the record state to
    # 'deleted'. The former will add this record to the dirty tables, if it was
    # clean before.
    self.write_dirty (table, row, u'gnue_id', None)
    self.__state [key] = 'deleted'


  # ---------------------------------------------------------------------------
  # Confirm changes
  # ---------------------------------------------------------------------------

  def confirm (self):
    """
    Close the current atomic operation by confirming all it's changes.
    """

    self.__undo.clear ()
    self.__backup.clear ()


  # ---------------------------------------------------------------------------
  # Cancel atomic operation
  # ---------------------------------------------------------------------------

  def cancel (self):
    """
    Cancel the current atomic operation by revoking all it's changes.
    """

    while self.__backup:
      (key, (state, items)) = self.__backup.popitem ()

      # Restore state information
      if state is not None:
        self.__state [key] = state

      elif self.__state.has_key (key):
        del self.__state [key]

      # Restore current (dirty) record data
      table, row = key.split ('-')
      if items is not None:
        rec = self.__new.setdefault (table, {}).setdefault (row, {})
        # Remove all fields from the row-dictionary and add all previously
        # stored (field, value) tuples
        rec.clear ()
        for (field, value) in items:
          rec [field] = value

      elif items is None:
        # No changed data was available before, so remove it from the cache
        self.__removeFromDict (self.__new, table, row)

      # If we have an entry in the current atomic cache, do further processing
      if self.__undo.has_key (key):
        action = self.__undo.pop (key)

        try:
          # Remove the record from the insertion queue
          if action == 'insert':
            del self.inserted [key]

          # Re-Add the previously deleted record to the insertion queue
          elif action == 'remove':
            self.inserted [key] = (table, row)

          # Remove the record from the deletion queue
          elif action == 'delete':
            del self.deleted [key]

        except KeyError:
          pass
            
    assert not self.__undo, "atomic operation cache *not* empty"
    assert not self.__backup, "atomic record cache *not* empty"


  # ---------------------------------------------------------------------------
  # Get the state of a record
  # ---------------------------------------------------------------------------

  def state (self, table, row):
    """
    Returns the state of the given record. Returns one of the following results:

      - 'initializing': newly created record with initialization not yet
        finished
      - 'initialized': newly created and initialized records with no
        modifications
      - 'inserted': newly created record with modifications
      - 'changed': existing record with modifications
      - 'deleted': deleted record
      - 'clean': record is available and has no modifications

    @return: current state of the given (table, row) tuple in the cache
    """

    key   = "%s-%s" % (table, row)
    state = self.__state.get (key, 'clean')

    if self.inserted.has_key (key):
      return (state, 'inserted') [state == 'commitable']

    elif self.deleted.has_key (key):
      return 'deleted'

    else:
      return (state, 'changed') [state == 'commitable']


  # ---------------------------------------------------------------------------
  # List all tables with dirty records
  # ---------------------------------------------------------------------------

  def dirtyTables (self):
    """
    Returns a dictionary of tables with dirty data (inserted, changed or
    deleted rows), where the key is the table name and the value is a
    dictionary of all dirty rows in the table, where the key is the row id and
    the value is a dictionary of all dirty fields in that row, where the key is
    the field name and the value is the current value of the field. Got it?
    """

    return self.__new


  # ---------------------------------------------------------------------------
  # Clear the whole cache
  # ---------------------------------------------------------------------------

  def clear (self, oldOnly = False):
    """
    Clear cached data, either all data or only the clean portion.

    @param oldOnly: if True, only the clean portion of the cache will be
        cleaned. This will be used to implement dirty reads.
    """

    if oldOnly:
      # on a commit we need to remove all 'commited' stuff from cache, in order
      # to get new data from other transactions in.
      for table in self.__old.keys ():
        for row in self.__old [table].keys ():
          # remove clean or deleted rows
          if self.state (table, row) in ['clean', 'deleted']:
            del self.__old [table][row]
            key = "%s-%s" % (table, row)
            if self.__state.has_key (key):
              del self.__state [key]

        # if a table has no more rows, remove it too
        if not self.__old [table]:
          del self.__old [table]

    else:
      self.__old.clear ()
      self.__new.clear ()
      self.__state.clear ()
      self.inserted.clear ()
      self.deleted.clear ()


  # ---------------------------------------------------------------------------
  # Make the given row in a table to be treated as 'clean'
  # ---------------------------------------------------------------------------

  def makeClean (self, table, row):
    """
    This function makes a row of a table 'clean'. It will be moved from the
    dirty into the clean cache

    @param table: name of the table
    @param row: gnue_id of the row to be moved
    """

    key = "%s-%s" % (table, row)

    if self.__new.has_key (table) and self.__new [table].has_key (row):
      self.__old.setdefault (table, {}) [row] = self.__new [table] [row]

    self.__removeFromDict (self.__new, table, row)

    if self.__state.has_key (key):
      del self.__state [key]

    if key in self.inserted:
      del self.inserted [key]

    elif key in self.deleted:
      del self.deleted [key]


  # ---------------------------------------------------------------------------
  # Remove a row of a table completely from the cache
  # ---------------------------------------------------------------------------

  def remove (self, table, row):
    """
    This function removes the given row of the table completely from the cache,
    no matter wether it's dirty or not.

    @param table: name of the table
    @param row: id of the row to be removed from the cache
    """

    self.__removeFromDict (self.__new, table, row)
    self.__removeFromDict (self.__old, table, row)

    key = "%s-%s" % (table, row)

    if key in self.__state:
      del self.__state [key]

    if key in self.deleted:
      del self.deleted [key]


  # ---------------------------------------------------------------------------
  # Remove a row of a table from a given cache-dictionary
  # ---------------------------------------------------------------------------

  def __removeFromDict (self, dictionary, table, row):
    """
    This function removes a row of a table from the given cache-dictionary. If
    the specified row was the last row of the table cache, the table-dictionary
    will be removed too.

    @param dictionary: cache-dictionary: dict [table][row][field]
    @param table: name of the table to remove a row from
    @param row: id of the row to be removed
    """

    try:
      if row in dictionary [table]:
        del dictionary [table] [row]

      if not dictionary [table]:
        del dictionary [table]

    except KeyError:
      pass


  # ---------------------------------------------------------------------------
  # Get a sequence of fields not yet kept in any cache
  # ---------------------------------------------------------------------------

  def uncachedFields (self, table, row, fields):
    """
    Return a sequence of fields which are not already cached in the clean or
    dirty cache. 

    @param table: name of the table
    @param row: id of the row
    @param fields: sequence of fields to be checked

    @return: subset from fields, which do not have a clean or dirty value
        cached at the moment
    """

    result = []
    append = result.append

    for item in fields:
      if not self.has (table, row, item):
        append (item)

    return result



# =============================================================================
# Helper methods
# =============================================================================

# -----------------------------------------------------------------------------
# Create a datasource object
# -----------------------------------------------------------------------------

def _createDatasource (connections, database, content, order = None):

  # build table list, field list, and join condition
  primarykey = u'gnue_id'
  master = []
  tables = []
  fields = []
  conditions = None
  for (alias, (table, fk_alias, fk_field, fieldlist)) in content.items ():
    if alias:
      if not fk_alias:
        master.append ((None, table + ' ' + alias))
        primarykey = alias + '.' + primarykey
      fields.extend ([alias + '.' + field for field in fieldlist])
    else:
      master.append ((None, table))
      fields.extend (fieldlist)

    if fk_alias:
      table = u"LEFT JOIN %s %s ON %s.gnue_id = %s.%s" % \
          (table, alias, alias, fk_alias, fk_field)
      tables.append ((alias, table))

  # After building up the table sequences we need to make sure they're sorted
  # by their alias, otherwise the left outer joins won't work.
  master.sort ()
  tables.sort ()

  # prepare attributes of the datasource
  attributes = {}
  attributes ['name']      = ''
  attributes ['database']  = database
  attributes ['table']     = "%s %s" % \
      (','.join ([m [1] for m in master]),
       ' '.join ([t [1] for t in tables]))

  attributes ['primarykey'] = primarykey

  # give the backend a hint that it's working for appserver :)
  datacon = connections.getConnection (database)
  datacon.parameters ['appserver'] = True

  if order is not None:
    if order:
      attributes ['order_by'] = order

  # create the datasource
  datasource = GDataSource.DataSourceWrapper (
    connections = connections,
    attributes = attributes,
    fields = fields)

  if conditions:
    datasource.setCondition (conditions)

  return datasource


# -----------------------------------------------------------------------------
# Create a result set with data
# -----------------------------------------------------------------------------

def _createResultSet (connections, database, content, conditions, order):

  datasource = _createDatasource (connections, database, content, order)
  condition  = GConditions.buildCondition (conditions)
  return datasource.createResultSet (condition, True)


# =============================================================================
# Session class
# =============================================================================

class connection:
  """
  This class encapsulates a connection to the database where data is cached on
  connection level. This means that if one query modifies data, another query
  using the same connection reads the new version even if the changes are not
  committed yet.
  """

  # ---------------------------------------------------------------------------
  # Initialize
  # ---------------------------------------------------------------------------

  def __init__ (self, connections, database):
    checktype (connections, GConnections.GConnections)
    checktype (database, str)

    self.__connections = connections
    self.__database    = database
    self.__cache       = _cache ()

    self.__constraints = {}

    self.__uuidType = gConfig ('uuidtype').lower ()

    self.__backend = connections.getConnection (database, True)


  # ---------------------------------------------------------------------------
  # Create a recordset from a query
  # ---------------------------------------------------------------------------

  def query (self, content, conditions, order):
    """
    Executes a query and returns a recordset object.

    @param content: A dictionary of tuples defining the content of the query.
        The format of the dictionary is {alias: (table, fk_alias, fk_field,
        fields)}.
        
        'alias' is a name for the table that is unique within the query (useful
        if two different references point to the same table).  The alias may
        be None if the query references only a single table.

        'table' is the name of the table.
        
        'fk_alias' is the alias of the table containing the foreign key (i.e.
        the table that references this table), and 'fk_field' is the field name
        of the foreign key (i.e. of the referencing field).  The primary key is
        of course always 'gnue_id'. For a single element in the dictionary
        (namely the main table), 'fk_alias' and 'fk_field' both are None.
        Note that an inner join will be executet, that means only results where
        all references can be resolved (and are non-NULL) will be returned.

        Finally, 'fields' is a list of field names to be included in the query.
        All these fields will be fetched from the database and cached, so that
        subsequential access to those fields won't trigger another access to
        the database backend.

        Alias names, table names, and field names all must be Unicode strings.

    @param conditions: The conditions. May be a GCondition tree Field values in
        conditions must be in native Python type; in case of strings they must
        be Unicode.  Field names in conditions must be Unicode.  In case
        aliases are defined in the content paramter, field names in conditions
        must be in the format 'alias.field'.

    @param order: A list of Unicode strings telling the field names to order
        by.  In case aliases are defined in the content parameter, these field
        names must be in the format 'alias.field'.
    """
    checktype (content, dict)
    for (content_key, content_value) in content.items ():
      checktype (content_key, [None, unicode])
      checktype (content_value, tuple)
      (table, fk_alias, fk_field, fields) = content_value
      checktype (table, unicode)
      checktype (fk_alias, [None, unicode])
      checktype (fk_field, [None, unicode])
      checktype (fields, list)
      for fields_element in fields: checktype (fields_element, unicode)

    checktype (order, [None, list])
    if order:
      for order_element in order:
        checktype (order_element, dict)

    return recordset (self.__cache, self.__connections, self.__database,
                      content, GConditions.buildCondition (conditions), order)


  # ---------------------------------------------------------------------------
  # Create a new record
  # ---------------------------------------------------------------------------

  def insertRecord (self, table):
    """
    Inserts a new record. A 'gnue_id' is assigned automatically.

    Table must be a unicode string.
    """

    checktype (table, unicode)

    if self.__uuidType == 'time':
      row = UUID.generate_time_based ()
    else:
      row = UUID.generate_random ()

    self.__cache.insertRecord (table, row)
    r = record (self.__cache, self.__connections, self.__database, table, row)
    return r


  # ---------------------------------------------------------------------------
  # Delete a record
  # ---------------------------------------------------------------------------

  def deleteRecord (self, table, row):
    """
    Deletes the given record (acutally marks it for deletion on commit). All
    data of the record will stay available until commit, but the field
    'gnue_id' will seem to have a value of None.

    Table and row must be unicode strings.
    """

    checktype (table, unicode)
    checktype (row, unicode)

    self.__cache.deleteRecord (table, row)


  # ---------------------------------------------------------------------------
  # Find a record
  # ---------------------------------------------------------------------------

  def findRecord (self, table, row, fields):
    """
    Loads a record from the database.  All fields given in 'fields' are fetched
    from the database and cached, so that subsequential access to those fields
    won't trigger another access to the db backend.

    This method won't query the db backend for data which is already cached.

    Table and row must be unicode strings, fields must be a list of unicode
    strings.
    """
    checktype (table, unicode)
    checktype (row, unicode)
    checktype (fields, list)
    for fields_element in fields:
      checktype (fields_element, unicode)

    # Let's have a look wether we need a requery or not. Newly inserted records
    # cannot have uncached fields since they are new :)
    state = self.__cache.state (table, row)
    if state in ['initializing', 'initialized', 'inserted']:
      uncachedFields = []
    else:
      uncachedFields = self.__cache.uncachedFields (table, row, fields)

    # If there are no uncached fields there is no need to reload from the
    # backend.
    if not uncachedFields:
      r = record (self.__cache, self.__connections, self.__database, table, row)
      self.__cache.write_clean (table, row, [(u'gnue_id', row)])

    # otherwise requery the current record for all uncached fields
    else:
      new = self.__backend.requery (table, {u'gnue_id': row}, uncachedFields)
      r = record (self.__cache, self.__connections, self.__database, table, row)
      r._fill (None, uncachedFields, new)

    return r


  # ---------------------------------------------------------------------------
  # Add constraints for a table
  # ---------------------------------------------------------------------------

  def setConstraints (self, table, constraints):
    """
    Add constraints for a given table. 

    @param table: name of the table to add constraints for
    @param constraints: dictionary of master-tables for the given table
    """

    if constraints:
      cdict = self.__constraints.setdefault (table.lower (), {})
      for (reftable, refitems) in constraints.items ():
        elements = cdict.setdefault (reftable.lower (), {})
        for item in refitems:
          elements [item] = True


  # ---------------------------------------------------------------------------
  # Write all changes back to the database
  # ---------------------------------------------------------------------------

  def commit (self):
    """
    Write all dirty data to the database backend by a single transaction that
    is committed immediately. This operation invalidates the cache.
    """

    tables = self.__cache.dirtyTables ()

    # We distinguish 2 cases here:
    # A: Backend knows transactions and needs a rollback when an exception has
    # happened: we send all changes to the backend and leave the cache
    # unchanged until the commit has succeeded, so we can restore to the state
    # before in case of an exception. With this case, we will never have any
    # uncommitted changes hanging around in the backend.
    # B: Backend does not know transactions or does not need a rollback when an
    # exception has happened: we clean up each single change from the cache as
    # soon as we have sent it to the backend, so in case of an exception we can
    # later continue where we've stopped. With this case, some changes at the
    # backend might remain uncommitted until the commit is tried again without
    # exception.

    try:
      # first perform all inserts
      if self.__cache.inserted:
        for (table, row) in self.__orderInserts ():
          fields = tables [table] [row]
          self.__backend.insert (table, {'gnue_id': row}, fields)
          if not self.__backend._need_rollback_after_exception_:
            self.__cache.makeClean (table, row)

      # second perform all updates
      for (table, rows) in tables.items ():
        for (row, fields) in rows.items ():
          if self.__cache.state (table, row) == 'changed':
            self.__backend.update (table, {'gnue_id': row}, fields)
            if not self.__backend._need_rollback_after_exception_:
              self.__cache.makeClean (table, row)

      # perform all deletes
      if len (self.__cache.deleted):
        for (table, row) in self.__orderDeletes ():
          self.__backend.delete (table, {'gnue_id': row})
          if not self.__backend._need_rollback_after_exception_:
            self.__cache.remove (table, row)

      # Commit the whole transaction
      self.__backend.commit ()

      if self.__backend._need_rollback_after_exception_:
        for (table, rows) in tables.items ():
          for row in rows.keys ():
            state = self.__cache.state (table, row)
            if state in ['inserted', 'changed']:
              self.__cache.makeClean (table, row)
            elif state == 'deleted':
              self.__cache.remove (table, row)

    except:
      if self.__backend._need_rollback_after_exception_:
        self.__backend.rollback ()
      raise

    # Clear any old stuff from cache, but keep initialized (and not
    # commitbable) records in there.
    self.__cache.clear (True)


  # ---------------------------------------------------------------------------
  # Create an ordered sequence of new records
  # ---------------------------------------------------------------------------

  def __orderInserts (self):
    """
    Order all records scheduled for insertion, so constraint violations are
    avoided.

    @return: sequence of (table, row) tuples in a sane order for insertion
    """

    records = [(table, row) for (table, row) in self.__cache.inserted.values ()\
                             if self.__cache.state (table, row) == 'inserted']
    result = self.__orderByDependency (records)
    assert gDebug (1, "Ordered inserts: %s" % result)

    return result


  # ---------------------------------------------------------------------------
  # Order all records scheduled for deletion
  # ---------------------------------------------------------------------------

  def __orderDeletes (self):
    """
    Order all records scheduled for deletion, so constraint violations are
    avoided.

    @return: sequence of (table, row) tuples in a sane order for deletion
    """

    order = self.__orderByDependency (self.__cache.deleted.values ())
    # since we do deletes we need a reversed order
    order.reverse ()

    return order


  # ---------------------------------------------------------------------------
  # Order a sequence of (table, row) tuples by their dependencies
  # ---------------------------------------------------------------------------

  def __orderByDependency (self, records):
    """
    Order a sequence of records (table, row) so their dependencies given by
    setConstraints () are fullfilled. The result starts with elements having no
    dependency at all.

    @param records: sequence with (table, row) tuples
    @return: ordered sequence with (table, row) tuples
    """

    tables = {}
    data   = {}
    fishes = {}

    assert gDebug (1, "Constraints: %s" % self.__constraints)
    assert gDebug (1, "Unordered  : %s" % records)

    # First create a dictionary with all tables scheduled for processing, and
    # another dictionary of all records grouped by their tablename
    for (table, row) in records:
      tablename = table.lower ()
      tables [tablename] = []
      
      # Note: We use the lowered tablename as key, but the original tablename
      # in the tuple, so later processing of won't get disturbed
      rows = data.setdefault (tablename, [])
      rows.append ((table, row))

    # Now, add all those constraints which are also scheduled for processing.
    # This eliminates constraints which are currently NULL.
    for (table, deps) in tables.items ():
      if self.__constraints.has_key (table):
        for constraint in self.__constraints [table]:
          if tables.has_key (constraint):
            if constraint != table:
              deps.append (constraint)
            else:
              fishes [table] = self.__constraints [table] [table]

    # Now create an ordered sequence taking care of dependencies
    order = []

    while tables:
      addition = []

      for (table, deps) in tables.items ():
        # If a table has no dependencies, add it to the result
        if not len (deps):
          addition.append (table)

          # and remove that table from all other tables dependency sequence
          for ref in tables.values ():
            if table in ref:
              ref.remove (table)

          # finally remove it from the dictionary
          del tables [table]

      # If no tables without a dependency was found, but there are still
      # entries in the tables dictionary, they *must* have circular references
      if not len (addition) and len (tables):
        raise CircularReferenceError

      order.extend (addition)

    # And finally flatten everything to a sequence of tuples
    result = []
    extend = result.extend

    for table in order:
      if table in fishes:
        extend (self.__fishSort (table, data [table], fishes [table]))
      else:
        extend (data [table])

    return result


  # ---------------------------------------------------------------------------
  # Create an order within a table so it does not violate fish-hook constraints
  # ---------------------------------------------------------------------------

  def __fishSort (self, table, rows, constraints):
    """
    Sort all given rows of a table to not violate the given constraints.

    @param table: name of the table
    @param rows: sequence of tuples (table, rowid)
    @param constraints: dictionary with all constraints per row

    @return: ordered sequence of tuples (table, rowid)

    @raises CircularReferenceError: if the given data contains reference cycles
    """

    # First create a dependency tree for all rows, holding a sequence of those
    # rowids which must be processed before the current row
    dTree = {}
    for (t, row) in rows:
      deps = dTree.setdefault (row, [])

      r = record (self.__cache, self.__connections, self.__database, table, row)
      for depField in constraints:
        value = r.getField (depField)
        if value and value not in deps:
          deps.append (value)

    # After we have created a complete dependency tree, we need to remove all
    # those rows from the master-sequences, which do not have a key in the tree
    # itself. Usually these are rows not contained in the current operation
    # (insertion/deletion) so we can assume that they do exist. Otherwise it
    # would not be possible to set a reference in a newly inserted record to an
    # existing record.
    for deps in dTree.values ():
      for row in deps [:]:
        if not row in dTree:
          deps.remove (row)

    # Now create an ordered sequence taking care of dependencies
    order = []

    while dTree:
      addition = []

      for (detail, masters) in dTree.items ():
        # If a row has no masters, add it to the result
        if not len (masters):
          addition.append (detail)

          # and remove that row from all dependency sequence it occurs
          for ref in dTree.values ():
            if detail in ref:
              ref.remove (detail)

          # finally remove it from the dictionary
          del dTree [detail]

      # If no row without a dependency was found, but there are still
      # entries in the dependency tree, they *must* have circular references
      if not len (addition) and len (dTree):
        raise CircularReferenceError

      order.extend (addition)

    return zip ([table] * len (order), order)



  # ---------------------------------------------------------------------------
  # Undo all changes
  # ---------------------------------------------------------------------------

  def rollback (self):
    """
    Undo all uncommitted changes.
    """

    # Send the rollback to the database. Although we have (most probably) not
    # written anything yet, we have to tell the database that a new transaction
    # starts now, so that commits from other sessions become valid now for us
    # too.
    self.__backend.rollback ()

    # The transaction has ended. Changes from other transactions could become
    # valid in this moment, so we have to clear the whole cache.
    self.__cache.clear ()


  # ---------------------------------------------------------------------------
  # Close the connection
  # ---------------------------------------------------------------------------

  def close (self):
    """
    Close the connection to the database backend.
    """

    self.__backend.close ()


  # ---------------------------------------------------------------------------
  # confirm all recent changes
  # ---------------------------------------------------------------------------

  def confirmChanges (self):
    """
    This function confirms all changes to be safe, so a subsequent call of the
    cancelChanges () function restores to this state.
    """

    self.__cache.confirm ()



  # ---------------------------------------------------------------------------
  # revoke all changes up the the last confirm
  # ---------------------------------------------------------------------------

  def cancelChanges (self):
    """
    This function revokes all changes up to the last call of the
    confirmChanges () function.
    """
    self.__cache.cancel ()


# =============================================================================
# Recordset class
# =============================================================================

class recordset:
  """
  This class manages the result of a query. An instance of this class can be
  created via the connection.query() method.
  """

  # ---------------------------------------------------------------------------
  # Initialize
  # ---------------------------------------------------------------------------

  def __init__ (self, cache, connections, database, content, conditions,
                order):
    self.__cache       = cache
    self.__connections = connections
    self.__database    = database
    self.__content     = content
    self.__order       = order

    # make sure gnue_id is selected for all tables
    for (alias, (table, fk_alias, fk_field, fields)) in self.__content.items():
      fields.append (u'gnue_id')

    self.__add     = []
    self.__added   = 0
    self.__remove  = {}
    self.__mergeWithCache (conditions)

    self.__resultSet = _createResultSet (self.__connections, self.__database,
                                         self.__content, conditions, order)
    self.__rawdata = self.__resultSet.raw()


  # ---------------------------------------------------------------------------
  # Return the number of records
  # ---------------------------------------------------------------------------

  def __len__ (self):
    """
    This function returns the number of records in this result set.
    """

    return len (self.__resultSet) + len (self.__add) + self.__added \
            - len (self.__remove)


  # ---------------------------------------------------------------------------
  # Iterator
  # ---------------------------------------------------------------------------

  def __iter__ (self):

      # next uncommmitted new record to be inserted between the records read
      # from the backend
      if self.__add:
          next_add = SortRecord(self.__add[0],
                  self.__getSortSequence(self.__add[0]))
      else:
          next_add = None

      for backend_record in [self.__buildRecords(r) for r in self.__rawdata]:
          # if there are any uncommitted new records, insert them at the
          # correct position
          if next_add is not None:
              gnue_id = backend_record._row
              while next_add and next_add < SortRecord(gnue_id,
                      self.__getSortSequence(gnue_id)):
                  self.__add.pop(0)
                  self.__added += 1
                  yield record(self.__cache, self.__connections,
                          self.__database, self.__getMasterTable(),
                          next_add.row)
                  if self.__add:
                      next_add = SortRecord(self.__add[0],
                              self.__getSortSequence(self.__add[0]))
                  else:
                      next_add = None

          # don't yield records that have been deleted in uncommitted deletes
          if self.__remove.has_key(backend_record._row):
              continue

          yield backend_record

      # If there new records left yield them as well
      while self.__add:
        next = self.__add[0]
        self.__add.pop(0)
        self.__added += 1
        yield record(self.__cache, self.__connections,
                     self.__database, self.__getMasterTable(), next)




  # ---------------------------------------------------------------------------
  # Close the record set
  # ---------------------------------------------------------------------------

  def close (self):
    """
    This function closes a record set which is no longer needed. It closes the
    underlying result set.
    """

    if self.__resultSet is not None:
      self.__resultSet.close ()

    self.__resultSet = None


  # ---------------------------------------------------------------------------
  # Build record objects from current recordSet
  # ---------------------------------------------------------------------------

  def __buildRecords (self, data):

    result = None
    records = {}                        # the record for each alias

    # iterate through all tables involved in the query
    for (alias, (table, fk_alias, fk_field, fields)) in self.__content.items():

      # find id of the record
      if alias:
        id = data [alias + u'.gnue_id']
      else:
        id = data [u'gnue_id']

      if id:
        # build the record object
        r = record (self.__cache, self.__connections, self.__database,
                    table, id)
        r._fill (alias, fields, data)
        records [alias] = r

        # the table having no join condition is the "main" table
        if not fk_alias:
          result = r

    # iterate again to put the foreign keys in the cache
    for (alias, (table, fk_alias, fk_field, fields)) in self.__content.items():
      if fk_alias:
        if records.has_key (alias):
          value = records [alias].getField (u'gnue_id')
        else:
          value = None

        if records.has_key (fk_alias):
          records [fk_alias]._cache (fk_field, value)

    return result


  # ---------------------------------------------------------------------------
  # Get the name of the master table of this recordset
  # ---------------------------------------------------------------------------

  def __getMasterTable (self):
    """
    This function returns the name of the master table of the recordset
    @return: name of the master table (=table without fk_alias)
    """
    result = None

    for (alias, (table, fk_alias, fk_field, fields)) in self.__content.items():
      if not fk_alias:
        result = table
        break

    return result



  # ---------------------------------------------------------------------------
  # Merge cache 
  # ---------------------------------------------------------------------------

  def __mergeWithCache (self, conditions):
    """
    """

    # If the master table is not listed in the dirty cache, we have nothing to
    # do, since everything which is in clean cache will not colide with the
    # backend's result
    tables = self.__cache.dirtyTables ()
    master = self.__getMasterTable ()

    if not tables.has_key (master):
      return

    # Iterate over all dirty rows of the master table and stick them into the
    # apropriate filter
    for (row, fields) in tables [master].items ():
      state = self.__cache.state (master, row)

      if state == 'inserted':
        # an inserted (new) row must match the current condition
        self.__addFilter (row, conditions)

      elif state == 'changed':
        # if a changed row has a modified condition we stick it both into the
        # 'append' and the 'remove' filter
        if self.__conditionChanged (row, conditions):
          self.__addFilter (row, conditions)
          self.__removeFilter (row, conditions)
        else:
          # otherwise we have to analyze the sort order
          if self.__sortOrderChanged (row):
            # if the sortorder has changed we remove it from the old position
            # and insert it at the new one
            self.__remove [row] = True
            self.__add.append (row)

      elif state == 'deleted':
        # a deleted row must match the current condition with it's *original*
        # values
        self.__removeFilter (row, conditions)

    # If we have a sort order defined, we need to sort all records listed for
    # addition
    if self.__order and len (self.__add):
      self.__sortAddition ()


  # ---------------------------------------------------------------------------
  # Filter with destination 'add'
  # ---------------------------------------------------------------------------

  def __addFilter (self, row, condition):
    """
    This function implements a filter for rows heading for the 'add'
    destination. Every row must meet the conditions specified in order to get
    in.
    """

    if condition is not None:
      lookup = self.__getLookupDictionary (condition, row)
      if not condition.evaluate (lookup):
        return

    self.__add.append (row)


  # ---------------------------------------------------------------------------
  # Filter with destination 'remove'
  # ---------------------------------------------------------------------------

  def __removeFilter (self, row, condition):
    """
    This function implements a filter for rows heading for the 'remove'
    destination. Every row must match the condition with it's *original* fields
    in order to get in.
    """

    if condition is not None:
      lookup = self.__getLookupDictionary (condition, row, True)
      if not condition.evaluate (lookup):
        return

    self.__remove [row] = True


  # ---------------------------------------------------------------------------
  # Create a lookup dictionary for condition-evaluation
  # ---------------------------------------------------------------------------

  def __getLookupDictionary (self, condition, row, original = False):
    """
    This function creates a dictionary with all fields listed in a given
    condition as keys and their values based on the given row.

    @param condition: GCondition tree with the conditions
    @param row: gnue_id of the master record to fetch values from
    @param original: if True, the original (clean) values will be used,
        otherwise the current values will be used.
    @return: dictionary with fields and their values
    """
    
    result = {}

    for condfield in condition.findChildrenOfType ('GCCField', True, True):
      path  = self.__getPropertyPath (condfield.name)
      result [condfield.name] = self.__getPathValue (row, path, original)

    return result
      

  # ---------------------------------------------------------------------------
  # Get the value for a property specified in a path
  # ---------------------------------------------------------------------------

  def __getPathValue (self, row, path, original = False):
    """
    This function returns the value of the property defined by the given path
    sequence, starting with the first element using a gnue_id of 'row'.
    @param path: sequence of tuples (alias, table, field) describing the
        property.
    @param row: gnue_id of the record to be used for the first path element
    @param original: if True the original (clean) values are returned,
        otherwise the current values will be returned.
    @return: value of the property
    """

    value = None
    for (alias, table, field) in path:
      r = record (self.__cache, self.__connections, self.__database, table, row)

      value = r.getField (field, original)
      row   = value

      if value is None:
        break

    return value


  # ---------------------------------------------------------------------------
  # Check if a field in a condition has changed
  # ---------------------------------------------------------------------------

  def __conditionChanged (self, row, condition):
    """
    This function iterates over all fields of a condition and returns True if a
    field has been changed or False otherwise.

    @param row: gnue_id of the record to check the condition for
    @param condition: GCondition tree with the condition or None
    @return: True if a condition field has been changed, False otherwise
    """

    if condition is not None:
      for condfield in condition.findChildrenOfType ('GCCField', True, True):
        path    = self.__getPropertyPath (condfield.name)
        if self.__fieldIsChanged (row, path):
          return True

    return False


  # ---------------------------------------------------------------------------
  # Check wether a field in the sort order has changed or not
  # ---------------------------------------------------------------------------

  def __sortOrderChanged (self, row):
    """
    This function checks if a field in the sort order has been changed.

    @param row: gnue_id of the record to check the sort order for
    @return: True if a field has been changed, False otherwise
    """
    
    for field in [f ['name'] for f in self.__order]:
      path = self.__getPropertyPath (field)
      if self.__fieldIsChanged (row, path):
        return True

    return False


  # ---------------------------------------------------------------------------
  # Check if a field has been changed
  # ---------------------------------------------------------------------------

  def __fieldIsChanged (self, row, path):
    """
    This function checks wether a field (described by a path) has been changed
    or not.

    @param row: gnue_id of the record to check the sort order for
    @param path: sequence of tuples (alias, table, field) describing the
        property.
    @return: True if the field has been changed, False otherwise
    """

    # A path of length one is a normal property without any references.
    # In this case if the property hasn't been changed, everything's fine.
    if len (path) == 1:
      (alias, table, field) = path [0]
      if not self.__cache.has (table, row, field, True):
        return False

    # Otherwise we need to compare the original and the current value
    oldvalue = self.__getPathValue (row, path, original = True)
    current  = self.__getPathValue (row, path)

    return cmp (oldvalue, current) != 0


  # ---------------------------------------------------------------------------
  # Create a path to access a given property
  # ---------------------------------------------------------------------------

  def __getPropertyPath (self, name):
    """
    This function creates a path to access a given property based on the
    content-dictionary. 
    @param name: property name including alias and fieldname, separated by a dot
    @return: sequence of tuples (alias, tablename, fieldname)
    """

    result = []
    if '.' in name:
      (alias, field) = name.split ('.', 1)
    else:
      (alias, field) = (None, name)

    if self.__content.has_key (alias):
      (table, fk_alias, fk_field, fields) = self.__content [alias]
      # add a tuple to the result
      result.append ((alias, table, field))

      if fk_alias is not None:
        add = self.__getPropertyPath ("%s.%s" % (fk_alias, fk_field))
        # after finishing recursion add all inner tuples to the result,
        # maintaining sort order
        for ix in xrange (len (add)):
          result.insert (ix, add [ix])

    return result


  # ---------------------------------------------------------------------------
  # Sort all records which we have to merge in
  # ---------------------------------------------------------------------------

  def __sortAddition (self):
    """
    This function sorts all records from the cache to fit the current sort
    order.
    """

    if len (self.__add) > 1:
      seq = []
      for row in self.__add:
        seq.append (SortRecord (row, self.__getSortSequence (row)))

      seq.sort ()
      self.__add = [item.row for item in seq]


  # ---------------------------------------------------------------------------
  # Create a sort sequence for a sort order and a given record
  # ---------------------------------------------------------------------------

  def __getSortSequence (self, row):
    """
    This function creates a sequence of tuples (fieldvalue, direction) for the
    record specified by 'row' using the given sortorder.

    @param row: gnue_id of the record to fetch fieldvalues from
    @return: sort sequence of tuples (value, direction)
    """

    result = []
    append = result.append

    if self.__order:
      for element in self.__order:
        field      = element ['name']
        direction  = element.get ('descending') or False
        ignorecase = element.get ('ignorecase') or False
        value = self.__getPathValue (row, self.__getPropertyPath (field))

        if ignorecase and hasattr (value, 'lower'):
          value = value.lower ()

        append ((value, direction))

    return result


# =============================================================================
# Record class
# =============================================================================

class record:
  """
  This class stands for a record in a database table. An instance of this class
  can be created via the recordset iterator, or by the connection.findRecord()
  method.
  """

  # ---------------------------------------------------------------------------
  # Initialize
  # ---------------------------------------------------------------------------

  def __init__ (self, cache, connections, database, table, row):
    """
    Create a new wrapper for a given record in the cache

    @param cache: the underlying cache instance
    @param connections: GConnections instance which can return a handle to the
        current backend connection
    @param database: name of the backend connection to be used
    @param table: name of the table this record instance is a row of
    @param row: id of the row represented by this record instance
    """

    self.__cache       = cache
    self.__connections = connections
    self.__database    = database
    self.__table       = table

    # performance note:
    # we make this directly accessible because that's faster than get_row()
    self._row          = row

    # backend database connection
    self.__backend     = None


  # ---------------------------------------------------------------------------
  # Cache a single value for this record
  # ---------------------------------------------------------------------------

  def _cache (self, field, value):
    """
    Write a field to the clean cache, but do not replace already cached values

    @param field: name of the field
    @param value: value of the field
    """

    self.__cache.write_clean (self.__table, self._row, [(field, value)])


  # ---------------------------------------------------------------------------
  # Fill the cache for this record with data from a (gnue-common) RecordSet
  # ---------------------------------------------------------------------------

  def _fill (self, alias, fields, RecordSet):
    """
    Write the given fields from the given RecordSet object to the clean cache,
    but do not replace already cached values.

    @param alias: alias used for fields in the RecordSet, i.e. 't0', 't1', ...
    @param fields: sequence of fieldnames to cache from the RecordSet
    @param RecordSet: dictionary like object to retrieve the data from
    """

    if alias:
      self.__cache.write_clean(self.__table, self._row,
            [(f, RecordSet[alias + u'.' + f]) for f in fields])
    else:
      self.__cache.write_clean(self.__table, self._row, RecordSet.iteritems())


  # ---------------------------------------------------------------------------
  # Get the value for a field
  # ---------------------------------------------------------------------------

  def getField (self, field, original = False):
    """
    Get the value for a field. If the value isn't cached, a new query to the
    database is issued to get the value.

    The field name must be given as a unicode string. The result will be
    returned as the native Python datatype, in case of a string field it will
    be Unicode.

    @param field: name of the field to get a value for
    @param original: if True return an original value (either from the clean
        cache or freshly fetched from the backend). If False return a dirty or
        clean value (in that order)

    @return: current or original value of the given field
    """

    checktype (field, unicode)

    # First try to return the value from the cache. Just trying it and ignoring
    # a KeyError exception is much faster than first checking if it exists.
    try:
      return self.__cache.read (self.__table, self._row, field, not original)
    except KeyError:
      pass

    # if the record is newly inserted, it cannot have values in the backend
    # until now. So there's no need to fetch anything.
    if self.state () in ['initializing', 'initialized', 'inserted']:
      return None

    # If we have no backend reference until now, create a weak reference to the
    # connection instance and make sure we're logged in.
    if self.__backend is None:
      self.__backend = self.__connections.getConnection (self.__database, True)

    new = self.__backend.requery (self.__table, {u'gnue_id': self._row},
                                     [field])
    if new:
      value = new.get (field)
      self.__cache.write_clean (self.__table, self._row, [(field, value)])
    else:
      value = None

    return value


  # ---------------------------------------------------------------------------
  # Put the value for a field
  # ---------------------------------------------------------------------------

  def putField (self, field, value):
    """
    Put the value for a field.

    The field name must be given as a unicode string, value must be the native
    Python datatype of the field, in case of a string field it must be Unicode.

    @param field: name of the field (as unicode string)
    @param value: value to be set for the given field (native Python datatype)
    """

    checktype (field, unicode)

    self.__cache.write_dirty (self.__table, self._row, field, value)


  # ---------------------------------------------------------------------------
  # Set this record to initialized state
  # ---------------------------------------------------------------------------

  def initialized (self):
    """
    This function marks this record as 'initialized' so following changes will
    make it commitable.

    @raises StateChangeError: if this function is called for a record having
        another state than 'initializing'.
    """

    self.__cache.initialized (self.__table, self._row)


  # ---------------------------------------------------------------------------
  # Get the state of the current record
  # ---------------------------------------------------------------------------

  def state (self):
    """
    Returns the state of the given record. Returns one of the following results:

      - 'initializing': newly created record with initialization not yet
        finished
      - 'initialized': newly created and initialized records with no
        modifications
      - 'inserted': newly created record with modifications
      - 'changed': existing record with modifications
      - 'deleted': deleted record
      - 'clean': record is available and has no modifications

    @return: current state of the record
    """

    return self.__cache.state (self.__table, self._row)


  # ---------------------------------------------------------------------------
  # Give an official string represenation of the record instance
  # ---------------------------------------------------------------------------

  def __repr__ (self):
    """
    This function returns an official string representation of the record
    instance.
    """

    return "<record of table '%s', row '%s'>" % (self.__table, self._row)
      

# =============================================================================
# This class implements a sortable hull for record instances
# =============================================================================

class SortRecord:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, row, sorting = None):

    self.nullFirstAsc = gConfig ('null_first_asc')
    self.nullFirstDsc = gConfig ('null_first_dsc')
    self.row          = row
    self.sorting      = sorting


  # ---------------------------------------------------------------------------
  # Comparison of two instances
  # ---------------------------------------------------------------------------

  def __cmp__ (self, other):
    """
    This function implements the comparison of two instances.
    """

    # If this or the other instance has no order-by rule, just do the
    # default-compare for instances
    if self.sorting is None or other.sorting is None:
      return cmp (self, other)

    # If both instance have an order-by rule, they must match in length
    if len (self.sorting) != len (other.sorting):
      raise OrderBySequenceError, \
          u_("Order-by sequence mismatch: '%(self)s' and '%(other)s'") \
          % {'self': self.sorting, 'other': other.sorting}

    for ix in xrange (len (self.sorting)):
      (left, descending)  = self.sorting [ix]
      (right, rightOrder) = other.sorting [ix]

      if descending != rightOrder:
        raise OrderBySequenceError, \
            u_("Order-by sequence element has different directions: "
               "'%(self)s', '%(other)s'") \
            % {'self': self.sorting [ix], 'other': other.sorting [ix]}

      noneOpt = self.nullFirstAsc
      if descending:
        (left, right) = (right, left)
        noneOpt = not self.nullFirstDsc

      if None in [left, right]:
        if not noneOpt:
          (left, right) = (right, left)
        result = cmp (left, right)
      else:
        result = cmp (left, right)

      if result != 0:
        return result

    # If no field gave a result, the two instances are treated to be equal
    return 0


  # ---------------------------------------------------------------------------
  # Official string representation
  # ---------------------------------------------------------------------------

  def __repr__ (self):
    """
    This function returns an official string representation of the sort record
    """

    return "<SortRecord for row %s at %s>" % (self.row, hex (id (self)))


# =============================================================================
# Self test code
# =============================================================================

if __name__ == '__main__':

  from gnue.common.apps import GClientApp
  from gnue.appserver import geasConfiguration

  app = GClientApp.GClientApp (application = 'appserver', defaults =
      geasConfiguration.ConfigOptions)

  print 'create connection object ...',
  c = connection (app.connections, 'gnue')
  print 'Ok'

  print 'connection.query for existing records ...'
  content = {u't0': (u'address_person', None, None, [u'address_name']),
             u't1': (u'address_country', u't0', u'address_country',
                     [u'address_name'])}
  rs = c.query (content, None, [{'name': u't0.address_name'}])
  print 'Ok'

  print 'iterate to first record ...',
  for r in rs:
    break
  print 'Ok'

  print 'record.getField with prefetched data ...',
  print repr (r.getField (u'address_name'))

  print 'record.getField with non-prefetched data ...',
  print repr (r.getField (u'address_city'))

  print 'connection.findRecord for joined table ...',
  r = c.findRecord (u'address_country', r.getField (u'address_country'), [])
  print 'Ok'

  print 'record.getField of joined table with prefetched data ...',
  print repr (r.getField (u'address_name'))

  print 'record.getField of joined table with non-prefetched data ...',
  print repr (r.getField (u'address_code'))

  print 'iterate to last record ...',
  for r in rs:
    pass
  print 'Ok'

  print 'record.getField with prefetched data ...',
  print repr (r.getField (u'address_name'))

  print 'connection.insertRecord ...',
  r = c.insertRecord (u'address_person')
  r.initialized ()
  print 'Ok'

  print 'record.getField ...',
  id = r.getField (u'gnue_id')
  print repr (id)

  print 'record.putField ...',
  r.putField (u'address_name', u'New Person')
  print 'Ok'

  print 'record.getField of inserted data ...',
  print repr (r.getField (u'address_name'))

  print 'connection.commit with an inserted record ...',
  c.commit ()
  print 'Ok'

  print 'closing record set ...',
  rs.close ()
  print 'Ok'


  print 'connection.query for previously inserted record ...',
  content = {None: (u'address_person', None, None, [u'address_name'])}
  rs = c.query (content,
                ['eq', ['field', u'address_name'],
                       ['const', u'New Person']], None)
  print 'Ok'

  print 'iterate to first record ...',
  for r in rs:
    break
  print 'Ok'

  print 'record.getField with prefetched data ...',
  print repr (r.getField (u'address_name'))

  print 'record.putField of prefetched data ...',
  r.putField (u'address_name', u'New Name')
  print 'Ok'

  print 'record.putField of non-prefetched data ...',
  r.putField (u'address_city', u'New City')
  print 'Ok'

  print 'record.getField of changed data ...',
  print repr (r.getField (u'address_name'))

  print 'connection.findRecord for previously changed record ...',
  r = c.findRecord (u'address_person', id, [u'address_name'])
  print 'Ok'

  print 'record.getField of changed data, independent query ...',
  print repr (r.getField (u'address_city'))

  print 'connection.commit with a changed record ...',
  c.commit ()
  print 'Ok'

  print 'record.getField of prefetched data ...',
  print repr (r.getField (u'address_name'))

  print 'connection.deleteRecord ...',
  c.deleteRecord (u'address_person', id)
  print 'Ok'

  print 'record.getField of deleted uncommitted record, prefetched ...',
  print repr (r.getField (u'address_name'))

  print 'record.getField of deleted uncommitted record, non-prefetched ...',
  print repr (r.getField (u'address_city'))

  print 'connection.commit with a deleted record ...',
  c.commit ()
  print 'Ok'

  print 'closing result set ...',
  rs.close ()
  print 'Ok'

  print 'check if the record is really gone now ...',
  content = {None: (u'address_person', None, None, [u'address_name'])}
  rs = c.query (content,
                ['eq', ['field', u'address_city'],
                       ['const', u'New City']],
                None)
  for r in rs:
    raise Exception
  print 'Ok'

  print 'closing result set ...',
  rs.close ()
  print 'Ok'

  print 'connection.close ...',
  c.close ()
  print 'Ok'

  print 'Thank you for playing!'
