# GNU Enterprise Application Server - Session Object
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: geasSession.py 9953 2009-10-11 18:50:17Z reinhard $

import copy

from gnue.common.apps import errors, i18n
from gnue.common.datasources import GConditions

from gnue.appserver import data, geasList, geasInstance
from gnue.appserver.geasAuthentication import AuthError
from gnue.appserver import repository


# =============================================================================
# Exceptions
# =============================================================================

class InstanceNotFoundError (errors.SystemError):
  def __init__ (self, classname, objectId):
    msg = u_("Instance '%(objectId)s' of class '%(classname)s' not found") \
          % {'objectId' : objectId,
             'classname': classname }
    errors.SystemError.__init__ (self, msg)

class AccessDeniedError (AuthError):
  def __init__ (self, classname):
    msg = u_("Access to class '%s' denied") % classname
    AuthError.__init__ (self, msg)

class ListNotFoundError (errors.SystemError):
  def __init__ (self, listId):
    msg = u_ ("Cannot find a list with ID '%s'") % listId
    errors.SystemError.__init__ (self, msg)

class NoFilterParamError (errors.UserError):
  def __init__ (self, filterName):
    msg = u_("No parameter for the filter %s specified") % filterName
    errors.UserError.__init__ (self, msg)

class InvalidFilterValueError (errors.AdminError):
  def __init__ (self, filterName, filterValue):
    msg = u_("'%(value)s' is not a valid filter of the type '%(name)s'") \
          % {'value': filterValue,
             'name': filterName}
    errors.AdminError.__init__ (self, msg)

class MultipleFilterValueError (errors.AdminError):
  def __init__ (self, filterName, filterValue):
    msg = u_("Multiple instances of the filter '%(name)s' matches the "
             "value '%(value)s'") \
          % {'name': filterName,
             'value': filterValue}
    errors.AdminError.__init__ (self, msg)

class ValidationCyclesError (errors.ApplicationError):
  def __init__ (self, classlist):
    msg = u_("Maximum validation cycle reached. Classes in current cycle: %s")\
          % ", ".join (classlist)
    errors.ApplicationError.__init__ (self, msg)

# =============================================================================
# Session class
# =============================================================================

class geasSession:

  _MAX_CYCLES = 50      # Number of OnValidate cycles

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self, connections, authAdapter, sm, params):

    self.loggedIn = 0
    self.connections = connections
    try:
      self.database = gConfig ('connection')
    except:
      self.database = "gnue"

    self.__lists          = {}
    self.__authAdapter    = authAdapter
    self.__connection     = None
    self.__dirtyInstances = {}
    self.__operation      = None

    self.sm     = sm                    # The session manager
    self.parameters = params
    self.locale  = params.get ('_language')
    self.user    = None
    self.filters = {}

    self.nullFirstAsc = gConfig ('null_first_asc')
    self.nullFirstDsc = gConfig ('null_first_dsc')


  # ---------------------------------------------------------------------------
  # Get a class definition from a class name and check access
  # ---------------------------------------------------------------------------

  def __getClassdef (self, classname):

    # check if user has access rights for this class
    if not self.__authAdapter.hasAccess (self, self.user, classname):
      raise AccessDeniedError, (classname)

    return self.sm.repository.classes [classname]


  # ---------------------------------------------------------------------------
  # Get a procedure definition from a classdefinition
  # ---------------------------------------------------------------------------

  def __getProcdef (self, classdef, procedurename):
    # add access control to procedures here
    return classdef.procedures [procedurename]


  # ---------------------------------------------------------------------------
  # Get a field name from a property name, resolving references
  # ---------------------------------------------------------------------------

  def __getFieldname (self, classdef, propertyname, contentdict, add,
                      skipCalculated = True):

    elements = propertyname.split ('.')

    a = u't0'                           # start with main table
    c = classdef
    for e in elements [:-1]:
      p = c.properties [e]
      if not p.isReference:
        raise geasInstance.ReferenceError, (propertyname, p.fullName)
      c = p.referencedClass
      # do we already have a link?
      found = False
      for (alias, (table, fk_alias, fk_field, fields)) in contentdict.items ():
        if table == c.table and fk_alias == a and fk_field == p.column:
          found = True
          a = alias
          break
      # have to create new alias (new link)
      if not found:
        new_alias = u't' + str (len (contentdict))
        contentdict [new_alias] = (c.table, a, p.column, [])
        a = new_alias

    # the real (final) propertydef
    p = c.properties [elements [-1]]
    if skipCalculated and p.isCalculated:
      return None

    # add new field to fieldlist
    if add:
      (contentdict [a] [3]).append (p.column)

    # return the column name
    return a + '.' + p.column

  # ---------------------------------------------------------------------------
  # Convert property names in conditions to field names
  # ---------------------------------------------------------------------------

  def __convertCondition (self, classdef, condition, contentdict):

    if condition is not None:
      if isinstance (condition, GConditions.GCField):
        condition.name = self.__getFieldname (classdef, condition.name,
                                            contentdict, False)
      else:
        for child in condition._children:
          self.__convertCondition (classdef, child, contentdict)

  # ---------------------------------------------------------------------------
  # Start a new 'atomic' operation if not already started
  # ---------------------------------------------------------------------------

  def __startOperation (self, operation):
    if self.__operation is None:
      self.__operation = operation
      return True
    else:
      return False



  # ---------------------------------------------------------------------------
  # Cancel all changes of an operation
  # ---------------------------------------------------------------------------

  def __cancelOperation (self):
    try:
      self.__connection.cancelChanges ()

    finally:
      self.__operation = None

  # ---------------------------------------------------------------------------
  # Confirm all changes of an operation
  # ---------------------------------------------------------------------------

  def __confirmOperation (self):
    try:
      self.__connection.confirmChanges ()

    finally:
      self.__operation = None


  # ---------------------------------------------------------------------------
  # Start the session
  # ---------------------------------------------------------------------------

  def _start (self, user, password):

    if self.locale:
      i18n.setcurrentlocale (self.locale)

    # This username/password is for the Application Server, not for the
    # database.
    self.user     = user
    self.loggedIn = self.__authAdapter.authenticate (self, user,
                                                     {'password': password})
    if self.loggedIn:
      self.__connection = data.connection (self.connections, self.database)

    return self.loggedIn


  # ---------------------------------------------------------------------------
  # Close and destroy a session
  # ---------------------------------------------------------------------------

  def _destroy (self):

    assert gEnter (4)

    if self.locale:
      i18n.setcurrentlocale (self.locale)

    for key in self.__lists.keys ():
      self.__lists [key]._destroy ()
      del self.__lists [key]

    self.__dirtyInstances = {}
    self.filters          = {}

    # FIXME: should the authAdapter be contacted?
    self.__connection.close ()

    self.loggedIn = 0

    self.__connection     = None
    self.__operation      = None

    self.sm               = None
    self.parameters       = None

    assert gLeave (4)


  # ---------------------------------------------------------------------------
  # Commit the active transaction
  # ---------------------------------------------------------------------------

  def commit (self):
    """
    This function commit the currently running transaction. But before the
    backend will be requested to do so, all dirty instances are validated.
    """

    assert gEnter (4)

    if self.locale:
      i18n.setcurrentlocale (self.locale)

    level = 0
    todo = self.__dirtyInstances.keys ()

    while todo:
      # If the maximum number of validation cycles has been reached, we'll stop
      # here. Looks like some changes in trigger code is needed.
      if level > self._MAX_CYCLES:
        raise ValidationCyclesError, \
            [i.getClassname () for i in self.__dirtyInstances.values ()]

      # Now validate all instances of the current level, and remove them one by
      # one from the list of dirty instances. This way we do not produce
      # additional cycles if instances are changing themselfs
      for objectId in todo:
        instance = self.__dirtyInstances [objectId]
        instance.validate ()
        del self.__dirtyInstances [objectId]

      # All remaining instances in the sequence are forming the next validation
      # cycle
      todo = self.__dirtyInstances.keys ()
      level += 1

    # If no validation raised an exception and all cycles are passed
    # successfully we can send a commit to the backend now.
    self.__connection.commit ()

    assert gLeave (4)


  # ---------------------------------------------------------------------------
  # Rollback the active transaction
  # ---------------------------------------------------------------------------

  def rollback (self):
    """
    Rollback the active transaction.
    """

    assert gEnter (4)

    if self.locale:
      i18n.setcurrentlocale (self.locale)

    self.__connection.rollback ()
    self.__dirtyInstances = {}

    assert gLeave (4)


  # ---------------------------------------------------------------------------
  # Create a new list of business objects of a given class
  # ---------------------------------------------------------------------------

  def request (self, classname, conditions, sortorder, propertylist):
    """
    Create a new list instance containing all items of a given class matching
    the requested conditions. The elements are sorted according to the given
    sortorder and preloaded with the given properties.

    @param classname: name of the class to fetch items from
    @param conditions: a condition (list, dictionary, GConditionTree, ...)
      acceptable to GCondition.buildCondition to filter items
    @param sortorder: list of sort-instructions to build the resulting list.
      Such a sort-instruction could be a string, a tuple, a list or a
      dictionary. A sort item consists of three parts: propertyname,
      sort-direction and case-sensitiveness. 

      string     : propertyname
      tuple, list: (propertyname, descending, ignorecase)
      dictionary : {'name': ..., 'descending': ..., 'ignorecase': ...}

      A sort item must not be completeley defined. If an element is omitted,
      the following defaults apply: descending = False, ignorecase = False

    @param propertylist: sequence of properties to fetch for each item in the
      resulting list. Such propertynames could also be reference properties
      like 'x_foo.y_bar'. In this case the needed join would be added
      automatically.
    @returns: L{geasList.geasList} instance
    """

    assert gEnter (4)

    if self.locale:
      i18n.setcurrentlocale (self.locale)

    classdef = self.__getClassdef (classname)

    (dsCond, asCond) = self.__splitCondition (conditions, classdef)

    # Start to build the content dictionary for self.__connection.query()
    content = {}
    content [u't0'] = (classdef.table, None, None, [u'gnue_id'])

    for p in propertylist:
      self.__getFieldname (classdef, p, content, True)

    if asCond is not None:
      self.__addConditionToContent (classdef, content, asCond)

    if dsCond is not None:
      self.__convertCondition (classdef, dsCond, content)

    sortlist = []     # translated sortlist for datasource
    dsSort   = []     # datasource sortlist
    asSort   = []     # appserver sortlist

    for ix in range (0, len (sortorder)):
      item = self.__getSortElement (sortorder [ix])
      fieldName = self.__getFieldname (classdef, item ['name'], content, False)

      if fieldName is None:
        asSort.extend ([self.__getSortElement (i) for i in sortorder [ix:]])

        for field in [element ['name'] for element in asSort]:
          self.__getFieldname (classdef, field, content, True)

        break

      else:
        dsSort.append (item)

        add = copy.copy (item)
        add ['name'] = fieldName
        sortlist.append (add)

    recordset = self.__connection.query (content, dsCond, sortlist)

    result = geasList.geasList (self, classdef, self.__connection, recordset,
                           [u'gnue_id'] + propertylist, asCond, dsSort, asSort)

    # We use the address of the geasList object in memory as the list id.  This
    # way, every list gets a really unique list id, and this is even thread
    # safe.
    self.__lists [id (result)] = result

    assert gLeave (4, result)

    return result


  # ---------------------------------------------------------------------------
  # Transform a given item to a proper sort-element dictionary
  # ---------------------------------------------------------------------------

  def __getSortElement (self, item):
    """
    This function converts the given sortelement to a properly defined
    sort-item-dictionary.

    @param item: sort-element in any form (string, list, tuple, dict, ...)
    @return: dictionary with three keys: 'name', 'descending', 'ignorecase'
    """

    result = {'name': None, 'descending': False, 'ignorecase': False}

    if isinstance (item, dict):
      result = item

    elif isinstance (item, tuple) or isinstance (item, list):
      result ['name'] = item [0]
      if len (item) > 1: result ['descending'] = item [1]
      if len (item) > 2: result ['ignorecase'] = item [2]

    else:
      result ['name'] = item

    return result


  # ---------------------------------------------------------------------------
  # Create a single geasInstance object from an existing record
  # ---------------------------------------------------------------------------

  def __findInstance (self, classdef, object_id, propertylist):

    table = classdef.table
    # Only query the main table here.  geasInstance.get will do the rest.
    fields = {}
    for p in propertylist:
      pdef = classdef.properties [p.split ('.') [0]]
      if not fields.has_key (pdef.column) and not pdef.isCalculated:
        fields [pdef.column] = True

    record = self.__connection.findRecord (table, object_id, fields.keys ())
    if record is None:
      raise InstanceNotFoundError, (classdef.fullName, object_id)

    return geasInstance.geasInstance (self, self.__connection, record,
                                      classdef)

  # ---------------------------------------------------------------------------
  # Create a single geasInstance object for a new record
  # ---------------------------------------------------------------------------

  def __newInstance (self, classdef):

    table      = classdef.table
    record     = self.__connection.insertRecord (table)
    repository = self.sm.repository

    instance = geasInstance.geasInstance (self, self.__connection, record,
                                          classdef)
    if classdef.gnue_filter:
      fId   = classdef.gnue_filter.gnue_id
      fName = classdef.gnue_filter.fullName
      instance.put ([fName], [self.__filterValue (fId)])

    instance.updateStamp (True)

    # Fire all OnInit triggers of the class definition
    for trigger in classdef.OnInit:
      instance.call (trigger, None)

    # all modifications are 'serious' from now on 
    record.initialized ()

    return instance


  # ---------------------------------------------------------------------------
  # Load data from the database backend
  # ---------------------------------------------------------------------------

  def load (self, classname, obj_id_list, propertylist):
    """
    Load data or datatypes of a given class from the database backend.

    @param classname: name of the class to retrieve data for
    @param obj_id_list: sequence of gnue_id's to fetch data for. If such a
      gnue_id is None the datatypes of the properties will be fetched instead of
      the data values.
    @param propertylist: sequence of propertynames to fetch data or datatypes
      for.

    @returns: sequence of sequences holding the result-values (one per gnue_id)
    """

    assert gEnter (4)

    if self.locale:
      i18n.setcurrentlocale (self.locale)

    classdef = self.__getClassdef (classname)

    result = []
    for object_id in obj_id_list:
      if object_id:
        instance = self.__findInstance (classdef, object_id, propertylist)
        result.append (instance.get (propertylist))

      else:
        typelist = []
        for p in propertylist:
          item = classdef.findItem (p)

          if item is None:
            raise repository.PropertyNotFoundError, (p, classdef.fullName)

          elif isinstance (item, repository.Procedure):
            typelist.append ('procedure')

          else:
            typelist.append (item.fullType)

        result.append (typelist)

    assert gLeave (4, result)

    return result


  # ---------------------------------------------------------------------------
  # Store data in the database backend
  # ---------------------------------------------------------------------------

  def store (self, classname, obj_id_list, propertylist, data):
    """
    Store data for a given class in the backend database. All changes within a
    single call of this function are treated as an atomic operation. The update
    stamps (gnue_create*, gnue_modify*) will be updated automatically if needed.
    
    @param classname: name of the class to store data for
    @param obj_id_list: sequence of gnue_id's representing the objects to store
      data for. A gnue_id of None represents a new object. In this case a new
      gnue_id will be generated.
    @param propertylist: sequence of the properties to store data for
    @param data: sequence of sequences with the actual data values, one
      sequence per object-id. Order of propertylist and data-sequence must
      match.
    """

    assert gEnter (4)

    if self.locale:
      i18n.setcurrentlocale (self.locale)

    opControl = self.__startOperation ('store')

    try:
      classdef = self.__getClassdef (classname)

      result = []
      i = 0

      for object_id in obj_id_list:
        if object_id:
          instance = self.__findInstance (classdef, object_id, [])
          new_object_id = object_id

        else:
          instance = self.__newInstance (classdef)
          new_object_id = instance.get ([u'gnue_id']) [0]

        instance.put (propertylist, data [i])

        i += 1
        result.append (new_object_id)

        if instance.state () in ['changed', 'inserted']:
          instance.updateStamp ()
          self.__dirtyInstances [new_object_id] = instance

    except:
      if opControl:
        self.__cancelOperation ()
      raise

    else:
      if opControl:
        self.__confirmOperation ()

    assert gLeave (4, result)

    return result


  # ---------------------------------------------------------------------------
  # Delete business objects
  # ---------------------------------------------------------------------------

  def delete (self, classname, obj_id_list):
    """
    Delete objects of a given class. A single call of this function is treated
    as an atomic operation. If on delete fails, all previous deletes will be
    revoked.

    @param classname: name of the class to delete objects from
    @param obj_id_list: sequence of gnue_id's representing the objects to
      delete
    """

    assert gEnter (4)

    if self.locale:
      i18n.setcurrentlocale (self.locale)

    opControl = self.__startOperation ('delete')

    try:
      classdef = self.__getClassdef (classname)

      for objectId in obj_id_list:
        instance = self.__findInstance (classdef, objectId, [])

        for trigger in classdef.OnDelete:
          instance.call (trigger, None)

      for object_id in obj_id_list:
        self.__connection.deleteRecord (classdef.table, object_id)
        if self.__dirtyInstances.has_key (object_id):
          del self.__dirtyInstances [object_id]

    except:
      if opControl:
        self.__cancelOperation ()
      raise

    else:
      if opControl:
        self.__confirmOperation ()

    assert gLeave (4)


  # ---------------------------------------------------------------------------
  # Call a procedure of business objects
  # ---------------------------------------------------------------------------

  def call (self, classname, obj_id_list, procedurename, parameters):
    """
    Call a procedure of a given class on a sequence of gnue_id's.

    @param classname: name of the class
    @param obj_id_list: sequence of gnue_id's representing the objects to call
      the procedure for
    @param procedurename: name of the procedure to call
    @param parameters: dictionary of parameters to pass to each call
    @returns: sequence of sequences with the procedure-results one per gnue_id
    """

    assert gEnter (4)

    if self.locale:
      i18n.setcurrentlocale (self.locale)

    opControl = self.__startOperation ('call')

    try:
      classdef = self.__getClassdef (classname)
      procdef  = self.__getProcdef (classdef, procedurename)

      result = []
      for object_id in obj_id_list:
        instance = self.__findInstance (classdef, object_id, [])
        result.append (instance.call (procdef, parameters))

    except:
      if opControl:
        self.__cancelOperation ()
      raise

    else:
      if opControl:
        self.__confirmOperation ()

    assert gLeave (4, result)

    return result


  # ---------------------------------------------------------------------------
  # Split a condition into a datasource- and an appserver-condition
  # ---------------------------------------------------------------------------

  def __splitCondition (self, condition, classdef):
    """
    This function splits up a given condition into a condition which is handled
    by the datasource, and another one which has to be processed by appserver.
    """

    if condition is None:
      return (self.__filterCondition (classdef), None)

    cTree = GConditions.buildCondition (condition)
  
    if classdef.gnue_filter:
      useFilter  = True
      filterName = classdef.gnue_filter.fullName
      cList = cTree.findChildrenOfType ('GCCField', allowAllChildren = True)
      if len (cList):
        for item in cList:
          if filterName in item.name.split ('.'):
            useFilter = False
            break
    else:
      useFilter = False

    dbTrees = []
    asTrees = []
    forest  = []

    if useFilter:
      self.__splitIntoAnd (self.__filterCondition (classdef), forest)

    if len (cTree._children):
      # NOTE: the first element of a condition tree is a GCondition object
      # self.__splitIntoAnd (cTree._children [0], forest)
      self.__splitIntoAnd (cTree, forest)

    for tree in forest:
      if self.__dbHandlesTree (tree, classdef):
        dbTrees.append (tree)
      else:
        asTrees.append (tree)

    # now create the final trees
    dbCond = None
    asCond = None

    if len (dbTrees):
      dbCond = GConditions.GCondition ()
      comb = GConditions.GCand (dbCond)

      for tree in dbTrees:
        comb._children.append (tree)
        tree.setParent (comb)

    if len (asTrees):
      asCond = GConditions.GCondition ()
      comb = GConditions.GCand (asCond)

      for tree in asTrees:
        comb._children.append (tree)
        tree.setParent (comb)

    return (dbCond, asCond)


  # ---------------------------------------------------------------------------
  # Create a condition tree for the current classdef representing it's filter
  # ---------------------------------------------------------------------------

  def __filterCondition (self, classdef):

    if classdef.gnue_filter:
      filterId   = classdef.gnue_filter.gnue_id
      filterName = classdef.gnue_filter.fullName
      filterCond = GConditions.buildConditionFromDict ( \
          {filterName: self.__filterValue (filterId)})
    else:
      filterCond = None

    return filterCond


  # ---------------------------------------------------------------------------
  # Get the value for the given filter
  # ---------------------------------------------------------------------------

  def __filterValue (self, filterId):

    filterName = self.sm.repository.classes [filterId].fullName
    if not self.filters.has_key (filterName):
      filterValue = self.parameters.get (filterName, None)
      if filterValue is None:
        raise NoFilterParamError, (filterName)

      cond   = {'gnue_id': filterValue}
      fList  = self.request (filterName, cond, [], ['gnue_id'])
      idList = fList.fetch (0, 5)

      if not len (idList):
        raise InvalidFilterValueError, (filterName, filterValue)

      elif len (idList) > 1:
        raise MultipleFilterValueError, (filterName, filterValue)

      self.filters [filterName] = idList [0][0]

    return self.filters [filterName]


  # ---------------------------------------------------------------------------
  # Split a condition tree into separate parts
  # ---------------------------------------------------------------------------

  def __splitIntoAnd (self, tree, forest):
    """
    This function splits a condition tree into several independant subtrees
    which can be re-combined via AND.
    """
    if isinstance (tree, GConditions.GConditionElement):
      if isinstance (tree, GConditions.GCand):
        for child in tree._children:
          self.__splitIntoAnd (child, forest)
      else:
        forest.append (tree)
    else:
      for child in tree._children:
        self.__splitIntoAnd (child, forest)


  # ---------------------------------------------------------------------------
  # Determine wether a condition tree can be handled by the datasource or not
  # ---------------------------------------------------------------------------

  def __dbHandlesTree (self, tree, classdef):
    """
    This function traverses the given condition tree and determines if it can
    be handled by the datasource completely.
    """
    if isinstance (tree, GConditions.GCField):
      return self.__usable (tree, classdef)
    # a backend cannot handle an exist-condition (yet)
    elif isinstance (tree, GConditions.GCexist):
      return False
    else:
      for child in tree._children:
        res = self.__dbHandlesTree (child, classdef)
        if not res:
          return res

    return True


  # ---------------------------------------------------------------------------
  # Return wether a GCField must be handled by appserver or datasource
  # ---------------------------------------------------------------------------
  def __usable (self, element, classdef):
    return not self.__getPropertyDef (element.name, classdef).isCalculated


  # ---------------------------------------------------------------------------
  # Get the property definition for a given property
  # ---------------------------------------------------------------------------

  def __getPropertyDef (self, propertyname, classdef):
    """
    This function gets the property definition for a given propertyname,
    resolving references.
    """
    cDef  = classdef
    parts = propertyname.split ('.')
    for prop in parts [:-1]:
      pDef = cDef.properties [prop]
      if not pDef.isReference:
        raise geasInstance.ReferenceError, (propertyname, prop)
      cDef = pDef.referencedClass

    return cDef.properties [parts [-1]]


  # ---------------------------------------------------------------------------
  # Add all GCField elements of a tree to the content dictionary
  # ---------------------------------------------------------------------------

  def __addConditionToContent (self, classdef, content, tree):
    """
    This function adds all GCField instances in a condition tree to the given
    content dictionary.
    """
    if isinstance (tree, GConditions.GCField):
      self.__getFieldname (classdef, tree.name, content, True)

    # Note: we do not add fields from an exist-condition to the current
    # content-dictionary, cause they will be handled by an extra request ()
    elif not isinstance (tree, GConditions.GCexist):
      for child in tree._children:
        self.__addConditionToContent (classdef, content, child)
