# GNU Enterprise Forms - wx 2.6 UI Driver - Button widget
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: button.py 9956 2009-10-11 18:54:57Z reinhard $

"""
Implementation of the <button> tag
"""

import wx

from gnue.forms.input import GFKeyMapper
from gnue.forms.uidrivers.wx26.widgets._base import UIHelper

__all__ = ['UIButton']

# =============================================================================
# Wrap an UI layer around a wxButton widget
# =============================================================================

class UIButton(UIHelper):
    """
    Implements a button object
    """

    # -------------------------------------------------------------------------
    # Create a button widget
    # -------------------------------------------------------------------------

    def _create_widget_(self, event, spacer):
        """
        Creates a new Button widget.
        """

        owner = self.getParent()

        if self.in_grid:
            parent = owner._get_cell(self, spacer)
        else:
            parent = event.container

        if not self.managed:
            csize = (self.get_default_size()[0], self._uiDriver.cellHeight - 2)
        else:
            csize = wx.DefaultSize

        self.widget = wx.Button(parent, -1, self._gfObject.label, size=csize)
        self.widget.Bind(wx.EVT_BUTTON   , self.__on_button)
        self.widget.Bind(wx.EVT_SET_FOCUS, self.__on_set_focus)
        self.widget.Bind(wx.EVT_KEY_DOWN , self.__on_key_down)
        self.widget.Bind(wx.EVT_CHAR     , self.__on_char)

        if self.in_grid:
            self.widget._gnue_label_ = None

        owner.add_widgets(self, spacer)

        return self.widget


    # -------------------------------------------------------------------------

    def __on_char (self, event):

        keycode = event.GetKeyCode()

        if keycode in [wx.WXK_SPACE, wx.WXK_RETURN]:
            self._gfObject._event_fire()

        else:
            # For all other keys ask the keymapper if he could do something
            # usefull
            (command, args) = GFKeyMapper.KeyMapper.getEvent(keycode,
                    event.ShiftDown(), event.CmdDown(), event.AltDown())

            if command:
                self._request(command, triggerName=args)

            else:
                event.Skip()

    # -------------------------------------------------------------------------

    def __on_key_down(self, event):

        # FIXME: Until a button can be flagged as 'Cancel'-Button, which closes
        # a dialog after the user pressed the escape key, we have to 'simulate'
        # that behaviour here.  This event handler can be removed, as soon as
        # such a button is available.  This handler cannot be integrated into
        # EVT_CHAR, since wxMSW does not generate such an event for WXK_ESCAPE
        keycode = event.GetKeyCode()

        if isinstance(self._uiForm.main_window, wx.Dialog) and \
                keycode == wx.WXK_ESCAPE:
            self._uiForm.main_window.Close()
        else:
            event.Skip()


    # -------------------------------------------------------------------------

    def __on_button (self, event):
    
        if 'wxMac' in wx.PlatformInfo:
            widget = self.widgets.index(event.GetEventObject())
            self._gfObject._event_set_focus(widget)
        self._gfObject._event_fire()


    # -------------------------------------------------------------------------

    def __on_set_focus (self, event):

        # This event is blocked while changing a page - the GF moves the focus
        # to where it wants to have it anyway.
        if self._uiForm.block_focus_events:
            return

        lookup = event.GetEventObject()
        self._gfObject._event_set_focus(self.widgets.index(lookup))
        event.Skip()


    # -------------------------------------------------------------------------
    # Enable/disable this button
    # -------------------------------------------------------------------------

    def _ui_enable_(self, index):
        self.widgets[index].Enable(True)

    # -------------------------------------------------------------------------

    def _ui_disable_(self, index):
        self.widgets[index].Enable(False)


# =============================================================================
# Configuration data
# =============================================================================

configuration = {
  'baseClass': UIButton,
  'provides' : 'GFButton',
  'container': 0,
}
