# GNU Enterprise Navigator - wx26 User Interface
#
# Copyright 2001-2009 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 3, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: GFBlock.py 9214 2007-01-05 19:23:02Z reinhard $

"""
User interface driver for wxWidgets 2.6.
"""

import os
import sys
if not hasattr(sys, 'frozen'):
    import wxversion
    wxversion.ensureMinimal('2.6')

from gnue import paths
try:
    from gnue.forms.uidrivers import wx26 as ui
except ImportError:
    # wx26 UIdriver will be renamed to wx in gnue-forms 0.7
    from gnue.forms.uidrivers import wx as ui

from gnue.navigator import VERSION

# Must import wx *after* importing wx26 uidriver from forms.
import wx.html

class Instance:
  def __init__(self, processes):
    self.processes = processes
    self.__current_form = None

    self.titlePage = """
    <html>
      <body>
        <center>
         <B>GNUe Navigator</B>
         <p><img src="%s"></p>
         <p>A part of the <a href="http://www.gnuenteprise.org/">GNU Enterprise Project</a></p>
        </center>
      </body>
    </html>
    """ % os.path.join(paths.data, "share", "gnue", "images", "ship2.png")

  def run(self, instance):
    #
    # Assign the proper login handler based upon the user interface choice
    #
    self.__instance = instance
    self.__instance.keepalive(True)

    app = wx.GetApp() or wx.App()
    self.frame = wx.Frame(None, -1, "GNUe Navigator", size=wx.Size(600,400))
    self.frame.Bind(wx.EVT_CLOSE, self.__on_close)

    frame_sizer = wx.BoxSizer(wx.VERTICAL)
    self.frame.SetSizer(frame_sizer)

    self.menu = MenuBar(self)
    self.frame.SetMenuBar(self.menu)

    wx.EVT_MENU(self.frame, ID_EXIT,  self.__on_exit)
    wx.EVT_MENU(self.frame, ID_ABOUT, self.__on_about)

    self.splitter= wx.SplitterWindow(self.frame,-1)
    frame_sizer.Add(self.splitter, 1, wx.EXPAND)

    self.panel1 = wx.Panel(self.splitter)
    self.panel2 = wx.Panel(self.splitter)

    sizer1 = wx.BoxSizer(wx.VERTICAL)
    sizer2 = wx.BoxSizer(wx.VERTICAL)

    self.panel1.SetSizer(sizer1)
    self.panel2.SetSizer(sizer2)

    # Panel 1 contents
    self.tree = wx.TreeCtrl(self.panel1, -1)
    sizer1.Add(self.tree, 1, wx.EXPAND)
    self.processes.setClientHandlers({'form':self.runForm})

    self.tree.Bind(wx.EVT_TREE_SEL_CHANGED, self.__on_select, self.tree)
    self.tree.Bind(wx.EVT_TREE_ITEM_ACTIVATED, self.__on_activate, self.tree)

    self.processes.walk(self.__buildTreeControl)

    self.tree.Expand(self.processes.__node)

    # Panel 2 contents
    self.helpText = wx.html.HtmlWindow(self.panel2, -1)
    sizer2.Add(self.helpText, 1, wx.EXPAND)
    self.helpText.SetPage(self.titlePage)

    self.frame.Show(True)

    self.splitter.SetMinimumPaneSize(20)
    self.splitter.SplitVertically(self.panel1, self.panel2)
    self.splitter.SetSashPosition(200)

    if gConfigNav('embedForms') == "%s" % True: 
        self.__instance._uiinstance._parentContainer = {
                'frame': self.frame,
                'parent': self.panel2,
                'navigator': self}

    app.MainLoop()


  def __buildTreeControl(self, object):

    if object._type == 'GNProcesses':
      node = self.tree.AddRoot(object.title)
    elif object._type in ('GNStep','GNProcess'):
      node = self.tree.AppendItem(object.getParent().__node, object.title)
    else:
      return

    object.__node = node
    self.tree.SetPyData(node, object)


  def __on_close(self, event):
    if event.CanVeto() and self.__current_form is not None:
        # Switch off keepalive so the application ends now.
        self.__instance.keepalive(False)
        self.__current_form.close()
        # If closing failed (user aborted when asked to save), continue app.
        self.__instance.keepalive(True)
        event.Veto()
    else:
        self.frame.Destroy()

  def __on_exit(self, event):
      self.frame.Close()

  def __on_about(self, event):
    text = _("GNUE Navigator")+"\n"+      \
    _("  Version : ")+"%s\n"+         \
    _("  Driver  : UIwxpython")+"\n"+ \
    _("-= Process Info =-")+"\n"+        \
    _("  Name   : ")+"%s\n"+          \
    _("  Version: ")+"%s\n"+          \
    _("  Author : ")+"%s\n"+          \
    _("  Description:")+"%s\n"
    dlg = wx.MessageDialog(self.frame,
                          text % (VERSION,"","","",""),
                          _("About"), wx.OK | wx.ICON_INFORMATION)
    dlg.ShowModal()

    dlg.Destroy()

  def buildMenu(self, process):
    self.tree.Expand(process.__node)
    return 1


  def __on_select(self, event):
    object = self.tree.GetPyData(event.GetItem())
    self.helpText.SetPage("")
    for item in object._children:
      if item._type == 'GNDescription':
        self.helpText.SetPage(item.getChildrenAsContent())
        break


  def __on_activate(self, event):
    object = self.tree.GetPyData(event.GetItem())

    if object._type != 'GNStep':
      self.buildMenu(object)
    else:
      object.run()


  def runForm(self, step, parameters = {}):

    if os.path.basename(step.location) == step.location:
      try:
        formdir = gConfigNav('FormDir')
      except KeyError:
        formdir = ""
      formfile = os.path.join (formdir, step.location)
    else:
      formfile = step.location

    if gConfigNav('embedForms') == "%s" % True: 
      if self.__current_form is None:
        self.helpText.Hide()
      else:
        self.__current_form.close()
    self.__current_form = self.__instance.run_from_file(formfile,
            step._params)


  # Called when an embedded form is closed.
  def event_form_closed(self):

      self.__current_form = None
      self.frame.SetMenuBar(self.menu)
      self.helpText.Show()


ID_EXIT = wx.NewId()
ID_ABOUT = wx.NewId()

class MenuBar(wx.MenuBar):
  def __init__(self, frame):
    wx.MenuBar.__init__(self)

    self._frame = frame

    self._menu = wx.Menu()
    self._help = wx.Menu()

    self._menu.Append(ID_EXIT, "E&xit", "Exit GNUe Navigator")
    self._help.Append(ID_ABOUT,'&About','About')

    self.Append(self._menu,'&Menu')
    self.Append(self._help,'&Help')
