/*
 * arch/arm/mach-dmw/css/coma.c - the cordless manager
 *
 * The cordless manager is responsible for maintaining the main cfifo between
 * Linux and cordless which is used for loading and unloading cordless and for
 * registration of other services (e.g., debug output, VoIP/RTP, etc.)
 *
 * Copyright (C) 2007 NXP Semiconductors
 * Copyright (C) 2008 - 2011 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/wait.h>
#include <linux/mutex.h>
#include <linux/kthread.h>
#include <linux/completion.h>
#include <linux/err.h>
#include <linux/kfifo.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/fiq.h>
#include <asm/uaccess.h>
#include <mach/platform.h>
#include <mach/hardware.h>


MODULE_AUTHOR("DSP Group Inc.");
MODULE_LICENSE("GPL");

#include "coma.h"
#include "cfifo.h"
#include "cmsg-coma.h"
#include "csignaling.h"


#define COMA_UART_BUF_SIZE  1024

static DECLARE_COMPLETION(reply_compl);
static DEFINE_MUTEX(coma_mutex);

static union cmsg_coma_params last_cmsg_coma_params;

unsigned char timing_stats[66 * 8];

struct service_s {
	int id;
	const char *name;
	int active;
	struct cfifo *l2c;
	struct cfifo *c2l;
	int (*process_message)(struct cmsg *cmsg);
	int (*setup)(void);
	void (*remove)(void);
	unsigned long received_bytes;
	unsigned long sent_bytes;
};

struct service_s services[COMA_NUM_SERVICES];

/*
 * helper functions
 */


static ssize_t show_services(struct device *dev, struct device_attribute *attr, char *buf)
{
	int i;
	ssize_t len;

	len = 0;

	for (i = 1; i < COMA_NUM_SERVICES; i++) {
		if (services[i].id != 0) {
			len += snprintf(buf + len, PAGE_SIZE - len, "%d", services[i].id);
			if (services[i].name != NULL)
				len += snprintf(buf + len, PAGE_SIZE - len, " %s", services[i].name);

			len += snprintf(buf + len, PAGE_SIZE - len, "\n");
		}
	}

	return len;
}

static ssize_t show_cfifo(struct device *dev, struct device_attribute *attr, char *buf)
{
	int i;
	ssize_t len;

	len = 0;

	for (i = 1; i < COMA_NUM_SERVICES; i++) {
		if (services[i].id != 0) {
			len += snprintf(buf + len, PAGE_SIZE - len, "%d %u %u %u %u %u %u\n", services[i].id,
				services[i].l2c->size, services[i].l2c->in, services[i].l2c->out,
				services[i].c2l->size, services[i].c2l->in, services[i].c2l->out);
		}
	}

	return len;
}

static ssize_t show_stats(struct device *dev, struct device_attribute *attr, char *buf)
{
	int i;
	ssize_t len;

	len = 0;

	for (i = 1; i < COMA_NUM_SERVICES; i++) {
		if (services[i].id != 0) {
			len += snprintf(buf + len, PAGE_SIZE - len, "%d %lu %lu\n", services[i].id,
				services[i].received_bytes, services[i].sent_bytes);
		}
	}

	return len;
}

static ssize_t reset_stats(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int id;

	id = (int)simple_strtoul(buf, NULL, 0);

	if (id > 0 && id < COMA_NUM_SERVICES)
	{
		if (services[id].id == id) {
			services[id].received_bytes = 0;
			services[id].sent_bytes = 0;
		}
	}

	return count;
}

static ssize_t show_loglevel(struct device *dev, struct device_attribute *attr, char *buf)
{
	return 0;
}

static ssize_t set_loglevel(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	return count;
}

static ssize_t show_logmodule(struct device *dev, struct device_attribute *attr, char *buf)
{
	return 0;
}

static ssize_t set_logmodule(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	return count;
}


static struct device_attribute coma_attrs[] = {

__ATTR(services, 		S_IRUGO,  		    show_services, 	0),
__ATTR(cfifo, 			S_IRUGO, 		    show_cfifo,    	0),
__ATTR(statistics, 		S_IRUGO | S_IWUSR, 	show_stats, 	reset_stats),
__ATTR(loglevel, 		S_IRUGO | S_IWUSR, 	show_loglevel, 	set_loglevel),
__ATTR(logmodule, 		S_IRUGO | S_IWUSR, 	show_logmodule, set_logmodule)
};

static int coma_sysfs_init(struct platform_device *pdev)
{
	int ret = 0;
	int i = 0;

	for (i = 0; i < ARRAY_SIZE(coma_attrs); i++) {
		ret = device_create_file(&pdev->dev, &coma_attrs[i]);
	}

	if (ret) {
		printk(KERN_ERR "coma: failed creating sysfs file %d\n", ret);
		return ret;
	}

	return 0;
}

static void coma_sysfs_release(struct platform_device *pdev)
{
	int i = 0;

	for (i = 0; i < ARRAY_SIZE(coma_attrs); i++) {
		device_remove_file(&pdev->dev, &coma_attrs[i]);
	}
}


#ifdef CONFIG_CSS_UART

int uart_send_message(int id, unsigned char *data);
int uart_new_message = 0;
wait_queue_head_t uart_wq, uart_send_wq;

#endif /* CONFIG_CSS_UART */

int
coma_create_message(int id, int type, void *params, unsigned int params_size,
                    unsigned char *payload, unsigned int payload_size)
{
	int ret;
	struct cmsg *cmsg;

	ret = cmsg_alloc_from_cfifo(&cmsg, services[id].l2c, params_size,
	                            payload_size);
	if (ret != 0)
		return ret;

	cmsg->type = type;

	if (params && params_size)
		memcpy(cmsg->params, params, params_size);

	if (payload && payload_size)
		memcpy(cmsg->payload, payload, payload_size);

	ret = cmsg_commit_to_cfifo(cmsg, services[id].l2c);

	services[id].sent_bytes += (params_size + payload_size);	

#ifdef CONFIG_CSS_UART
	if (ret == 0) {
		uart_new_message = 1;
		wake_up(&uart_send_wq);
	}
#endif /* CONFIG_CSS_UART */

	return ret;
}
EXPORT_SYMBOL(coma_create_message);

static int
coma_create_coma_message(enum cmsg_coma_types type,
                         union cmsg_coma_params *params,
                         void *payload, unsigned int payload_size)
{
	return coma_create_message(COMA_SERVICE, (int)type,
	                           (void *)params, sizeof(*params),
	                           payload, payload_size);
}

/*
 * coma functions
 */

static void
coma_generic_reply(union cmsg_coma_params *params)
{
	last_cmsg_coma_params = *params;
	complete(&reply_compl);
}

static int
coma_request_timing(int options)
{
	union cmsg_coma_params params;
	int ret;

	params.request_timing.options = options;
	ret = coma_create_coma_message(CMSG_COMA_REQUEST_TIMING, &params,
	                               NULL, 0);
	if (ret == 0)
		coma_signal(COMA_SERVICE);

	return ret;
}

int
coma_timing(int options)
{
	int ret;
	mutex_lock(&coma_mutex);

	ret = coma_request_timing(options);
	if (ret < 0) {
		mutex_unlock(&coma_mutex);
		return -EFAULT;
	}

	wait_for_completion(&reply_compl);

	ret = last_cmsg_coma_params.reply_timing.timestamp;
	mutex_unlock(&coma_mutex);

	return ret;
}
EXPORT_SYMBOL(coma_timing);

/*
 * service (de)registration
 */

static int
coma_request_register(int id, struct cfifo *l2c, struct cfifo *c2l)
{
	union cmsg_coma_params params;
	int ret;

	mutex_lock(&coma_mutex);
	params.request_register.id = id;
	params.request_register.l2c = l2c->self_phys;
	params.request_register.c2l = c2l->self_phys;

	ret = coma_create_coma_message(CMSG_COMA_REQUEST_REGISTER,
	                               &params, NULL, 0);
	if (ret == 0)
		coma_signal(COMA_SERVICE);

	mutex_unlock(&coma_mutex);

	return ret;
}

int
coma_register(int id, const char *name, struct cfifo *l2c, struct cfifo *c2l,
              int (*process_message)(struct cmsg *cmsg),
              int (*setup)(void), void (*remove)(void))
{
	if (id >= COMA_NUM_SERVICES)
		return -EINVAL;

	services[id].id = id;
	services[id].name = name;
	services[id].active = 0;
	services[id].l2c = l2c;
	services[id].c2l = c2l;
	services[id].process_message = process_message;
	services[id].setup = setup;
	services[id].remove = remove;
	services[id].received_bytes = 0;
	services[id].sent_bytes = 0;

	return coma_request_register(id, l2c, c2l);
}
EXPORT_SYMBOL(coma_register);

static int
coma_request_deregister(int id)
{
	union cmsg_coma_params params;
	int ret;

	mutex_lock(&coma_mutex);
	params.request_deregister.id = id;

	ret = coma_create_coma_message(CMSG_COMA_REQUEST_DEREGISTER,
	                               &params, NULL, 0);
	if (ret == 0)
		coma_signal(COMA_SERVICE);

	mutex_unlock(&coma_mutex);

	return ret;
}

int
coma_deregister(int id)
{
	int ret = 0;

	if (id >= COMA_NUM_SERVICES)
		return -EINVAL;

	ret = coma_request_deregister(id);

	services[id].id = 0;
	services[id].active = 0;
	services[id].l2c = NULL;
	services[id].c2l = NULL;
	services[id].process_message = NULL;
	services[id].setup = NULL;
	services[id].remove = NULL;
	services[id].received_bytes = 0;
	services[id].sent_bytes = 0;

	return ret;
}
EXPORT_SYMBOL(coma_deregister);

/*
 * message handling
 */

static int
coma_process_message(struct cmsg *cmsg)
{
	int ret = 0;
	struct reply_register *rr;
	struct reply_deregister *rd;
	union cmsg_coma_params *params =
	                       (union cmsg_coma_params *)cmsg->params;

	switch(cmsg->type) {
	case CMSG_COMA_REPLY_INIT:
	case CMSG_COMA_REPLY_REGISTER:
		rr = (struct reply_register *)params;
		if (rr->result == 0) {
			services[rr->id].active = 1;
			if (services[rr->id].setup)
				services[rr->id].setup();
		}
		break;
	case CMSG_COMA_REPLY_DEREGISTER:
		rd = (struct reply_deregister *)params;
		services[rd->id].active = 0;
		break;
	case CMSG_COMA_REPLY_TIMING:
		memcpy(timing_stats, cmsg->payload, 66 * 8);
		coma_generic_reply(params);
		break;
	case CMSG_COMA_WRITE_CONFIGURATION:
		ret = 0;
		break;
	default:
		ret = -1;
		break;
	}

	return ret;
}

/*
 * interrupt handling
 */

void
coma_receive(int i)
{
	struct cmsg *cmsg;
	int ret;
    int msg_len;

	if (/*!services[i].active ||*/ !services[i].c2l ||
	    !services[i].process_message)
		return;

	while ((msg_len = cfifo_get(services[i].c2l, (void **)&cmsg)) > 0) {
		cmsg->params = cmsg + 1;
		cmsg->payload = (unsigned char *)(cmsg + 1) + cmsg->params_size;

        if (msg_len == sizeof(*cmsg) + cmsg->params_size + cmsg->payload_size) {
            ret = services[i].process_message(cmsg);
            if (ret < 0)
                printk(KERN_ERR "coma: error processing cmsg "
                       "(type = %d, ret = %d, service %i)\n",
                       cmsg->type, ret, i);
        }
        else {
            printk(KERN_ERR "coma_receive: bad msg\n"); 
        }

		cfifo_processed(services[i].c2l);
			
		services[i].received_bytes += cmsg->params_size;
		services[i].received_bytes += cmsg->payload_size;
	}
}

#ifdef CONFIG_CSS_UART

static int uart_fd;
static struct file *uart_file;
static struct task_struct *uart_receiver;
static struct task_struct *uart_sender;
static struct completion uart_receiver_exit;
static struct completion uart_sender_exit;

#ifdef COMA_USE_ACK

int waiting_for_ack = 0;


int uart_send_ack(void)
{
	int ret;
	loff_t pos;
	unsigned char ack = 0x9E;
	pos = uart_file->f_pos;
	ret = vfs_write(uart_file, &ack, 1, &pos);
	uart_file->f_pos = pos;

	return ret;
}

int uart_check_ack(unsigned char ack)
{
	if (waiting_for_ack && (ack == 0x9E)) {
		waiting_for_ack = 0;
		wake_up(&uart_wq);
		return 1;
	}

	return 0;
}


void uart_wait_for_ack(void)
{
	while (1) {
		if (! wait_event_interruptible(uart_wq, (!waiting_for_ack)))
			return;
	}
}

#endif // COMA_USE_ACK

int uart_raw_read(unsigned char *buf, unsigned int len)
{
	unsigned char tmp[20];
	int ret;
	int i;
	loff_t pos;

    printk(KERN_WARNING "uart_raw_read() - enter len %d\n", len);
    printk(KERN_WARNING "\n");
//    COMA_LOG(0, COMA_LOG_WARNING, "uart_raw_read() - enter len %d\n", len);
//    COMA_LOG(0, COMA_LOG_WARNING, "\n");

	i = 0;
	while (i < len) {
		pos = uart_file->f_pos;
		ret = vfs_read(uart_file, tmp, 1, &pos);
		if (ret < 0)
        {
            printk(KERN_ERR "uart_raw_read() - read error\n");
			return -1;
        }
		uart_file->f_pos = pos;

#ifdef COMA_USE_ACK
		if (uart_check_ack(tmp[0]))
			continue;
#endif //COMA_USE_ACK

		buf[i] = tmp[0];
		i++;

        printk("%.2X ", tmp[0]);
//        COMA_LOG_RAW(0, COMA_LOG_DEBUG, "%.2X ", tmp[0]);
	}

    printk("\n");
    printk(KERN_WARNING "uart_raw_read() - exit (read %d bytes)\n", len);

//    COMA_LOG(0, COMA_LOG_WARNING, "\n");
//    COMA_LOG(0, COMA_LOG_WARNING, "uart_raw_read() - exit (read %d bytes)\n", len);

	return len;
}

int uart_raw_write(unsigned char *buf, unsigned int len)
{
	int ret;
	int i;
	loff_t pos;

    printk(KERN_WARNING "uart_raw_write() - enter\n");
	printk(KERN_WARNING "\n");

//    COMA_LOG(0, COMA_LOG_WARNING, "uart_raw_write() - enter\n");
//    COMA_LOG_RAW(0, COMA_LOG_WARNING, "\n");

    for (i = 0; i < len; i++) {
		pos = uart_file->f_pos;
		ret = vfs_write(uart_file, /*uart_user_*/buf + i, 1, &pos);
		if (ret < 0)
        {
            printk(KERN_ERR "uart_raw_write() - write error\n");
			return -1;
        }
		uart_file->f_pos = pos;
        printk("%.2X ", *(buf+i));
//        COMA_LOG_RAW(0, COMA_LOG_DEBUG, "%.2X ", *(buf+i));
	}

    printk("\n");
    printk(KERN_WARNING "uart_raw_write() - exit (wrote %d bytes)\n", len);

//    COMA_LOG_RAW(0, COMA_LOG_DEBUG, "\n");
//    COMA_LOG(0, COMA_LOG_DEBUG, "uart_raw_write() - exit (wrote %d bytes)\n", len);

	return len;
}

int uart_send_message(int id, unsigned char *data)
{
	int ret = 0, len;
	unsigned char tmp[20];
	unsigned char *p;
	int i, to_write;
	struct cmsg *cmsg;

	cmsg = (struct cmsg *)data;

	len = sizeof(*cmsg) + cmsg->params_size + cmsg->payload_size;

#ifdef COMA_USE_ACK
	waiting_for_ack = 1;
#endif // COMA_USE_ACK

	*(int *)tmp = 0xDADADADA;
	ret = uart_raw_write(tmp, 4);
	/* printk(KERN_WARNING "send_message: ret=%d len=%d\n", ret, 4); */
	if (ret < 0)
		return ret;

#ifdef COMA_USE_ACK
	uart_wait_for_ack();
#endif // COMA_USE_ACK

	*(int *)tmp = id;
	ret = uart_raw_write(tmp, 1);
	/* printk(KERN_WARNING "send_message: ret=%d len=%d\n", ret, 1); */
	if (ret < 0)
		return ret;

#ifdef COMA_USE_ACK
	waiting_for_ack = 1;
#endif // COMA_USE_ACK

	*(int *)tmp = len;
	ret = uart_raw_write(tmp, 4);
	/* printk(KERN_WARNING "send_message: ret=%d len=%d\n", ret, 4); */
	if (ret < 0)
		return ret;

#ifdef COMA_USE_ACK
	uart_wait_for_ack();
#endif // COMA_USE_ACK

	i = 0;
	p = data;
	while (i < len) {

#ifdef COMA_USE_ACK
		waiting_for_ack = 1;
#endif // COMA_USE_ACK

		to_write = 20;
		if ((len - i) < to_write)
			to_write = len - i;
		ret = uart_raw_write(p, to_write);
		if (ret < 0)
			return ret;
		/* printk(KERN_WARNING "send_message: ret=%d len=%d\n", ret, to_write); */
		p += to_write;
		i += to_write;

#ifdef COMA_USE_ACK
		uart_wait_for_ack();
#endif // COMA_USE_ACK
	}

	return 0;
}

int uart_get_message(int *id, char *buf, int *len)
{
	int ret, i, to_read;
	unsigned char tmp[20];
	unsigned char *p;

	while (1) {

start:
        for (i = 0; i < 4; i++) {
            ret = uart_raw_read(tmp, 1);
            if (ret < 0)
                return -1;
    
            if (tmp[0] != 0xDA) {
                printk(KERN_ERR "wrong signature byte %d 0x%X %c\n", i + 1, tmp[0], tmp[0]);
                goto start;
            }
        }

#ifdef COMA_USE_ACK
		if (uart_send_ack() < 0)
			return -1;
#endif // COMA_USE_ACK

		ret = uart_raw_read(tmp, 1);
		if (ret < 0) {
            printk(KERN_ERR "uart_get_message read error\n");
			return -1;
        }

		*id = tmp[0];

		ret = uart_raw_read(tmp, 4);
		if (ret < 0) {
            printk(KERN_ERR "uart_get_message read error\n");
			return -1;
        }

#ifdef COMA_USE_ACK
		if (uart_send_ack() < 0)
			return -1;
#endif // COMA_USE_ACK

		*len = *(int *)tmp;

        if (*len <= 0 || *len > COMA_UART_BUF_SIZE) {
            printk(KERN_ERR "uart_get_message len %d\n", *len);
            goto start;
        }

		i = 0;
		p = buf;
		while (i < *len) {
			to_read = 20;
			if ((*len - i) < to_read)
				to_read = *len - i;
			ret = uart_raw_read(p, to_read);
			if (ret < 0)
				return -1;
			p += to_read;
			i += to_read;

#ifdef COMA_USE_ACK
			if (uart_send_ack() < 0)
				return -1;
#endif // COMA_USE_ACK
		}

		break;
	}

	return 0;
}


char uart_recv_buf[COMA_UART_BUF_SIZE];

static int
uart_thread_recv(void *data)
{
	//struct voice_chan *chan = (struct voice_chan *)data;
	char *c2l_buf;
	int id, len, ret;
	unsigned long flags;

	/* daemonize() disabled all signals, allow SIGTERM */
	allow_signal(SIGTERM);

	while (!signal_pending(current) && !kthread_should_stop()) {
		ret = uart_get_message(&id, uart_recv_buf, &len);
		if (ret < 0) {
            set_current_state(TASK_INTERRUPTIBLE);
            printk(KERN_ERR "uart_thread_recv: can not read from uart\n");
            continue;
        }

		/* Some sanity checks */
		if (len < 0 || kthread_should_stop() || signal_pending(current))
			break;


        if (id < 0 || id >= COMA_NUM_SERVICES) {
            set_current_state(TASK_INTERRUPTIBLE);
            printk(KERN_ERR "uart_thread_recv: wrong service id %d\n", id);
            continue;
        }

		//mutex_lock(c2l_fifo_mutex);
		if (/*!services[id].active ||*/ !services[id].c2l) {
            set_current_state(TASK_INTERRUPTIBLE);
            printk(KERN_ERR "uart_thread_recv: service is not registered %d\n", id);
			continue;
		}

		ret = cfifo_request(services[id].c2l, (void **)&c2l_buf, len);
		if (ret <= 0) {
            set_current_state(TASK_INTERRUPTIBLE);
            printk(KERN_ERR "uart_thread_recv: can not allocate cfifo buffer size %d\n", len);
            continue;
        }

		memcpy(c2l_buf, uart_recv_buf, len);
		cfifo_commit(services[id].c2l, len);

//        printk("uart_thread_recv commit len %d\n", len);

		local_save_flags(flags);
		local_irq_disable();
		writel((1<<3), IO_ADDRESS(DMW_PLICU_BASE) + DMW_PLICU_SW_CH_REQ);
		local_irq_restore(flags);

		//mutex_unlock(c2l_fifo_mutex);

		set_current_state(TASK_INTERRUPTIBLE);
	}

    printk(KERN_WARNING "uart_thread_recv EXIT\n");

    complete_and_exit(&uart_receiver_exit, 0);
}

static int
uart_thread_send(void *ignore)
{
	int ret, id, new;
	unsigned char *data;

	/* daemonize() disabled all signals, allow SIGTERM */
	allow_signal(SIGTERM);

	while (!signal_pending(current) && !kthread_should_stop()) {
		if (wait_event_interruptible(uart_send_wq, uart_new_message))
			return -ERESTARTSYS;
		uart_new_message = 0;

		do {
			new = 0;
			for (id = 0; id < COMA_NUM_SERVICES; id++) {
				if (/* !services[id].active ||*/ !services[id].l2c)
					continue;

				if (cfifo_get(services[id].l2c, (void **)&data) > 0) {
					new = 1;
					ret = uart_send_message(id, data);
					if (ret < 0)
						printk(KERN_ERR "uart_send_message error\n");
					cfifo_processed(services[id].l2c);
				}
			}
		} while (new);

		set_current_state(TASK_INTERRUPTIBLE);
	}

    printk(KERN_WARNING "uart_thread_send EXIT\n");

	complete_and_exit(&uart_sender_exit, 0);
}

static void
uart_stop_kthread(struct task_struct *task, struct completion *exit)
{
	int err;

	/*
	 * The thread maybe blocked in read()
	 * we will send SIGTERM to the kthread, to avoid 
	 * crashes we must hold the big kernel lock
	 */
	lock_kernel();
	err = kill_pid(task_pid(task), SIGTERM, 1);
	unlock_kernel();

	if (err)
		printk(KERN_ERR "%s(): Can't stop uart kernel thread!\n", __FUNCTION__);

	wait_for_completion(exit);
}

int
uart_kthreads_start(int fd)
{
	int ret;

	uart_fd = fd;

	uart_file = fget(uart_fd);
	if (uart_file == NULL)
		return -1;

	init_completion(&uart_receiver_exit);
	init_completion(&uart_sender_exit);

	init_waitqueue_head(&uart_wq);
	init_waitqueue_head(&uart_send_wq);

	/* create receiver thread */
	uart_receiver = kthread_create(uart_thread_recv, 0, "coma_uart_recv");
	if (IS_ERR(uart_receiver)) {
		return PTR_ERR(uart_receiver);
	}

	/* create sender thread */
	uart_sender = kthread_create(uart_thread_send, 0, "coma_uart_send");
	if (IS_ERR(uart_sender)) {
		ret = PTR_ERR(uart_sender);
		goto err_stop_recv;
	}

	/* start the threads */
	wake_up_process(uart_receiver);
	wake_up_process(uart_sender);

	return 0;

err_stop_recv:
	uart_stop_kthread(uart_receiver, &uart_receiver_exit);

	return ret;
}

void
uart_kthreads_stop(void)
{
	/*
	 * Stop the threads.
	 *
	 * As the receiver thread is heavily blocked by read, we have
	 * to kill it differently and harder.
	 */
	kthread_stop(uart_sender);
	uart_stop_kthread(uart_receiver, &uart_receiver_exit);

	fput(uart_file);
}

#endif /* CONFIG_CSS_UART */

/*
 * start-up and release
 */

int coma_setup(struct platform_device *pdev, struct cfifo *l2c, struct cfifo *c2l)
{
	int ret;

	if (coma_sysfs_init(pdev) != 0) {
		printk(KERN_ERR "coma: init sysfs filesystem\n");
	}

	services[COMA_SERVICE].id = COMA_SERVICE;
	services[COMA_SERVICE].name = "coma";
	services[COMA_SERVICE].active = 1;
	services[COMA_SERVICE].l2c = l2c;
	services[COMA_SERVICE].c2l = c2l;
	services[COMA_SERVICE].setup = NULL;
	services[COMA_SERVICE].process_message = coma_process_message;
	services[COMA_SERVICE].received_bytes = 0;
	services[COMA_SERVICE].sent_bytes = 0;

	if ((ret = coma_signaling_init()) != 0)
		return ret;

	return 0;
}
EXPORT_SYMBOL(coma_setup);

void coma_release(struct platform_device *pdev)
{
	int i;

	coma_sysfs_release(pdev);

	for (i = 1 ; i < COMA_NUM_SERVICES; i++)
		services[i].active = 0;

	coma_signaling_exit();
}
EXPORT_SYMBOL(coma_release);
