/*---------------------------------------------------------------------------------------------------------
 * driver/input/touchscreen/goodix_touch.c
 *
 * Copyright(c) 2010 Goodix Technology Corp.     
 *
 * This software is licensed under the terms of the GNU General Public
 * License version 2, as published by the Free Software Foundation, and
 * may be copied, distributed, and modified under those terms.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * Change Date: 2010.11.11, add point_queue's definiens.     
 *                             
 * Change Data: 2011.03.09, rewrite point_queue's definiens.  
 *                                                                                                  
 *---------------------------------------------------------------------------------------------------------*/
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/time.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/earlysuspend.h>
#include <linux/platform_device.h>
#include <linux/hrtimer.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/irq.h>
#include <mach/gpio.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/list.h>
#include <linux/earlysuspend.h>
#include <linux/hrtimer.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/platform_data/goodix-ts.h>

#define DEVICE_NAME  "Goodix GT80X"
#define GOODIX_I2C_NAME "Goodix-TS"

#define TOUCH_MAX_HEIGHT	7680
#define TOUCH_MAX_WIDTH		5120

#define SCREEN_MAX_HEIGHT	480
#define SCREEN_MAX_WIDTH	800

#define GOODIX_MULTI_TOUCH
#ifndef GOODIX_MULTI_TOUCH
#define MAX_FINGER_NUM 1
#else
#define MAX_FINGER_NUM 5
#endif

#if MAX_FINGER_NUM <= 3
#define READ_BYTES_NUM (1+2+MAX_FINGER_NUM*5)
#elif MAX_FINGER_NUM == 4
#define READ_BYTES_NUM (1+28)
#elif MAX_FINGER_NUM == 5
#define READ_BYTES_NUM (1+34)
#endif

#ifndef swap
#define swap(x, y) do { typeof(x) z = x; x = y; y = z; } while (0)
#endif

enum key_state {
	KEY_FLAG_UNKNOW,
	KEY_FLAG_UP,
	KEY_FLAG_DOWN,
};

#define MAX_FINGER_BUF (MAX_FINGER_NUM+4)

struct point_data {
	int status;
	unsigned int x;
	unsigned int y;
	unsigned char pressure;
	unsigned char id;
	struct list_head node;
};
struct point_queue {
	struct mutex use_mutex;
	struct mutex free_mutex;
	struct list_head use_head;
	struct list_head free_head;
};

struct goodix_ts_data {
	int retry;
	int bad_data;
	int panel_type;
	char phys[32];
	struct i2c_client *client;
	struct input_dev *input_dev;
	uint8_t use_irq;
	uint8_t finger_bit;
	uint8_t use_shutdown;
	uint32_t gpio_shutdown;
	uint32_t gpio_irq;
	uint32_t width;
	uint32_t height;
	uint32_t full_width;
	uint32_t full_height;
	uint32_t tswidth;
	uint32_t tsheight;
	uint8_t read_buf[READ_BYTES_NUM];
	struct hrtimer timer;
	struct work_struct work;
	struct early_suspend early_suspend;
	int (*power) (struct goodix_ts_data * ts, int on);
	struct point_data buf[MAX_FINGER_BUF];
	struct point_queue queue;
	struct workqueue_struct *wq;
};

static uint8_t ts_config_info[] = {
#if defined(CONFIG_TOUCHSCREEN_5POINT)
	0x30, 0x19, 0x05, 0x06, 0x28, 0x02, 0x14, 0x14, 0x10, 0x40, 0xB8,
	TOUCH_MAX_WIDTH  >> 8, TOUCH_MAX_WIDTH  & 0xFF,
	TOUCH_MAX_HEIGHT >> 8, TOUCH_MAX_HEIGHT & 0xFF,
	0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xE1, 0x00, 0x00, 0x00,
	0x00, 0x0D, 0xCF, 0x20, 0x03, 0x05, 0x83, 0x50, 0x3C, 0x1E, 0x28,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x01
#elif defined(CONFIG_TOUCHSCREEN_COMMON)
	0x30, 0x19, 0x05, 0x03, 0x28, 0x02, 0x14, 0x14, 0x10, 0x46, 0xB8,
	TOUCH_MAX_WIDTH  >> 8, TOUCH_MAX_WIDTH  & 0xFF,
	TOUCH_MAX_HEIGHT >> 8, TOUCH_MAX_HEIGHT & 0xFF,
	0xCB, 0xA9, 0x87, 0x65, 0x43, 0x21, 0x0D, 0xE1, 0x00, 0x00, 0x00,
	0x00, 0x4D, 0xC1, 0x20, 0x01, 0x01, 0x83, 0x50, 0x3C, 0x1E, 0xB4,
	0x00, 0x0A, 0x50, 0x82, 0x1E, 0x00, 0x64, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x01
#else
#error Please select your goodix touchscreen type
#endif
};

static int
add_point(struct point_queue *queue, unsigned char id)
{
	struct point_data *node;
	int ret = 0;

	if (id >= MAX_FINGER_NUM)
		return -1;
	if (list_empty(&queue->free_head)) {
		printk(KERN_ERR "%s free list is empty \n", __FUNCTION__);
		return -1;
	}
	node = list_first_entry(&queue->free_head, struct point_data, node);
	if (node) {
		list_del_init(&node->node);
	} else {
		printk(KERN_ERR "%s Can't get free node.\n", __FUNCTION__);
		return -1;
	}
	node->id = id;
	node->status = KEY_FLAG_DOWN; /* pressed down */
	list_add(&node->node, &queue->use_head);

	return ret;
}

static int
set_up_point(struct point_queue *queue, unsigned char id)
{
	struct point_data *p;
	int ret = 0;

	if (id >= MAX_FINGER_NUM)
		return -1;

	list_for_each_entry(p, &queue->use_head, node) {
		if (id == p->id) {
			p->status = KEY_FLAG_UP;
			ret = 0;
			break;
		}
	}

	return ret;
}

static void
analyse_points(struct point_queue *queue, unsigned char finger_bit,
               unsigned char finger)
{
#define POINT_DOWN_MASK 0x01
	unsigned char count;

	for (count = 0; (finger != 0) && (count < MAX_FINGER_NUM); count++) {
		if ((finger & POINT_DOWN_MASK) != 0) {
			/* current bit is 1, so position of finger has changed */
			if ((finger_bit & POINT_DOWN_MASK) != 0)
				set_up_point(queue, count); /* finger is UP */
			else
				add_point(queue, count);
		}
		finger >>= 1;
		finger_bit >>= 1;
	}
}

static void
delete_points(struct point_queue *queue)
{
	struct point_data *p = NULL;
	int status = 1;
	int loops = MAX_FINGER_BUF + 1;

	while (status && (loops--)) {
		status = 0;
		list_for_each_entry(p, &queue->use_head, node) {
			if (p->status != KEY_FLAG_DOWN) {
				status = 1;
				break;
			}
		}
		if (status) {
			list_del_init(&p->node);
			list_add(&p->node, &queue->free_head);
		}
	}
}

static int
i2c_read_bytes(struct i2c_client *client, uint8_t * buf, int len)
{
	struct i2c_msg msgs[2];
	int ret = -1;

	msgs[0].flags = !I2C_M_RD;
	msgs[0].addr = client->addr;
	msgs[0].len = 1;
	msgs[0].buf = &buf[0];

	msgs[1].flags = I2C_M_RD;
	msgs[1].addr = client->addr;
	msgs[1].len = len - 1;
	msgs[1].buf = &buf[1];

	ret = i2c_transfer(client->adapter, msgs, 2);

	return ret;
}

static int
i2c_write_bytes(struct i2c_client *client, uint8_t * data, int len)
{
	struct i2c_msg msg;
	int ret = -1;

	msg.flags = !I2C_M_RD;
	msg.addr = client->addr;
	msg.len = len;
	msg.buf = data;
	ret = i2c_transfer(client->adapter, &msg, 1);

	return ret;
}

static int
goodix_init_panel(struct goodix_ts_data *ts)
{
	int ret = -1;
	int count;

	/* There are some demo configs. May be changed as some different panels. */
	for (count = 5; count > 0; count--) {
		ret = i2c_write_bytes(ts->client, ts_config_info,
		                      sizeof(ts_config_info));
		if (ret == 1)
			break;
		else
			msleep(10);
	}

	return ret == 1 ? 0 : ret;
}

static int
goodix_read_version(struct goodix_ts_data *ts)
{
#define GT80X_VERSION_LENGTH 40
	int ret;
	/* command of reading Guitar's version */
	uint8_t version[2] = { 0x69, 0xff };
	/* store touchscreen version infomation */
	uint8_t version_data[GT80X_VERSION_LENGTH + 1];

	memset(version_data + 1, 0, GT80X_VERSION_LENGTH);
	ret = i2c_write_bytes(ts->client, version, 2);
	if (ret < 0)
		return ret;
	msleep(50);
	version_data[0] = 0x6A;
	ret = i2c_read_bytes(ts->client, version_data, GT80X_VERSION_LENGTH);
	if (ret < 0)
		return ret;
	dev_info(&ts->client->dev, "Guitar Version: %s\n", version_data + 1);
	version[1] = 0x00; /* cancel the command */
	i2c_write_bytes(ts->client, version, 2);

	return 0;
}

static void
goodix_ts_work_func(struct work_struct *work)
{
	struct point_data *p = NULL;
	uint8_t read_position = 0;
	uint8_t *point_data;
	uint8_t finger = 0; /* record which finger has changed */
	uint8_t check_sum = 0;
	int ret = -1;
	int count = 0;
	struct goodix_ts_data *ts;

	ts = container_of(work, struct goodix_ts_data, work);
	if (ts->use_shutdown && gpio_get_value(ts->gpio_shutdown))
		goto NO_ACTION; /* the data is invalid */

	point_data = ts->read_buf;

	do {
		ret = i2c_read_bytes(ts->client, ts->read_buf, READ_BYTES_NUM);
		if (ret <= 0)
			ts->retry++;

		if (ts->retry > 5) {
			dev_err(&ts->client->dev, "Repeated I2C transfer errors. Resetting device.\n");
			if (ts->power) {
				ts->power(ts, 0);
				ts->power(ts, 1);
			} else {
				goodix_init_panel(ts);
				msleep(260);
			}
			ts->retry = 0;
			goto XFER_ERROR;
		}
	} while (ret <= 0);

	if (!ts->use_irq) {
		switch (point_data[1] & 0x1f) {
		case 0:
			break;
		case 1:
			for (count = 1; count < 8; count++)
				check_sum += (int)point_data[count];
			read_position = 8;
			break;
		case 2:
		case 3:
			for (count = 1; count < 13; count++)
				check_sum += (int)point_data[count];
			read_position = 13;
			break;
		default:	/* (point_data[1]& 0x1f) > 3 */
			for (count = 1; count < 34; count++)
				check_sum += (int)point_data[count];
			read_position = 34;
		}
		if (check_sum != point_data[read_position])
			goto XFER_ERROR;
	}
	/* The bits indicate which fingers are pressed down */
	point_data[1] &= 0x1f;
	finger = ts->finger_bit ^ point_data[1];
	if (finger == 0 && point_data[1] == 0)
		goto NO_ACTION; /* no fingers and no action */
	else if (finger != 0) {
		/* the same as last time; check which points are DOWN or UP */
		analyse_points(&ts->queue, ts->finger_bit, finger);

		if (list_empty(&ts->queue.use_head))
			goto NO_ACTION;
		/*
		   if(ts->finger_list.head == NULL)
		   goto NO_ACTION;
		   else
		   dev_dbg(&ts->client->dev, "fingers count:%d\n", ts->finger_list.length);
		 */
	}

	list_for_each_entry(p, &ts->queue.use_head, node) {
		/* for (p = ts->finger_list.head; p != NULL; p = p->next) { */
		if (p->status == KEY_FLAG_UP) {
			p->x = p->y = 0;
			p->pressure = 0;
			continue;
		}
		if (p->id < 3)
			read_position = p->id * 5 + 3;
		else
			read_position = 29;

		if (p->id != 3) {
			p->x =
			    (unsigned int)(point_data[read_position] << 8) +
			    (unsigned int)(point_data[read_position + 1]);
			p->y =
			    (unsigned int)(point_data[read_position + 2] << 8) +
			    (unsigned int)(point_data[read_position + 3]);
			p->pressure = point_data[read_position + 4];
		}
#if MAX_FINGER_NUM > 3
		else {
			p->x =
			    (unsigned int)(point_data[18] << 8) +
			    (unsigned int)(point_data[25]);
			p->y =
			    (unsigned int)(point_data[26] << 8) +
			    (unsigned int)(point_data[27]);
			p->pressure = point_data[28];
		}
#endif

		/* map touch point to screen window */
		/* p->x = (TOUCH_MAX_WIDTH - p->x)*SCREEN_MAX_WIDTH/TOUCH_MAX_WIDTH; */
		p->x = p->x * ts->full_width / ts->tswidth;
		p->y = p->y * ts->full_height / ts->tsheight;

		if (!((ts->full_width == ts->width) && (ts->full_height == ts->height))) {
			/* map from full size to emulated size */
			/* if (p->x,p->y) is out of the rectangle [(0,0), (ts->width, ts->height)] */
			/* then input_report_abs will drop it */
			unsigned int xstart = (ts->full_width - ts->width) / 2;
			unsigned int ystart = (ts->full_height - ts->height) / 2;

			p->x = p->x - xstart;
			p->y = p->y - ystart;
		}

		swap(p->x, p->y);
	}

#ifndef GOODIX_MULTI_TOUCH
	p = list_first_entry(&ts->queue.use_head, struct point_data, node);
	if (p->status == KEY_FLAG_DOWN) {
		input_report_abs(ts->input_dev, ABS_X, p->x);
		input_report_abs(ts->input_dev, ABS_Y, p->y);
	}
	input_report_abs(ts->input_dev, ABS_PRESSURE, p->pressure);
	input_report_key(ts->input_dev, BTN_TOUCH, p->state);
#else

	/* ABS_MT_TOUCH_MAJOR is used as ABS_MT_PRESSURE in android. */
	list_for_each_entry(p, &ts->queue.use_head, node) {
		if (p->status == KEY_FLAG_DOWN) {
			input_report_abs(ts->input_dev, ABS_MT_POSITION_X,
					 p->x);
			input_report_abs(ts->input_dev, ABS_MT_POSITION_Y,
					 p->y);
		}
		input_report_abs(ts->input_dev, ABS_MT_TRACKING_ID, p->id);
		input_report_abs(ts->input_dev, ABS_MT_TOUCH_MAJOR,
				 p->pressure);
		input_report_abs(ts->input_dev, ABS_MT_WIDTH_MAJOR,
				 p->pressure);
		input_mt_sync(ts->input_dev);
	}
#endif
	input_sync(ts->input_dev);
	delete_points(&ts->queue);
	ts->finger_bit = point_data[1] & 0x1f; /* restore last press state */

XFER_ERROR:
NO_ACTION:
	if (ts->use_irq)
		enable_irq(ts->client->irq);
}

static enum hrtimer_restart goodix_ts_timer_func(struct hrtimer *timer)
{
	struct goodix_ts_data *ts =
	    container_of(timer, struct goodix_ts_data, timer);

	queue_work(ts->wq, &ts->work);
	if (ts->timer.state != HRTIMER_STATE_INACTIVE)
		hrtimer_start(&ts->timer, ktime_set(0, 16000000),
			      HRTIMER_MODE_REL);
	return HRTIMER_NORESTART;
}

static irqreturn_t goodix_ts_irq_handler(int irq, void *dev_id)
{
	struct goodix_ts_data *ts = dev_id;

	disable_irq_nosync(ts->client->irq);
	queue_work(ts->wq, &ts->work);
	return IRQ_HANDLED;
}

static int goodix_ts_power(struct goodix_ts_data *ts, int on)
{
	int ret = 0;

	if (ts == NULL || (ts && !ts->use_shutdown))
		return -1;

	switch (on) {
	case 0:
		gpio_set_value(ts->gpio_shutdown, 1);
		msleep(5);
		if (gpio_get_value(ts->gpio_shutdown)) /* has been suspended */
			ret = 0;
		break;
	case 1:
		gpio_set_value(ts->gpio_shutdown, 0);
		msleep(5);
		if (gpio_get_value(ts->gpio_shutdown)) /* has been woken up */
			ret = -1;
		else
			msleep(200);
		break;
	}
	dev_dbg(&ts->client->dev, "Set Guitar's Shutdown %s. Ret:%d.\n",
		on ? "HIGH" : "LOW", ret);
	return ret;
}

static int goodix_ts_suspend(struct i2c_client *client, pm_message_t mesg)
{
	int ret;
	struct goodix_ts_data *ts = i2c_get_clientdata(client);

	if (ts->use_irq)
		disable_irq(client->irq);
	else if (ts->timer.state)
		hrtimer_cancel(&ts->timer);
	ret = cancel_work_sync(&ts->work);
	if (ret && ts->use_irq) /* irq was disabled twice */
		enable_irq(client->irq);

	if (ts->power) {
		ret = ts->power(ts, 0);
		if (ret < 0)
			dev_warn(&client->dev, "%s power off failed\n",
				 __FUNCTION__);
	}

	return 0;
}

static int goodix_ts_resume(struct i2c_client *client)
{
	int ret;
	struct goodix_ts_data *ts = i2c_get_clientdata(client);

	if (ts->power) {
		ret = ts->power(ts, 1);
		if (ret < 0)
			dev_warn(&client->dev, "power on failed\n");
	}

	if (ts->use_irq)
		enable_irq(client->irq);
	else
		hrtimer_start(&ts->timer, ktime_set(1, 0), HRTIMER_MODE_REL);
	return 0;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void goodix_ts_early_suspend(struct early_suspend *h)
{
	struct goodix_ts_data *ts;
	ts = container_of(h, struct goodix_ts_data, early_suspend);

	goodix_ts_suspend(ts->client, PMSG_SUSPEND);
}

static void goodix_ts_late_resume(struct early_suspend *h)
{
	struct goodix_ts_data *ts;
	ts = container_of(h, struct goodix_ts_data, early_suspend);

	goodix_ts_resume(ts->client);
}
#endif

static int
goodix_ts_probe(struct i2c_client *client,
                const struct i2c_device_id *id)
{
	struct goodix_ts_data *ts;
	uint8_t version[2] = { 0x69, 0xff };
	struct point_queue *q;
	struct point_data *pd;
	int ret = 0;
	int retry = 0;
	int count;
	struct goodix_ts_pdata *pdata;

	if (!i2c_check_functionality(client->adapter, I2C_FUNC_I2C)) {
		dev_err(&client->dev, "I2C functionality insufficient.\n");
		ret = -ENODEV;
		goto err_check_functionality_failed;
	}

	ts = kzalloc(sizeof(*ts), GFP_KERNEL);
	if (ts == NULL) {
		ret = -ENOMEM;
		goto err_alloc_data_failed;
	}

	pdata = client->dev.platform_data;
	BUG_ON(!pdata);

	if (!pdata->screen_width)
		pdata->screen_width = SCREEN_MAX_WIDTH;
	if (!pdata->screen_height)
		pdata->screen_height = SCREEN_MAX_HEIGHT;
	if (!pdata->full_screen_width)
		pdata->full_screen_width = pdata->screen_width;
	if (!pdata->full_screen_height)
		pdata->full_screen_height = pdata->screen_height;

	ts->tsheight = TOUCH_MAX_HEIGHT;
	ts->tswidth = TOUCH_MAX_WIDTH;
	ts->width = pdata->screen_width;
	ts->height = pdata->screen_height;
	ts->full_width = pdata->full_screen_width;
	ts->full_height = pdata->full_screen_height;

	ts->gpio_shutdown = pdata->gpio_shutdown;
	if (ts->gpio_shutdown) {
		ret = gpio_request(ts->gpio_shutdown, GOODIX_I2C_NAME);
		if (ret < 0) {
			dev_err(&client->dev, "Failed to request gpio.\n");
			goto err_gpio_request;
		}
		gpio_direction_output(ts->gpio_shutdown, 0);
		ret = gpio_get_value(ts->gpio_shutdown);
		if (ret) {
			dev_err(&client->dev, "Failed to power up touchscreen.\n");
			goto err_i2c_failed;
		}
		ts->use_shutdown = 1;
		msleep(25); /* waiting for initialization of Guitar */
	}

	for (retry = 0; retry < 5; retry++) {
		ret = i2c_write_bytes(client, version, 2);
		if (ret > 0)
			break;
	}
	if (ret < 0) {
		dev_err(&client->dev, "Repeated I2C errors.\n");
		goto err_i2c_failed;
	}
	if (ts->use_shutdown)
		gpio_set_value(ts->gpio_shutdown, 1);

	INIT_WORK(&ts->work, goodix_ts_work_func);
	ts->client = client;
	i2c_set_clientdata(client, ts);
	pdata = client->dev.platform_data;

	ts->input_dev = input_allocate_device();
	if (ts->input_dev == NULL) {
		ret = -ENOMEM;
		dev_err(&client->dev, "Failed to allocate input device.\n");
		goto err_input_dev_alloc_failed;
	}

	ts->input_dev->evbit[0] = BIT_MASK(EV_SYN) | BIT_MASK(EV_KEY) | BIT_MASK(EV_ABS);
#ifndef GOODIX_MULTI_TOUCH
	ts->input_dev->keybit[BIT_WORD(BTN_TOUCH)] = BIT_MASK(BTN_TOUCH);
	ts->input_dev->absbit[0] = BIT(ABS_X) | BIT(ABS_Y);

	input_set_abs_params(ts->input_dev, ABS_X, 0, ts->height, 0, 0);
	input_set_abs_params(ts->input_dev, ABS_Y, 0, ts->width, 0, 0);
	input_set_abs_params(ts->input_dev, ABS_PRESSURE, 0, 255, 0, 0);
#else
	ts->input_dev->absbit[0] = BIT_MASK(ABS_MT_TRACKING_ID) | BIT_MASK(ABS_MT_TOUCH_MAJOR) | BIT_MASK(ABS_MT_WIDTH_MAJOR) | BIT_MASK(ABS_MT_POSITION_X) | BIT_MASK(ABS_MT_POSITION_Y);	// for android
	input_set_abs_params(ts->input_dev, ABS_MT_WIDTH_MAJOR, 0, 255, 0, 0);
	input_set_abs_params(ts->input_dev, ABS_MT_TOUCH_MAJOR, 0, 255, 0, 0);
	input_set_abs_params(ts->input_dev, ABS_MT_POSITION_X, 0, ts->height, 0,
			     0);
	input_set_abs_params(ts->input_dev, ABS_MT_POSITION_Y, 0, ts->width, 0,
			     0);
#endif
	ts->wq = create_workqueue(dev_name(&client->dev));
	if (!ts->wq) {
		dev_err(&client->dev, "Failed to create workqueue.\n");
		ret = -ENOMEM;
		goto err_input_register_device_failed;
	}
	q = &ts->queue;
	mutex_init(&q->use_mutex);
	mutex_init(&q->free_mutex);
	INIT_LIST_HEAD(&q->free_head);
	INIT_LIST_HEAD(&q->use_head);
	pd = ts->buf;
	for (count = 0; count < MAX_FINGER_BUF; count++) {
		INIT_LIST_HEAD(&pd->node);
		list_add(&pd->node, &q->free_head);
		pd++;
	}
	sprintf(ts->phys, "input/ts)");
	ts->input_dev->name = DEVICE_NAME;
	ts->input_dev->phys = ts->phys;
	ts->input_dev->id.bustype = BUS_I2C;
	ts->input_dev->id.vendor = 0xDEAD;
	ts->input_dev->id.product = 0xBEEF;
	ts->input_dev->id.version = 0x1103;

	ret = input_register_device(ts->input_dev);
	if (ret) {
		dev_err(&client->dev,
			"Probe: Unable to register %s input device\n",
			ts->input_dev->name);
		goto err_work_queue;
	}
	ts->use_irq = 0;
	ts->retry = 0;
	ts->bad_data = 0;

	if (ts->use_shutdown) {
		gpio_set_value(ts->gpio_shutdown, 0);
		ts->power = goodix_ts_power;

		msleep(30);
	}

	ret = goodix_init_panel(ts);
	if (ret != 0)
		goto err_init_godix_ts;

	goodix_read_version(ts);

	ts->gpio_irq = pdata->gpio_irq;

	if (client->irq) {
		ret = request_irq(client->irq, goodix_ts_irq_handler,
		                  IRQ_TYPE_EDGE_RISING, client->name, ts);
		if (ret != 0) {
			dev_err(&client->dev, "Cannot allocate interrupt.\n");
			gpio_direction_input(ts->gpio_irq);
			gpio_free(ts->gpio_irq);
		} else {
			ts->use_irq = 1;
		}
	}

	if (!ts->use_irq) {
		hrtimer_init(&ts->timer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
		ts->timer.function = goodix_ts_timer_func;
		hrtimer_start(&ts->timer, ktime_set(1, 0), HRTIMER_MODE_REL);
	}
	flush_workqueue(ts->wq);

#ifdef CONFIG_HAS_EARLYSUSPEND
	ts->early_suspend.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN + 1;
	ts->early_suspend.suspend = goodix_ts_early_suspend;
	ts->early_suspend.resume = goodix_ts_late_resume;
	register_early_suspend(&ts->early_suspend);
#endif
	dev_info(&client->dev, "Started %s in %s mode\n",
		 ts->input_dev->name, ts->use_irq ? "interrupt" : "polling");
	return 0;

err_init_godix_ts:
	if (ts->use_irq) {
		free_irq(client->irq, ts);
		gpio_free(ts->gpio_irq);
	}
err_work_queue:
	if (ts->wq)
		destroy_workqueue(ts->wq);
err_input_register_device_failed:
	input_free_device(ts->input_dev);
err_input_dev_alloc_failed:
	i2c_set_clientdata(client, NULL);
err_i2c_failed:
	if (ts->use_shutdown) {
		gpio_direction_input(ts->gpio_shutdown);
		gpio_free(ts->gpio_shutdown);
	}
err_gpio_request:
	kfree(ts);
err_alloc_data_failed:
err_check_functionality_failed:
	return ret;
}

static int goodix_ts_remove(struct i2c_client *client)
{
	struct goodix_ts_data *ts = i2c_get_clientdata(client);
#ifdef CONFIG_HAS_EARLYSUSPEND
	unregister_early_suspend(&ts->early_suspend);
#endif
	if (ts->use_irq) {
		free_irq(client->irq, ts);
		gpio_free(ts->gpio_irq);
	} else
		hrtimer_cancel(&ts->timer);

	if (ts->use_shutdown) {
		gpio_direction_input(ts->gpio_shutdown);
		gpio_free(ts->gpio_shutdown);
	}

	i2c_set_clientdata(client, NULL);
	input_unregister_device(ts->input_dev);
	if (ts->wq)
		destroy_workqueue(ts->wq);
	if (ts->input_dev)
		kfree(ts->input_dev);
	kfree(ts);
	return 0;
}

static const struct i2c_device_id goodix_ts_id[] = {
	{ GOODIX_I2C_NAME, 0 },
	{ }
};

static struct i2c_driver goodix_ts_driver = {
	.probe = goodix_ts_probe,
	.remove = goodix_ts_remove,
#ifndef CONFIG_HAS_EARLYSUSPEND
	.suspend = goodix_ts_suspend,
	.resume = goodix_ts_resume,
#endif
	.id_table = goodix_ts_id,
	.driver = {
		.name = GOODIX_I2C_NAME,
		.owner = THIS_MODULE,
	},
};

static int __devinit goodix_ts_init(void)
{
	return i2c_add_driver(&goodix_ts_driver);
}

static void __exit goodix_ts_exit(void)
{
	i2c_del_driver(&goodix_ts_driver);
}

late_initcall(goodix_ts_init);
module_exit(goodix_ts_exit);

MODULE_DESCRIPTION("Goodix Touchscreen Driver");
MODULE_LICENSE("GPL");
