/*
 * Driver for DMW96 Generig DMA Controller
 *
 * Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#ifndef __DMW96DMA_H
#define __DMW96DMA_H

#include <linux/kernel.h>

struct dmw96dma_list;
struct dmw96dma_ch_config;

/**
 * struct dmw96dma_platform_data - Configuration parameters
 * @timeout: Transfer timeout in AHB/64 clock cycles
 * @desc_port: Port which has access to descriptors in RAM (0: Port A, 1: Port B)
 * @num_channels: Number of hardware channels
 * @ch_configs: Pointer to default channel configurations
 * @ch_priorities: Pointer to channel arbitration priorities (0..31, 0=highest)
 * @rotate_priority: Arbitration priority for image rotation (0..31, 0=highest)
 * @memcpy_priority: Arbitration priority for memcpy (0..31, 0=highest)
 * @memcpy_base: Physical address of memcpy bounce buffer (256 bytes)
 *
 * Each hardware channel must have its own priority, that is there must be no
 * two channel (including 2D rotate and memcpy) on the same priority.
 */
struct dmw96dma_platform_data {
	u32 timeout;
	u32 desc_port;
	int num_channels;
	struct dmw96dma_ch_config *ch_configs;
	u8 *ch_priorities;
	u8 rotate_priority;
	u8 memcpy_priority;
	u32 memcpy_base;
};

enum dmw96dma_mode {
	DMW96DMA_MODE_SINGLE_BLOCK   = 0x00,
	DMW96DMA_MODE_CHAINED_BLOCK  = 0x01,
	DMW96DMA_MODE_HW_SINGLE      = 0x02,
	DMW96DMA_MODE_HW_CONSECUTIVE = 0x03,
	DMW96DMA_MODE_2D_SWAP        = 0x04,
	DMW96DMA_MODE_2D_ROTATE      = 0x05,
};

enum dmw96dma_width {
	DMW96DMA_WIDTH_8  = 0,
	DMW96DMA_WIDTH_16 = 1,
	DMW96DMA_WIDTH_32 = 2,
};

enum dmw96dma_rotate {
	DMW96DMA_ROTATE_90_CCW = 0,
	DMW96DMA_ROTATE_90_CW  = 1,
	DMW96DMA_ROTATE_180_CW = 2,
};

/**
 * struct dmw96dma_ch_config - Hardware channel configuration
 * @timeout: Enable timeout handling
 * @direction: Transfer direction (0: A->B, 1: B->A)
 * @hw_port: Port where the peripheral is connected (0: A, 1: B)
 * @transfer_length: Length of each transfer in 32bit words (0=one word, 1=two
 *                   words, ...; actual transfer requests must be a multiple of
 *                   this value)
 * @consecutive: Number of consecutive peripheral registers (only valid for
 *               DMW96DMA_MODE_HW_CONSECUTIVE mode, 0=one register, 1=two
 *               registers, ...)
 * @width: Width of peripheral register(s)
 * @mode: Transfer mode
 * @base: Physical base address of peripheral
 *
 * Each hardware DMA requestor has a dedicated configuration which can be read
 * and written via dmw96dma_config_get() / dmw96dma_config_set(). The
 * configuration is used when a transfer descriptor list is assembled. Changing
 * the configuration after the transfer list was filled has no influence.
 */
struct dmw96dma_ch_config {
	u8 timeout   : 1;
	u8 direction : 1;
	u8 hw_port   : 1;
	u8 transfer_length;
	u8 consecutive;
	enum dmw96dma_width width;
	enum dmw96dma_mode mode;
	u32 base;
};

/**
 * dmw96dma_callback - Transfer callback
 * @status: error status of transfer (0=ok)
 * @context: user supplied context when transfer was set up
 *
 * Callback when a transfer descriptor or a transfer list has been finished.
 * The callback will be directly called from a tasklet context so keep
 * processing inside the callback to a minimum.
 *
 * ATTENTION: Do not manipulate (for example cancel/free) your list from a
 * transfer request (processed) callback. You may start/stop other transfer
 * lists, though. This restriction does not apply to transfer list (finish)
 * callbacks.
 */
typedef void (*dmw96dma_callback)(int status, void *context);

/**
 * struct dmw96dma_image - 2D image
 * @format: Pixel format
 * @coherent: 0: kernel buffer, 1: coherent memory
 * @width: Image width in pixels
 * @height: Image height in pixels
 * @data.coherent: Physical address of image buffer (if coherent == 1)
 * @data.buf: Kernel buffer of image (if coherent == 0)
 *
 * The image buffer could either be a kernel buffer or a coherent DMA region. A
 * kernel buffer is passed with coherent==0 and the virtual address in
 * data.buf. The physical pointer to a coherent memory buffer is passed with
 * coherent==1 in data.coherent.
 */
struct dmw96dma_image {
	enum dmw96dma_width format : 8;
	int coherent : 1;
	int width;
	int height;
	struct {
		dma_addr_t coherent;
		void *buf;
	} data;
};

struct dmw96dma_rect {
	int x;
	int y;
	int width;
	int height;
};

/**
 * dmw96dma_config_get - Get DMA HW channel configuration
 * @hw_channel: DMA HW channel
 * @cfg: pointer to configuration struct
 *
 * The current configuration of the channel is copied into cfg.
 */
int dmw96dma_config_get(unsigned int hw_channel, struct dmw96dma_ch_config *cfg);

/**
 * dmw96dma_config_set - Set DMA HW channel configuration
 * @hw_channel: DMA HW channel
 * @cfg: pointer to configuration struct
 *
 * Change the channel configuration to cfg values.
 */
int dmw96dma_config_set(unsigned int hw_channel, struct dmw96dma_ch_config *cfg);


/**
 * dmw96dma_alloc_io_list - Allocate IO transfer list
 * @circular: Circular, self repeating transfer list; one shot otherwise
 * @hw_channel: DMA HW channel of the IO device
 * @finish: Completion/error callback (never invoked for circular lists)
 * @context: User defined context pointer, passed unchanged to finish callback
 *
 * This allocates an empty io transfer list. Before using it the caller has to
 * add transfer request via dmw96dma_add_io_transfer(). After the list has been
 * assembled it must be passed to dmw96dma_submit() for execution. The list may
 * be reused after it has been processed.
 */
struct dmw96dma_list *dmw96dma_alloc_io_list(int circular,
	unsigned int hw_channel, dmw96dma_callback finish, void *context);

/**
 * dmw96dma_alloc_image_list - Allocate 2D image operation list
 * @finish: Completion/error callback
 * @context: User defined context pointer, passed unchanged to finish callback
 *
 * This allocates an empty image operation DMA list. Before using it the caller
 * has to add rotate requests via dmw96dma_add_rotate(). After the list has
 * been assembled it must be passed to dmw96dma_submit() for execution. The
 * list may be reused after it has been processed.
 */
struct dmw96dma_list *dmw96dma_alloc_image_list(dmw96dma_callback finish,
	void *context);

/*
 * dmw96dma_alloc_memcpy_list - Allocate memcpy operation list
 * @finish: Completion/error callback
 * @context: User defined context pointer, passed unchanged to finish callback
 *
 * This allocates an empty memcpy DMA list. Before using it the caller has to
 * add memcpy requests via dmw96dma_add_memcpy_buf() or
 * dmw96dma_add_memcpy_coherent().  After the list has been assembled it must
 * be passed to dmw96dma_submit() for execution. The list may be reused after
 * it has been processed.
 */
struct dmw96dma_list *dmw96dma_alloc_memcpy_list(dmw96dma_callback finish,
	void *context);

/**
 * dmw96dma_free_list - Dispose a DMA request list
 * @list: DMA request list
 *
 * Free a DMA list which was allocated by dmw96dma_alloc_io_list() or
 * dmw96dma_alloc_image_list(). Do not free a list while it is being processed!
 */
int dmw96dma_free_list(struct dmw96dma_list *list);

/**
 * dmw96dma_add_io_transfer_buf - Add IO transfer request to list
 * @list: DMA request list, allocated by dmw96dma_alloc_io_list()
 * @buffer: kernel memory buffer
 * @length: length of transfer (must be a multiple of 4)
 * @processed: request processed callback
 * @context: User defined context pointer, passed unchanged to processed callback
 *
 * The configuration for the requested hardware channel is taken from the
 * internal configuration list and an appropriate transfer request is added to
 * the transfer list. The direction of the transfer is also derived from
 * channel configuration.
 */
int dmw96dma_add_io_transfer_buf(struct dmw96dma_list *list, void *buffer,
	size_t length, dmw96dma_callback processed, void *context);

/**
 * dmw96dma_add_io_transfer_coherent - Add IO transfer request to list
 * @list: DMA request list, allocated by dmw96dma_alloc_io_list()
 * @buffer: physical address of coherent buffer
 * @length: length of transfer (must be a multiple of 4)
 * @processed: request processed callback
 * @context: User defined context pointer, passed unchanged to processed callback
 *
 * The configuration for the requested hardware channel is taken from the
 * internal configuration list and an appropriate transfer request is added to
 * the transfer list. The direction of the transfer is also derived from
 * channel configuration.
 */
int dmw96dma_add_io_transfer_coherent(struct dmw96dma_list *list, dma_addr_t buffer,
	size_t length, dmw96dma_callback processed, void *context);

/**
 * dmw96dma_add_rotate - Add 2D image rotation request to list
 * @list: DMA request list, allocated by dmw96dma_alloc_image_list()
 * @src: source image
 * @dst: destination image
 * @src_rect: source rectangle (NULL = whole source image)
 * @dst_rect: destination rectangle (NULL = whole destination image)
 * @degree: requested rotation
 * @processed: request processed callback
 * @context: User defined context pointer, passed unchanged to processed callback
 *
 * The source and destination images must not be the same. If src_rect is NULL
 * the whole source image is rotated. If dst_rect is NULL the whole destination
 * image is used as output rectangle. In any case the destination rectangle
 * must match the source image rectangle after the rotation is applied. All
 * dimensions are given in pixels but, depending on the pixel format, they have
 * to be a multiple of 64 (8bpp), 32 (16bpp) or 16 (32bpp).
 */
int dmw96dma_add_rotate(struct dmw96dma_list *list,
	struct dmw96dma_image *src,
	struct dmw96dma_image *dst,
	struct dmw96dma_rect *src_rect,
	struct dmw96dma_rect *dst_rect,
	enum dmw96dma_rotate degree,
	dmw96dma_callback processed,
	void *context);

/**
 * dmw96dma_add_memcpy_buf - Add memcpy request to list
 * @list: DMA request list, allocated by dmw96dma_alloc_memcpy_list()
 * @src: source kernel memory buffer
 * @dst: destination kernel memory buffer
 * @length: length of transfer (must be a multiple of 4)
 * @done: request processed callback
 * @context: User defined context pointer, passed unchanged to processed callback
 *
 * This variant copies between kmalloc'ed buffer. Preferably the transfer
 * length is a multiple of 256 bytes, otherwise additional transfers are
 * necessary to copy any remainder.
 */
int dmw96dma_add_memcpy_buf(struct dmw96dma_list *list, void *src, void *dst,
	size_t length, dmw96dma_callback done, void *context);

/**
 * dmw96dma_add_memcpy_coherent - Add memcpy request to list
 * @list: DMA request list, allocated by dmw96dma_alloc_memcpy_list()
 * @src: physical address of coherent source buffer
 * @src: physical address of coherent destination buffer
 * @length: length of transfer (must be a multiple of 4)
 * @done: request processed callback
 * @context: User defined context pointer, passed unchanged to processed callback
 *
 * This variant copies between coherent DMA buffers. Preferably the transfer
 * length is a multiple of 256 bytes, otherwise additional transfers are
 * necessary to copy any remainder.
 */
int dmw96dma_add_memcpy_coherent(struct dmw96dma_list *list, dma_addr_t src,
	dma_addr_t dst, size_t length, dmw96dma_callback done, void *context);

/**
 * dmw96dma_submit - Execute a DMA list
 * @list: DMA request list
 *
 * Start execution of the DMA request list. Until the request is finished
 * (finish callback) the list is owned by the driver.
 */
int dmw96dma_submit(struct dmw96dma_list *list);

/**
 * dmw96dma_cancel - Cancel DMA request
 * @list: currently executed DMA request list
 *
 * Cancel a running DMA request list. The processed and finish callbacks of the
 * list will be invoked but there is no information available on how far the
 * list has already been processed by the hardware.
 */
int dmw96dma_cancel(struct dmw96dma_list *list);

#endif

