/*
 * Copyright (c) 2009 Vlad Sachnovich <vlad.sachnovich@dsp.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 */

#include <linux/init.h>
#include <linux/err.h>
#include <linux/platform_device.h>
#include <linux/jiffies.h>
#include <linux/slab.h>
#include <linux/time.h>
#include <linux/wait.h>
#include <linux/moduleparam.h>
#include <linux/delay.h>
#include <linux/input.h>
#include <mach/hardware.h>
#include <mach/platform.h>
#include <linux/spinlock.h>
#include <linux/interrupt.h>
#include <linux/mfd/dp52/core.h>

#include "tsc_dp52.h"

#define TP_INTERRUPT_MASK (1 << 11)

#define DRIVER_NAME "dp52-touchscreen"

static int g_first_press = 1;
static long	a0=1;
module_param(a0, long, 0644);
MODULE_PARM_DESC(a0, "a0");
static long	a1=0;
module_param(a1, long, 0644);
MODULE_PARM_DESC(a1, "a1");
static long	a2=0;
module_param(a2, long, 0644);
MODULE_PARM_DESC(a2, "a2");
static long	a3=0;
module_param(a3, long, 0644);
MODULE_PARM_DESC(a3, "a3");
static long	a4=1;
module_param(a4, long, 0644);
MODULE_PARM_DESC(a4, "a4");
static long	a5=0;
module_param(a5, long, 0644);
MODULE_PARM_DESC(a5, "a5");
static long	a6=1;
module_param(a6, long, 0644);
MODULE_PARM_DESC(a6, "a6");

static unsigned long timer_interval = 1; /* In 10 msec units */
module_param(timer_interval, ulong, 0644);
MODULE_PARM_DESC(timer_interval, "Number of jiffies to wait between polling of the touch-screen.");

static unsigned long untouch_timeout = 15; /* In 10 msec units */
module_param(untouch_timeout, ulong, 0644);
MODULE_PARM_DESC(untouch_timeout, "Number of jiffies to wait before signalling un-touch event.");

static ssize_t
show_resolution(struct device *dev, struct device_attribute *attr, char *buf)
{
	int ret;

	ret = sprintf(buf, "%lu (1/s)\n", HZ / timer_interval);

	return ret;
}

static ssize_t
store_resolution(struct device *dev, struct device_attribute *attr,
                 const char *buf, size_t count)
{
	if (strict_strtoul(buf, 10, &timer_interval))
		return -EINVAL;

	timer_interval = HZ / timer_interval;

	return count;
}

static ssize_t
show_release(struct device *dev, struct device_attribute *attr, char *buf)
{
	int ret;

	ret = sprintf(buf, "%lu (ms)\n", untouch_timeout * (1000/HZ));

	return ret;
}

static ssize_t
store_release(struct device *dev, struct device_attribute *attr,
              const char *buf, size_t count)
{
	if (strict_strtoul(buf, 10, &untouch_timeout))
		return -EINVAL;

	untouch_timeout /= (1000/HZ);

	return count;
}

struct device_attribute resolution = \
	__ATTR(resolution, S_IRUGO|S_IWUSR, show_resolution, store_resolution);

struct device_attribute release_timeout = \
	__ATTR(release_timeout, S_IRUGO|S_IWUSR, show_release, store_release);

static struct attribute *main_attributes[] = {
	&resolution.attr,
	&release_timeout.attr,
	NULL
};

static const struct attribute_group main_group = {
	.attrs = main_attributes,
};

struct tsc_dp52
{
	struct dp52 *dp52;
	struct input_dev *input;
	char phys[32];
	int cond;
	int irq;	
	wait_queue_head_t waitqueue;
};

#define MAX_12BIT ((1<<12)-1)
#define MAX_8BIT  ((1<<8)-1)

static struct tsc_dp52_platform_data g_tsc_dp52_info = {
	.x_min			= 0,
	.x_max			= MAX_8BIT,
	.y_min			= 0,
	.y_max			= MAX_8BIT,
	.pressure_min		= 0,
	.pressure_max		= 2000,
};

static void
tsc_dp52_disable_irq(struct dp52 *dp52 )
{
	dp52_update(dp52, DP52_ICU_MASK, TP_INTERRUPT_MASK, ~(TP_INTERRUPT_MASK));
}

static void
tsc_dp52_enable_irq(struct dp52 *dp52 )
{
	dp52_update(dp52, DP52_ICU_MASK, TP_INTERRUPT_MASK, TP_INTERRUPT_MASK);
}

/* Note: this function is called from schedule_work context (not interrupt context) */
static irqreturn_t
dp52_tsc_irq(int event, void *dev)
{
	struct tsc_dp52 *tsc_dp52 = (struct tsc_dp52 *)dev;
	struct dp52 *dp52 = tsc_dp52->dp52;

	tsc_dp52_disable_irq(dp52); /* disable the irq */

	tsc_dp52->cond = 1;
	wake_up_interruptible(&tsc_dp52->waitqueue); /* Signal the worker thread */

	return IRQ_HANDLED;
}

int tsc_dp52_read_values( struct tsc_dp52_measure* pvalues, struct dp52 *dp52 )
{
	pvalues->x	= dp52_read(dp52,DP52_AUX_TPDX) & 0x7FFF;
	pvalues->y	= dp52_read(dp52,DP52_AUX_TPDY) & 0x7FFF;
	pvalues->z1	= dp52_read(dp52,DP52_AUX_TPDZ1) & 0x7FFF;
	pvalues->z2	= dp52_read(dp52,DP52_AUX_TPDZ2) & 0x7FFF;
	return 0;
}

static void
tsc_dp52_get_values(struct tsc_dp52_measure *values, struct dp52 *dp52 )
{
	tsc_dp52_read_values(values,dp52);
	dp52_update(dp52, DP52_TP_CTL2, DP52_TP_RDY, ~(DP52_TP_RDY));
}

static int
tsc_dp52_report_events(struct input_dev *input, struct dp52 *dp52)
{
	struct tsc_dp52_measure values;
	unsigned char adjusted_x;
	unsigned char adjusted_y;
	int xtemp,ytemp;

	tsc_dp52_get_values(&values,dp52);
	if (values.z1 || g_first_press) {

		/* adjusted the x and y using linear algorithm*/
		xtemp = values.x; ytemp = values.y;	
		adjusted_x = ( a2 + a0*xtemp + a1*ytemp ) / a6;
		adjusted_y = ( a5 + a3*xtemp + a4*ytemp ) / a6;

		/* report position and pressure */
		input_report_abs(input, ABS_X, adjusted_x);
		input_report_abs(input, ABS_Y, adjusted_y);
		input_report_abs(input, ABS_PRESSURE, 100);
		input_sync(input);
	}

	return values.z1;
}

static int
tsc_dp52_thread_code(void *data)
 {
	struct tsc_dp52 *tsc_dp52 = (struct tsc_dp52 *)data;
	struct input_dev *input = tsc_dp52->input;
	struct dp52 *dp52 = tsc_dp52->dp52;
	struct tsc_dp52_measure values;
	long ret;

	daemonize(DRIVER_NAME);
	allow_signal(SIGTERM);

	while (1) {
		tsc_dp52->cond = 0;
		ret = wait_event_interruptible(tsc_dp52->waitqueue, tsc_dp52->cond);
		if (ret < 0)
			return -EINTR;

		dp52_aux_mutex_lock(dp52);

		tsc_dp52_get_values(&values,dp52);

		input_report_key(input, BTN_TOUCH, 1);

		while (1) {
			ret = schedule_timeout_interruptible(timer_interval);
			if (ret < 0)
				return -EINTR;

			tsc_dp52_report_events(input,dp52);
			g_first_press = 0;
			tsc_dp52_enable_irq(dp52);

			tsc_dp52->cond = 0;
			ret = wait_event_interruptible_timeout(tsc_dp52->waitqueue,
			                                       tsc_dp52->cond,
			                                       untouch_timeout);
			if (!ret) { /* timeout */
				/* Did we miss an interrupt because the
				 * processing of the dp52-workqueue is delayed
				 * by other interrupt handlers or tasklets?
				 * Alternatively, we could check bit 6 in the
				 * interrupt status (DP52_AUX_RAWPMINTSTAT).
				 */
				if (dp52_read(dp52, DP52_TP_CTL2) & DP52_TP_RDY) {
					tsc_dp52_report_events(input,dp52);
					continue;
				}
				input_report_key(input, BTN_TOUCH, 0);
				input_report_abs(input, ABS_PRESSURE, 0);
				input_sync(input);
				g_first_press = 1;
				dp52_aux_mutex_unlock(dp52);
				break;
			} else if (ret > 0) { /* DP52-interrupt */
				tsc_dp52_report_events(input,dp52);
				continue;
			} else {
				return -EINTR;
			}
		}
	}
	return 0;
}

void
tsc_dp52_bypass_init(struct dp52 *dp52)
{
	dp52_aux_mutex_lock(dp52);

	/* Clock Control Register 0 */
	/* Set Auxiliary A/D analog part is enabled &&  Bandgap circuit is enabled */
	dp52_update(dp52, DP52_AUX_EN, 0x3, 0x3);

	/* Power Management Touch Panel Control 2 Register */
	dp52_write(dp52, DP52_TP_CTL2, 0x1); /* Amplifier is always enabled */

	/* Power Management Touch Panel Control 1 Register */
	dp52_update(dp52, DP52_TP_CTL1, 0x1000, 0x1000);

	/* Workaround for DP ES0 touch panel issue (can be ignored in ES1) */
	/* Power Management Touch Panel Control 2 Register */
	if (dp52_get_chip_version(dp52) == 0) {
		dp52_write(dp52, DP52_TP_CTL2, 0x040C & ~(DP52_TP_CTL2_AUTOMODE) & ~(DP52_TP_ST_TIME));

		/* Power Management Touch Panel Control 1 Register */
		dp52_write(dp52, DP52_TP_CTL1, 0x6005);

		dp52_write(dp52, DP52_TP_CTL1, 0x0000);
	}

	dp52_write(dp52, DP52_AUX_INTSTAT, ~(0x40)); /* clear interrupt */

	/* Start touch panel automatic machine */
	dp52_write(dp52, DP52_TP_CTL2, 0x040C /* pre-scale by 8; SEQ_Z1&Z2 */|
	                  DP52_TP_CTL2_AUTOMODE | DP52_TP_ST_TIME);

	dp52_update(dp52, DP52_AUX_INTMSKN, DP52_AUX_INT_TPEN, DP52_AUX_INT_TPEN);

	dp52_aux_mutex_unlock(dp52);
}

static int __devinit
tsc_dp52_probe(struct platform_device *pdev)
{
	struct tsc_dp52 *tsc_dp52;
	struct input_dev *input_dev;
	struct tsc_dp52_platform_data *pdata = &g_tsc_dp52_info;
	struct dp52 *dp52 = dev_get_drvdata(pdev->dev.parent);
	int err;

	if (!pdata) {
		dev_err(&pdev->dev, "no platform data?\n");
		return -ENODEV;
	}

	tsc_dp52 = kzalloc(sizeof(struct tsc_dp52), GFP_KERNEL);
	input_dev = input_allocate_device();
	if (!tsc_dp52 || !input_dev) {
		err = -ENOMEM;
		goto err_free_mem;
	}

	tsc_dp52->dp52 = dp52;
	tsc_dp52->input = input_dev;

	if ((tsc_dp52->irq = platform_get_irq_byname(pdev, "dp52-touchscreen-irq")) < 0) {
		err = tsc_dp52->irq;
		dev_err(&pdev->dev, "no IRQ!\n");
		goto err_free_mem;
	}
	init_waitqueue_head(&tsc_dp52->waitqueue);

	snprintf(tsc_dp52->phys, sizeof(tsc_dp52->phys), "dp52-touchscreen/input1");

	input_dev->name = DRIVER_NAME;
	input_dev->phys = tsc_dp52->phys;

	platform_set_drvdata(pdev, tsc_dp52);

	set_bit(EV_KEY, input_dev->evbit);
	set_bit(EV_ABS, input_dev->evbit);
	set_bit(EV_SYN, input_dev->evbit);
	set_bit(BTN_TOUCH, input_dev->keybit);

	input_dev->keybit[BIT_WORD(BTN_TOUCH)] = BIT_MASK(BTN_TOUCH);

	input_dev->absmin[ABS_X] = pdata->x_min;
	input_dev->absmax[ABS_X] = pdata->x_max;
	input_dev->absfuzz[ABS_X] = 0;
	input_dev->absflat[ABS_X] = 0;
	set_bit(ABS_X, input_dev->absbit);

	input_dev->absmin[ABS_Y] = pdata->y_min;
	input_dev->absmax[ABS_Y] = pdata->y_max;
	input_dev->absfuzz[ABS_Y] = 0;
	input_dev->absflat[ABS_Y] = 0;
	set_bit(ABS_Y, input_dev->absbit);

	input_dev->absmin[ABS_PRESSURE] = pdata->pressure_min;
	input_dev->absmax[ABS_PRESSURE] = pdata->pressure_max;
	input_dev->absfuzz[ABS_PRESSURE] = 0;
	input_dev->absflat[ABS_PRESSURE] = 0;
	set_bit(ABS_PRESSURE, input_dev->absbit);

	err = input_register_device(input_dev);
	if (err) {
		dev_err(&pdev->dev, "cannot register input device\n");
		goto err_free_mem;
	}

	kernel_thread(tsc_dp52_thread_code, tsc_dp52, CLONE_KERNEL);

	tsc_dp52_bypass_init(dp52);


	/* Register sysfs hooks */
	err = sysfs_create_group(&input_dev->dev.kobj, &main_group);
	if (err) {
		dev_err(&pdev->dev, "cannot create sysfs group\n");
		goto err_free_device;
	}

	/* Register irqhandler */
	err = request_threaded_irq(tsc_dp52->irq, NULL, dp52_tsc_irq, IRQF_SHARED,
	                           "dp52-touchscreen", tsc_dp52);
	if (err) {
		dev_err(&pdev->dev, "failed to request IRQ %d: %d\n", tsc_dp52->irq, err);
		goto err_remove_group;
	}

	dev_info(&pdev->dev, "registered touchscreen\n");


	return 0;

err_remove_group:
	sysfs_remove_group(&input_dev->dev.kobj, &main_group);
err_free_device:
	input_unregister_device(input_dev);
err_free_mem:
	input_free_device(input_dev);
	kfree(tsc_dp52);

	return err;
}

static int
tsc_dp52_suspend(struct platform_device *dev, pm_message_t state)
{
	return 0;
}

static int
tsc_dp52_resume(struct platform_device *dev)
{
	return 0;
}

static int __devexit
tsc_dp52_remove(struct platform_device *pdev)
{
	struct tsc_dp52 *tsc_dp52 = platform_get_drvdata(pdev);
	struct dp52 *dp52 = tsc_dp52->dp52;

	input_unregister_device(tsc_dp52->input);
	kfree(tsc_dp52);

	/* disable touchpad; it might interfere with sampling (auto-mode) */
	dp52_write(dp52,DP52_TP_CTL1, 0);
	dp52_write(dp52,DP52_TP_CTL2, 0);

	return 0;
}

static void
tsc_dp52_shutdown(struct platform_device *pdev)
{
	struct tsc_dp52 *tsc_dp52 = platform_get_drvdata(pdev);
	struct dp52 *dp52 = tsc_dp52->dp52;

	/* disable touchpad; it might interfere with sampling (auto-mode) */
	dp52_write(dp52,DP52_TP_CTL1, 0);
	dp52_write(dp52,DP52_TP_CTL2, 0);
}

static struct platform_driver dp_touch_driver = {
	.probe		= tsc_dp52_probe,
	.remove		= __devexit_p(tsc_dp52_remove),
	.suspend	= tsc_dp52_suspend,
	.resume		= tsc_dp52_resume,
	.shutdown	= tsc_dp52_shutdown,
	.driver		= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
	},
};

static int __init
tscdp52_init(void)
{
	return platform_driver_register(&dp_touch_driver);
}

module_init(tscdp52_init);

static void __exit
tscdp52_exit(void)
{
	platform_driver_unregister(&dp_touch_driver);
}

module_exit(tscdp52_exit);

MODULE_DESCRIPTION("DP52 TouchScreen Driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:dp52-touchscreen");
