/*
 * dmw96cam.c
 *
 * Copyright (C) 2010 DSPG
 *
 * Originally based on the DMW69 2 camera driver.
 *
 * Written by Dudi Dolev

 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/io.h>
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/kernel.h>
#include <linux/slab.h>
#include <linux/interrupt.h>
#include <linux/videodev2.h>
#include <linux/version.h>
#include <linux/platform_device.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/dmw96ciu_common.h>
#include <media/v4l2-common.h>
#include <media/v4l2-ioctl.h>
#include <media/videobuf-core.h>
#include <media/v4l2-dev.h>
#include <media/videobuf-dma-contig.h>
#include <media/dmw96ciu.h>
#include <media/dmw96ciu_w.h>
#include <media/ov3640.h>
#include "dmw96ciu_regs.h"
#define DMW96CAM_VERSION KERNEL_VERSION(0, 0, 0)
#define CLEAR(x) memset (&(x), 0, sizeof (x))

/* global variables */
static struct dmw96cam_device *dmw96cam;
extern void __iomem *dmw96ciu_regs;
extern struct ciu_device *dmw96ciu;
unsigned int ciu_decimator_factor[2] = {1,2};
#define CIU_DECIMATOR_FACTOR_SIZE ( sizeof(ciu_decimator_factor)/sizeof(ciu_decimator_factor[0]) )
#define CIU_DECIMATOR_FACTOR_1 (ciu_decimator_factor[0]) 
#define CIU_DECIMATOR_FACTOR_2 (ciu_decimator_factor[1]) 
/*
A dummy slave that has no functionality. Helps
managing slaves in the CIU camera driver; no need to check for NULL
pointers.
*/

static struct v4l2_int_slave dummy_slave = {
/* Dummy pointer to avoid underflow in find_ioctl. */
.ioctls = (void *)0x80000000,
.num_ioctls = 0,
};

static struct v4l2_int_device dummy = {
.type = v4l2_int_type_slave,
.u = {
.slave = &dummy_slave,
},
};

struct v4l2_int_device *v4l2_int_device_dummy(void)
{
return &dummy;
}
EXPORT_SYMBOL_GPL(v4l2_int_device_dummy); 


char* format_to_string(int fmt)
{
	switch (fmt)
	{
	case V4L2_PIX_FMT_SGRBG10:
		return "V4L2_PIX_FMT_SGRBG10";
	case V4L2_PIX_FMT_RGB565:
		return "V4L2_PIX_FMT_RGB565";
	case V4L2_PIX_FMT_RGB565X:
		return "V4L2_PIX_FMT_RGB565X";
	case V4L2_PIX_FMT_YUYV:
		return "V4L2_PIX_FMT_YUYV";
	case V4L2_PIX_FMT_UYVY:
		return "V4L2_PIX_FMT_UYVY";
	case V4L2_PIX_FMT_RGB555:
		return "V4L2_PIX_FMT_RGB555";
	case V4L2_PIX_FMT_RGB555X:
		return "V4L2_PIX_FMT_RGB555X";
	case V4L2_PIX_FMT_NV16:
		return "V4L2_PIX_FMT_NV16";
	case V4L2_PIX_FMT_NV61:
		return "V4L2_PIX_FMT_NV61";
	case V4L2_PIX_FMT_NV12:
		return "V4L2_PIX_FMT_NV12";
	case V4L2_PIX_FMT_NV21:
		return "V4L2_PIX_FMT_NV21";
	default:
		return "UNKNOWN";
	}

	return "UNKNOWN";
}


/*
 *
 * Sensor handling.
 *
 */

/**
 * dmw96cam_slave_power_set - set slave power state
 * @vdev: per-video device data structure
 * @power: new power state
 */
static int dmw96cam_slave_power_set(struct dmw96cam_videodev *vdev,
				       enum v4l2_power power,
				       int mask)
{
	int rval = 0, i = 0;
    BUG_ON(!mutex_is_locked(&vdev->mutex));

#ifdef DMW96CAM_POWEROFF_DELAY
	vdev->power_state_wish = -1;
#endif
	PDEBUG("\n");
	for (i = 0; i <= DMW96CAM_SLAVE_FLASH; i++) {
		if (vdev->slave[i] == v4l2_int_device_dummy())
			continue;

		if (!(mask & (1 << i))
		    || ( (power == vdev->power_state[i]) &&  (power != V4L2_POWER_ON) ))
			continue;
		
		rval = vidioc_int_s_power(vdev->slave[i], power);
        
		if (rval && power != V4L2_POWER_OFF) {
			power = V4L2_POWER_OFF;
			goto out;
		}

		vdev->power_state[i] = power;
	}
	
	return 0;

out:
    for (i--; i >= 0; i--) {
		if (vdev->slave[i] == v4l2_int_device_dummy())
			continue;

		if (!(mask & (1 << i)))
			continue;

		vidioc_int_s_power(vdev->slave[i], power);
		vdev->power_state[i] = power;
	}

	return rval;
}

#ifdef DMW96CAM_POWEROFF_DELAY
static void dmw96cam_slave_power_work(struct work_struct *work)
{
	struct dmw96cam_videodev *vdev =
		container_of(work, struct dmw96cam_videodev, poweroff_work);
	PDEBUG("\n");
	mutex_lock(&vdev->mutex);

	if (vdev->power_state_wish != -1)
		dmw96cam_slave_power_set(vdev, vdev->power_state_wish,
					    vdev->power_state_mask);

	mutex_unlock(&vdev->mutex);
}

static void dmw96cam_slave_power_timer(unsigned long ptr)
{
	struct dmw96cam_videodev *vdev = (void *)ptr;

	schedule_work(&vdev->poweroff_work);
}

/**
 * dmw96cam_slave_power_suggest - delayed power state change
 *
 * @vdev: per-video device data structure
 * @power: new power state
 */
static void dmw96cam_slave_power_suggest(struct dmw96cam_videodev *vdev,
					    enum v4l2_power power,
					    int mask)
{
	BUG_ON(!mutex_is_locked(&vdev->mutex));

	del_timer_sync(&vdev->poweroff_timer);

	vdev->power_state_wish = power;
	vdev->power_state_mask = mask;

	mod_timer(&vdev->poweroff_timer, jiffies + DMW96CAM_POWEROFF_DELAY);
}
#else /* DMW96CAM_POWEROFF_DELAY */
#define dmw96cam_slave_power_suggest(a, b, c) do {} while(0)
#endif /* DMW96CAM_POWEROFF_DELAY */



/**
 * struct dmw96cam_fh - per-filehandle data structure
 * @vbq_lock: spinlock for the videobuf queue
 * @vbq: V4L2 video buffer queue structure
 * @field_count: field counter for videobuf_buffer
 * @vdev: our /dev/video specific structure
 */
struct dmw96cam_fh {
	spinlock_t vbq_lock; /* spinlock for the videobuf queue */
	struct videobuf_queue vbq;
	atomic_t field_count;
	struct dmw96cam_videodev *vdev;
};




/**
 * dmw96cam_update_vbq - Updates VBQ with completed input buffer
 * @vb: ptr. to standard V4L2 video buffer structure
 *
 * Updates video buffer queue with completed buffer passed as
 * input parameter.  Also updates CIU H3A timestamp and field count
 * statistics.
 */
void dmw96cam_vbq_complete(struct videobuf_buffer *vb, void *priv)
{
	struct dmw96cam_fh *fh = (struct dmw96cam_fh *)priv;
	CIU_START();
    do_gettimeofday(&vb->ts);
	
	vb->field_count = atomic_add_return(2, &fh->field_count);

	PDEBUG("vb->i = %d completed\n", vb->i );
	CIU_END();
	wake_up(&vb->done);
	
}


/**
 * dmw96cam_vbq_setup - Calcs size and num of buffs allowed in queue
 * @vbq: ptr. to standard V4L2 video buffer queue structure
 * @cnt: ptr to location to hold the count of buffers to be in the queue
 * @size: ptr to location to hold the size of a frame
 *
 */
static int dmw96cam_vbq_setup(struct videobuf_queue *vbq, unsigned int *cnt,
				 unsigned int *size)
{
	struct dmw96cam_fh *fh = (struct dmw96cam_fh*)vbq->priv_data;
	struct dmw96cam_videodev *vdev = fh->vdev;
	PDEBUG("\n");
	if (*cnt <= 0)
		*cnt = VIDEO_MAX_FRAME;	/* supply a default number of buffers */

	if (*cnt > VIDEO_MAX_FRAME)
		*cnt = VIDEO_MAX_FRAME;

	/*sizeimage can be set when the device was opened or after setting fmt*/
	*size = PAGE_ALIGN(vdev->pix.sizeimage + vdev->pix.sizeimage/16 + CIU_BUFFER_HEADER);
    
    return 0;
	
}

/**
 * dmw96cam_vbq_release - Free resources for input VBQ and VB
 * @vbq: ptr. to standard V4L2 video buffer queue structure
 * @vb: ptr to standard V4L2 video buffer structure
 *
 * Unmap and free all memory associated with input VBQ and VB, also
 * unmap the address in CIU MMU.  Reset the VB state.
 */
static void dmw96cam_vbq_release(struct videobuf_queue *vbq,
				    struct videobuf_buffer *vb)
{
	CIU_START();
	if (!vbq->streaming) {
		ciu_vbq_release(vbq, vb);
		videobuf_dma_contig_free(vbq, vb);
        vb->state = VIDEOBUF_NEEDS_INIT;
	}
	CIU_END();
	return;
}

/**
 * dmw96cam_vbq_prepare - V4L2 video ops buf_prepare handler
 * @vbq: ptr. to standard V4L2 video buffer queue structure
 * @vb: ptr to standard V4L2 video buffer structure
 * @field: standard V4L2 field enum
 *
 * Verifies there is sufficient locked memory for the requested
 * buffer, or if there is not, allocates, locks and initializes
 * it.
 */
static int dmw96cam_vbq_prepare(struct videobuf_queue *vbq,
				   struct videobuf_buffer *vb,
				   enum v4l2_field field)
{
	struct dmw96cam_fh *fh = vbq->priv_data;
	struct dmw96cam_videodev *vdev = fh->vdev;
	int err = 0;
	CIU_START();
	/*
	 * Accessing pix here is okay since it's constant while
	 * streaming is on (and we only get called then).
	 */
	if (vb->baddr) {
		/* This is a userspace buffer. */
		if (vdev->pix.sizeimage > vb->bsize){
			/* The buffer isn't big enough. */
			return -EINVAL;
		}
	} else {
		if (vb->state != VIDEOBUF_NEEDS_INIT
		    && vdev->pix.sizeimage > vb->bsize) {
			/*
			 * We have a kernel bounce buffer that has
			 * already been allocated.
			 */
			dmw96cam_vbq_release(vbq, vb);
		}
	}
	PDEBUG("vdev->pix.sizeimage=%d vdev->pix.width=%d vdev->pix.height=%d\n", vdev->pix.sizeimage, vdev->pix.width , vdev->pix.height);
	vb->size = vdev->pix.sizeimage;
	vb->width = vdev->pix.width;
	vb->height = vdev->pix.height;
	vb->field = field;

	if (vb->state == VIDEOBUF_NEEDS_INIT)
		err = videobuf_iolock(vbq, vb, NULL);

	if (!err){
		vb->state = VIDEOBUF_PREPARED;
	} else {
		dmw96cam_vbq_release(vbq, vb);
	}
	CIU_END();
	return err;
}

/**
 * dmw96cam_vbq_queue - V4L2 video ops buf_queue handler
 * @vbq: ptr. to standard V4L2 video buffer queue structure
 * @vb: ptr to standard V4L2 video buffer structure
 *
 * Maps the video buffer to sgdma and through the ciu, sets
 * the ciu buffer done callback and sets the video buffer state
 * to active.
 */
static void dmw96cam_vbq_queue(struct videobuf_queue *vbq,
				  struct videobuf_buffer *vb)
{
	struct dmw96cam_fh *fh = (struct dmw96cam_fh *)vbq->priv_data;

	vb->state = VIDEOBUF_ACTIVE;
	CIU_START();
    ciu_buf_queue(vb, dmw96cam_vbq_complete, (void *)fh);
	CIU_END();
	
}

static struct videobuf_queue_ops dmw96cam_vbq_ops = {
	.buf_setup = dmw96cam_vbq_setup,
	.buf_prepare = dmw96cam_vbq_prepare,
	.buf_queue = dmw96cam_vbq_queue,
	.buf_release = dmw96cam_vbq_release,
};

/*
 *
 * IOCTL interface.
 *
 */

/**
 * vidioc_querycap - V4L2 query capabilities IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @cap: ptr to standard V4L2 capability structure
 *
 * Fill in the V4L2 capabliity structure for the camera device
 */
static int vidioc_querycap(struct file *file, void *fh,
			   struct v4l2_capability *cap)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	CIU_START();
	strlcpy(cap->driver, CAM_NAME, sizeof(cap->driver));
	strlcpy(cap->card, vdev->vfd->name, sizeof(cap->card));
	cap->version = DMW96CAM_VERSION;
	if (vdev->vdev_sensor != v4l2_int_device_dummy())
		cap->capabilities = V4L2_CAP_VIDEO_CAPTURE | V4L2_CAP_STREAMING;
	CIU_END();
	return 0;
}

/**
 * vidioc_enum_fmt_vid_cap - V4L2 enumerate format capabilities IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @f: ptr to standard V4L2 format description structure
 *
 * Fills in enumerate format capabilities information for sensor (if SOC
 * sensor attached) or CIU (if raw sensor attached).
 */
static int vidioc_enum_fmt_vid_cap(struct file *file, void *fh,
				   struct v4l2_fmtdesc *f)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval;
	unsigned int j = 0;
	struct v4l2_fmtdesc fmt1;
    CIU_START();
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	memcpy(&fmt1,f,sizeof(fmt1));

	rval = vidioc_int_enum_fmt_cap(vdev->vdev_sensor, &fmt1);
	if (rval) {
		return rval;
	}

	for (j = 0; ; j++) {

		struct v4l2_fmtdesc fmt2;

		fmt2.type =  fmt1.type;
		fmt2.index = j;

		rval = ciu_enum_fmt_cap(&fmt2);

		if (rval)
			return rval;

		PDEBUG("frms.index=%d frms.pixel_format=%s\n",fmt2.index ,format_to_string(fmt2.pixelformat));
		if (fmt2.pixelformat == fmt1.pixelformat){
			*f = fmt1;
			PDEBUG("found common default format for sensor and ciu frms.index=%d frms.pixel_format=%s\n",fmt2.index ,format_to_string(fmt2.pixelformat));
			return 0;
		}
	
	}
	CIU_END();
    return rval;
}

/**
 * vidioc_g_fmt_vid_cap - V4L2 get format capabilities IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @f: ptr to standard V4L2 format structure
 *
 * Fills in format capabilities for sensor (if SOC sensor attached) or CIU
 * (if raw sensor attached).
 */
static int vidioc_g_fmt_vid_cap(struct file *file, void *fh,
				struct v4l2_format *f)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	CIU_START();
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	mutex_lock(&vdev->mutex);
	f->fmt.pix = vdev->pix;
	mutex_unlock(&vdev->mutex);
	CIU_END();
	return 0;
}

static int get_ciu_sensor_default_fmt(struct dmw96cam_videodev *vdev, struct v4l2_pix_format* pix_tmp)
{
	unsigned int i,j;
	int rval = -1;
	pix_tmp->pixelformat = -1;
	PDEBUG("\n");
	/*	Compare bwtween the application fixel format and the one which is supported by the CIU */
	for (i = 0; ; i++) {

		struct v4l2_fmtdesc fmt1;

		fmt1.type =  V4L2_BUF_TYPE_VIDEO_CAPTURE;
		fmt1.index = i;

		rval = ciu_enum_fmt_cap(&fmt1);
		if (rval){
			printk("Finished passing all the CIU formats, no common default format with the sensor\n");
			rval=0;
			break;
		}

		PDEBUG("frms.index=%d frms.pixel_format=%s\n",i ,format_to_string(fmt1.pixelformat));

		/*	Compare bwtween the wanted fixel formt and the one which is supported by the sensor 
		if not supported by the sensor set the default format	*/
		
		for (j = 0; ; j++) {
	
			struct v4l2_fmtdesc fmt2;
	
			fmt2.type =  V4L2_BUF_TYPE_VIDEO_CAPTURE;
			fmt2.index = j;
	
			rval = vidioc_int_enum_fmt_cap(vdev->vdev_sensor, &fmt2);
			if (rval){
				rval=0;
				break;
			}

			PDEBUG("frms.index=%d frms.pixel_format=%s\n",fmt2.index ,format_to_string(fmt2.pixelformat));
			if (fmt2.pixelformat == fmt1.pixelformat){
				pix_tmp->pixelformat = fmt1.pixelformat;
				PDEBUG("found common default format for sensor and ciu frms.index=%d frms.pixel_format=%s\n",fmt2.index ,format_to_string(fmt2.pixelformat));
				return 0;
			}
        
		}
		
	}

	return -1;
}


static int get_ciu_sensor_common_framesizes(struct dmw96cam_videodev *vdev, struct v4l2_pix_format* pix_max, struct v4l2_pix_format* pix_min)
{
    struct v4l2_frmsizeenum frms;
    int rval=0;
	struct v4l2_pix_format ciu_max , ciu_min;
	unsigned int sq_max = 0, sq_min = 0xffffffff, sq = 0;
	//unsigned int factor[]={1,2,4};
	unsigned int  i=0, height=0, width=0;
	CIU_START();
	frms.index = 0;
	frms.pixel_format = pix_max->pixelformat;

	rval = ciu_enum_framesize(&frms);
	if (rval) {
		PDEBUG("unable to query frame sizes which is supported by CIU\n");	
		return rval;
	}

	/*We know CIU is stepwise*/
	ciu_max.height = frms.stepwise.max_height;
	ciu_max.width = frms.stepwise.max_width;

	ciu_min.height = frms.stepwise.min_height;
	ciu_min.width = frms.stepwise.min_width;

	rval = vidioc_int_enum_framesizes(vdev->vdev_sensor, &frms);
	if (rval) {
		PDEBUG("unable to query frame sizes which is supported by camera sensor\n");	
		return rval;
	}

	if (frms.type == V4L2_FRMSIZE_TYPE_DISCRETE)
	{
		unsigned int size_index = 0;

		for (size_index = 0; ; size_index++) {

			frms.index = size_index;
            
			/* query from the sensor the supported discrete sizes*/
			rval = vidioc_int_enum_framesizes(vdev->vdev_sensor, &frms);
			if (rval) {
				PDEBUG("Invalid requested format or reach to max discrete sizes\n");
				break;
			}
			
			for (i=0 ; i < CIU_DECIMATOR_FACTOR_SIZE ; i++) {
				
				width = frms.discrete.width / ciu_decimator_factor[i] ;
				height = frms.discrete.height / ciu_decimator_factor[i];
				
				sq = width * height;
	
				if (ciu_max.width >= width &&
					ciu_max.height >= height &&
					sq >= sq_max ){
	
					sq_max = sq;
					pix_max->width = width;
					pix_max->height = height;
	
				}
	
				if (ciu_min.width <= width &&
					ciu_min.height <= height &&
					sq <= sq_min){
	
					sq_min = sq;
					pix_min->width = width;
					pix_min->height = height;
				}

			}
				
		}
		
	}
	else if (frms.type == V4L2_FRMSIZE_TYPE_STEPWISE)
	{

		pix_max->width = frms.stepwise.max_width <= ciu_max.width ? frms.stepwise.max_width :  ciu_max.width;
		pix_max->height = frms.stepwise.max_height <= ciu_max.height ? frms.stepwise.max_height :  ciu_max.height;


		pix_min->width = frms.stepwise.min_width >= ciu_min.width ? frms.stepwise.min_width :  ciu_min.width;
		pix_min->height = frms.stepwise.min_height >= ciu_min.height ? frms.stepwise.min_height :  ciu_min.height;

	}
	CIU_END();
	return 0;

}


static int get_ciu_sensor_user_fmt(struct dmw96cam_videodev *vdev, struct v4l2_pix_format *wanted_pix )
{
	unsigned int size_index;
	int was_found = -1;
	int rval;
	CIU_START();

	/* TODO There is need to explor all the sensor format per each CIU format*/

	/*	Compare bwtween the application fixel format and the one which is supported by the CIU */
	for (size_index = 0; ; size_index++) {

		struct v4l2_fmtdesc fmt1;

		fmt1.type =  V4L2_BUF_TYPE_VIDEO_CAPTURE;
		fmt1.index = size_index;

		rval = ciu_enum_fmt_cap(&fmt1);
		if (rval){
            return -1;
		}

		PDEBUG("frms.index=%d frms.pixel_format=%s\n",fmt1.index ,format_to_string(fmt1.pixelformat));

		if (fmt1.pixelformat == wanted_pix->pixelformat){
			PDEBUG("found common pixel fmt for user and ciu frms.index=%d frms.pixel_format=%s\n",fmt1.index ,format_to_string(fmt1.pixelformat));
			was_found = 1;
			break;
		}
	}

	/*	Compare bwtween the wanted fixel formt and the one which is supported by the sensor 
	if not supported by the sensor set the default format	*/
	if (was_found) {

		for (size_index = 0; ; size_index++) {
	
			struct v4l2_fmtdesc fmt;
	
			fmt.type =  V4L2_BUF_TYPE_VIDEO_CAPTURE;
			fmt.index = size_index;
	
			rval = vidioc_int_enum_fmt_cap(vdev->vdev_sensor, &fmt);
			if (rval){
                return -1;
			}
	
			PDEBUG("frms.index=%d frms.pixel_format=%s\n",fmt.index ,format_to_string(fmt.pixelformat));
			if ((fmt.pixelformat == wanted_pix->pixelformat) && was_found){
				PDEBUG("found common pixel fmt for user sensor and ciu: frms.index=%d frms.pixel_format=%s\n",fmt.index ,format_to_string(fmt.pixelformat));
				return 0;
			}
		}
	}

	return -1;

}


int get_stepwise_ciu_sensor_user_dim(unsigned int* ciu_min,
									 unsigned int* ciu_max,
									 unsigned int* ciu_step,
									 unsigned int* sensor_min,
									 unsigned int* sensor_max,
									 unsigned int* sensor_step,
									 unsigned int* wanted,
									 unsigned int* best)
{
	unsigned int common_step = 0;
	unsigned int dim_min = 0;
	unsigned int dim_max = 0;
	PDEBUG("\n");
	if ( *ciu_min % *ciu_step != 0 || *ciu_max % *ciu_step != 0 ||
		  *sensor_min % *sensor_step != 0 || *sensor_max % *sensor_step != 0)
	{
			printk("Unable to calculate sensor or ciu is dimention are not divided by their correspoting steps\n");
			return -1;
	}


	if (  ( max( *ciu_step , *sensor_step ) % min( *ciu_step , *sensor_step ) ) == 0 ) 
		common_step = max( *ciu_step , *sensor_step );
    else
		common_step = *ciu_step * *sensor_step;

	if (common_step == 0)
	{
		printk("common step for sensor and for ciu is invalid\n");
		return -1;
	}

    dim_min = max (*sensor_min , *ciu_min) ;
	dim_max = min (*sensor_max , *ciu_max) ;

	if(common_step > dim_max)
	{
		printk("common step for sensor and ciu is bigger then the maximun sensor or ciu supported\n");
		return -1;
	}

	if (dim_min > common_step)
		dim_min = (dim_min/common_step) * common_step;
	else
		dim_min = common_step;


	dim_max = (dim_max/common_step) * common_step;

	if ( (dim_min < *ciu_min) ||
		 (dim_max > *ciu_max) ||
		 (dim_min < *sensor_min) ||
		 (dim_max > *sensor_max) ||
		 (common_step > dim_max) )
	{
		printk("common step for sensor and ciu is bigger then the maximun supported\n");
		return -1;
	}


    *best = (*wanted / common_step) * common_step;

	if (*best <= dim_min)
	 *best = dim_min;
	if (*best >= dim_max)
	 *best = dim_max;

	PDEBUG("dim_min = %d\n",dim_min);
	PDEBUG("dim_max = %d\n",dim_max);
	PDEBUG("common_step = %d\n\n",common_step);
	PDEBUG("*best = %d\n\n",*best);

	return 0;

}


static int get_common_fmt(struct dmw96cam_videodev *vdev, struct v4l2_pix_format* wanted_pix)
{
	int rval = 0;
	struct v4l2_pix_format ciu_sensor_default_pix;

	if (!vdev || !wanted_pix) 
		return -EINVAL;

	CIU_START();
	/*Get the default format for the sensor and for the CIU in case we don't find the common format to what user request*/
	rval = get_ciu_sensor_default_fmt(vdev, &ciu_sensor_default_pix);
	if (rval) {
		return rval;
	}

	/*Get commot format for use sensor and the CIU*/
	rval = get_ciu_sensor_user_fmt(vdev,wanted_pix);
	if (rval) {
		PDEBUG("Fail to find common format for CIU sensor and user hence use default common between CIU and the sensor\n");
		wanted_pix->pixelformat = ciu_sensor_default_pix.pixelformat;
		rval = 0;
	}
	CIU_END();
	return rval;
}

/*
static int get_stepwise_ciu_sensor_user(struct v4l2_frmsizeenum *ciu_frms,
								   struct v4l2_frmsizeenum *sensor_frms,
									struct v4l2_pix_format *best_pix,
								   struct v4l2_pix_format *want_pix)
{
	int res = 0;
	PDEBUG("\n");
	if (!ciu_frms || !sensor_frms || !want_pix || !best_pix)
	{
		printk("Invalid parameters\n");
		return -1;
	}
	

	res = get_stepwise_ciu_sensor_user_dim(	&ciu_frms->stepwise.min_width ,
											 &ciu_frms->stepwise.max_width ,
											 &ciu_frms->stepwise.step_width,
											 &sensor_frms->stepwise.min_width ,
											 &sensor_frms->stepwise.max_width ,
											 &sensor_frms->stepwise.step_width,
											 &want_pix->width,
											 &best_pix->width);
	if (res)
		return res;

	res = get_stepwise_ciu_sensor_user_dim(&ciu_frms->stepwise.min_height ,
										 &ciu_frms->stepwise.max_height ,
										 &ciu_frms->stepwise.step_height,
										 &sensor_frms->stepwise.min_height ,
										 &sensor_frms->stepwise.max_height ,
										 &sensor_frms->stepwise.step_height,
										 &want_pix->height,
										 &best_pix->height);
	if (res)
		return res;

	return res;
}
*/


static int fill_out_image_info(struct v4l2_pix_format *pix , unsigned int height , unsigned int width){

	int res = 0;


	if (!pix || 
	(pix->pixelformat != V4L2_PIX_FMT_NV12 && pix->pixelformat != V4L2_PIX_FMT_NV16 && pix->pixelformat != V4L2_PIX_FMT_NV21 && pix->pixelformat != 	V4L2_PIX_FMT_NV61)
	 || !height || !width )
		return -EINVAL;

	pix->height = height;
	pix->width = width;
	pix->bytesperline = bytes_per_line(width , pix->pixelformat);
    pix->sizeimage = image_size(width,height,pix->pixelformat);
	pix->field = V4L2_FIELD_NONE;
	pix->priv = 0;
	pix->colorspace = V4L2_COLORSPACE_JPEG;

	return res;
}

unsigned int to_ciu_scaler_mode(unsigned int factor)
{
	if ( (factor != 1) && (factor != 2) )
		return CIU_FULL;

	if (factor == 1) return CIU_FULL;
	if (factor == 2)return CIU_HALF;
	

	return CIU_FULL;
}

static int get_discrate_ciu_scaler_sensor_user(struct v4l2_pix_format *wanted_pix,
												struct v4l2_pix_format *best_pix_sensor,
												struct v4l2_pix_format* best_pix_ciu,
												unsigned int* scaler_mode,
												struct v4l2_rect* crect)
{

	int res = 0;
	unsigned int i = 0;
	unsigned int l_height=0 , height=0 , l_width=0 , width=0, l_factor=1;
	unsigned int sq_size = 0xffffffff;

	if (!wanted_pix || !best_pix_sensor || !best_pix_ciu || !scaler_mode || !crect) 
		return -EINVAL;

	/*Find at least one full or scaled resolution which is bigger then the wanted pixel*/
	for (i=0 ; i< CIU_DECIMATOR_FACTOR_SIZE  ; i++)
	{
		width = best_pix_sensor->width / ciu_decimator_factor[i] ;
		height = best_pix_sensor->height / ciu_decimator_factor[i];

		if (width &&
			width >= wanted_pix->width &&

			height &&
			height >= wanted_pix->height )

		{
			if ((height * width) < sq_size)
			{
				sq_size = height * width;
				l_height = height;
				l_width = width;
				l_factor = ciu_decimator_factor[i];

			}
		}
	}

	if (l_height && l_width) {

		*scaler_mode = to_ciu_scaler_mode(l_factor);

		if (*scaler_mode == CIU_FULL) {

			fill_out_image_info(best_pix_ciu, wanted_pix->height & ~(0x1)  , wanted_pix->width & ~(0x3));
			crect->left = ((l_width - best_pix_ciu->width)/2) & ~(0x3);
			crect->top = ((l_height - wanted_pix->height)/2)  & ~(0x1);
			crect->width = 0;
			crect->height = 0;

			fill_out_image_info(wanted_pix, best_pix_ciu->height  , best_pix_ciu->width);
		}
		if (*scaler_mode == CIU_HALF) {
			fill_out_image_info(best_pix_ciu, (wanted_pix->height * 2) & ~(0x1)  , (wanted_pix->width * 2) & ~(0x3));
			crect->left = ((l_width*2 - best_pix_ciu->width)/2) & ~(0x3);
			crect->top = ((l_height*2 - best_pix_ciu->height)/2) & ~(0x1);
			crect->width = 0;
			crect->height = 0;
			
			fill_out_image_info(wanted_pix, best_pix_ciu->height / 2  , best_pix_ciu->width / 2);
		}

		crect->width = ( wanted_pix->width / ((*scaler_mode == CIU_HALF) ? 2 : 4) ) & ~(0xf);
		crect->height = (wanted_pix->height / ((*scaler_mode == CIU_HALF) ? 2 : 4) ) & ~(0xf);
	}
	
	if ( (*scaler_mode == CIU_HALF) && 
		(wanted_pix->pixelformat == V4L2_PIX_FMT_NV16 || wanted_pix->pixelformat == V4L2_PIX_FMT_NV61)) {
		printk("CIU dosen't support subsumpling 1/2 and 4:2:2 format\n");
		res = -EINVAL;
	}

	return res;
}

void print_output_pixel(const char* name, struct v4l2_pix_format* pix){

		if (!name || !pix) return;

		PDEBUG("-------------------\n");
		PDEBUG("%s\n",name);
		PDEBUG("-------------------\n");
		PDEBUG("width = %d\n",pix->width);
		PDEBUG("height = %d\n",pix->height);
		PDEBUG("bytesperline = %d\n",pix->bytesperline);
		PDEBUG("sizeimage = %d\n",pix->sizeimage);
		PDEBUG("pixelformat = %s\n",format_to_string(pix->pixelformat));
		PDEBUG("-------------------\n");

}

int get_discrate_ciu_sensor_user(struct dmw96cam_videodev *vdev,
								struct v4l2_frmsizeenum *ciu_frms,
								struct v4l2_pix_format *wanted_pix,
								struct v4l2_pix_format *best_pix_sensor,
								struct v4l2_pix_format* best_pix_ciu,
								unsigned int* scaler_mode,
								struct v4l2_rect* crect)
{
	struct v4l2_frmsizeenum sensor_frms;
	unsigned int sq_size = 0xffffffff;
	int i = 0;
	int res = 0;
	unsigned int height = 0;
	unsigned int width = 0;
	
	CLEAR(sensor_frms);
	CIU_START();

	for (i=0 ;; i++)
	{
		sensor_frms.index = i;
        sensor_frms.pixel_format = wanted_pix->pixelformat;

		res = vidioc_int_enum_framesizes(vdev->vdev_sensor, &sensor_frms);
		if (res)
			break;
		/*Go over all the sensor resolutions and get the first resolution for the sensor which is bigger then what user asks for*/
		if (sensor_frms.discrete.width >= ciu_frms->stepwise.min_width &&
			sensor_frms.discrete.width <= ciu_frms->stepwise.max_width &&
			sensor_frms.discrete.width >= wanted_pix->width &&
			
			sensor_frms.discrete.height >= ciu_frms->stepwise.min_height &&
			sensor_frms.discrete.height <= ciu_frms->stepwise.max_height &&
			sensor_frms.discrete.height >= wanted_pix->height )

		{
			if ((sensor_frms.discrete.height * sensor_frms.discrete.width) < sq_size)
			{
				sq_size = sensor_frms.discrete.height * sensor_frms.discrete.width;
				height = sensor_frms.discrete.height;
				width = sensor_frms.discrete.width;

			}
		}
	}
		
	if (height && width) 
	{
		fill_out_image_info(best_pix_sensor, height , width);
		res = get_discrate_ciu_scaler_sensor_user(wanted_pix, best_pix_sensor ,best_pix_ciu , scaler_mode, crect);
	
		print_output_pixel("best_pix_sensor", best_pix_sensor);
		print_output_pixel("wanted_pix", wanted_pix);
		print_output_pixel("best_pix_ciu", best_pix_ciu);
		PDEBUG("ciu scale = %d\n",*scaler_mode);
		PDEBUG("crect->left = %d\n", crect->left);
		PDEBUG("crect->top = %d\n", crect->top);
		PDEBUG("crect->width = %d\n", crect->width);
		PDEBUG("crect->height = %d\n", crect->height);

	}else
		PDEBUG("user size is not supported by both ciu and senseor\n");

	CIU_END();
	return res;
}



static int get_common_framesizes(struct dmw96cam_videodev *vdev,
								 struct v4l2_pix_format* wanted_pix,
								 struct v4l2_pix_format *best_pix_sensor,
								 struct v4l2_pix_format* best_pix_ciu,
								 unsigned int* scaler_mode,
								 struct v4l2_rect* crect)
{
	int rval = 0;
	struct v4l2_frmsizeenum frms_sensor;
	struct v4l2_frmsizeenum frms_ciu;
	struct v4l2_pix_format max_pix;
	struct v4l2_pix_format min_pix;
	CLEAR(frms_ciu);
	CLEAR(frms_sensor);
	memcpy(&max_pix, wanted_pix, sizeof(max_pix));
	memcpy(&min_pix, wanted_pix, sizeof(min_pix));
	CIU_START();

	frms_ciu.pixel_format = wanted_pix->pixelformat;
	frms_sensor.pixel_format = wanted_pix->pixelformat;
	
	rval = get_ciu_sensor_common_framesizes(vdev , &max_pix, &min_pix);
	if (rval)
		return rval;

	/* After this wanted_pix will hold the min,max common for all user sensor and CIU */
	if (wanted_pix->height > max_pix.height || wanted_pix->width > max_pix.width) {
		wanted_pix->height = max_pix.height;
		wanted_pix->width = max_pix.width;
	}
	if (wanted_pix->height < min_pix.height || wanted_pix->width < min_pix.width) {
		wanted_pix->height = min_pix.height;
		wanted_pix->width = min_pix.width;
	}

	/* At this step we don't know if the sensor size support is discrete or stepwise */
	rval = vidioc_int_enum_framesizes(vdev->vdev_sensor, &frms_sensor);
	if (rval){
		PDEBUG("Unable to query frame format which is supported by camera sensor\n");	
		return rval;
	}

	/* We assume that the CIU is stepwise */
	rval = ciu_enum_framesize(&frms_ciu);
	if (rval) {
		PDEBUG("unable to query frame sizes which is supported by CIU\n");	
		return rval;
	}

	if (frms_sensor.type == V4L2_FRMSIZE_TYPE_DISCRETE)
	{
		rval = get_discrate_ciu_sensor_user(vdev, &frms_ciu, wanted_pix, best_pix_sensor, best_pix_ciu, scaler_mode, crect);
		if (rval) {
            return rval;
		}

	} 
	else if (frms_sensor.type == V4L2_FRMSIZE_TYPE_STEPWISE)
	{

		/*TODO implement the case below*/
		printk("CIU dosen't support stepwise sensor\n");
		return -1;
		/*
		rval = get_stepwise_ciu_sensor_user(&frms_ciu,&frms_sensor, &best_pix ,wanted_pix);
		if (rval) {
            return rval;
		}
		*/

	}
	else{
		PDEBUG("unable to handle the frame size type which is supported by camera sensor\n");	
		return -1;
	}

	CIU_END();

	return 0;
	
}

static unsigned int get_common_frameintervals(struct dmw96cam_videodev *vdev, struct v4l2_pix_format* pix, struct v4l2_fract *wanted_ival)
{
	unsigned int ival_index = 0;
	struct v4l2_fract ival_tmp;
	struct v4l2_frmivalenum frmi;
	int rval = 0;

	if (!vdev || !pix || !wanted_ival)
		return -1;
	CIU_START();
	ival_tmp.denominator = 3;
	ival_tmp.numerator = 1;

	frmi.pixel_format = pix->pixelformat;
	frmi.width = pix->width;
	frmi.height = pix->height;

	PDEBUG("ival_tmp.denominator=%d ival_tmp.numerator=%d \n",ival_tmp.denominator, ival_tmp.numerator);

    /* find the oprimized FPS*/
	for (ival_index = 0; ; ival_index++) {
	
		frmi.index = ival_index;

		PDEBUG("frmi.index=%d  frmi.width=%d frmi.height=%d pix_frmi.pixel_format=%s\n",
		    frmi.index ,frmi.width , frmi.height , format_to_string(frmi.pixel_format));

		rval = vidioc_int_enum_frameintervals(vdev->vdev_sensor, &frmi);
		if (rval) {
			PDEBUG("Out of bound or not supported frameintreval\n");
			rval = 0;
			break;
		}

		PDEBUG("frmi.index=%d frmi.discrete.denominator=%d frmi.discrete.numerator=%d\n",
				frmi.index , frmi.discrete.denominator , frmi.discrete.numerator);

#define FPS_DIFF(ival1, ival2) ( (((ival1).denominator / (ival1).numerator) >= ((ival2).denominator / (ival2).numerator)) ? 1 : 0)

		if (FPS_DIFF(*wanted_ival, frmi.discrete)){
				ival_tmp = frmi.discrete;
				PDEBUG("ival_tmp.denominator=%d ival_tmp.numerator=%d\n",ival_tmp.denominator , ival_tmp.numerator);
		}

	}

	*wanted_ival = ival_tmp;
	PDEBUG("wanted_ival->denominator = %d  wanted_ival->numerator = %d \n", wanted_ival->denominator , wanted_ival->numerator  );

	CIU_END();
	return 0;
}

static int try_pix_parm(struct dmw96cam_videodev *vdev,
			struct v4l2_pix_format *wanted_pix,
			struct v4l2_fract *wanted_ival,
			struct v4l2_pix_format *best_pix_sensor,
			struct v4l2_pix_format *best_pix_ciu,
			unsigned int* scaler_mode,
			struct v4l2_rect* crect)
{
    int rval = 0;
	CIU_START();
	
	if (!vdev || !wanted_pix || !wanted_ival || !best_pix_sensor || !best_pix_ciu || !scaler_mode || !crect)
		return -EINVAL;

	if (wanted_ival->numerator == 0 || wanted_ival->denominator == 0){
			wanted_ival->denominator = 15;
			wanted_ival->numerator = 2;
	}
	
	/* This default values are under the assumption that there is config that is commont to both CIU and the sensor*/
	rval = get_common_fmt(vdev, wanted_pix);
	if (rval)
		return rval;
	else 
		best_pix_ciu->pixelformat = best_pix_sensor->pixelformat = wanted_pix->pixelformat;
	
    /* At this point wanted_pix should hold the pixel format which common to CIU, sensor
	and might be also common to what userspace requests. this because userspace might request pixel format which is not supported 
	both by CIU and the sensor */
	rval = get_common_framesizes(vdev,wanted_pix,best_pix_sensor, best_pix_ciu , scaler_mode, crect);
	if (rval)
		return rval;

	rval = get_common_frameintervals(vdev,best_pix_sensor,wanted_ival);
	if (rval)
		return rval;

	CIU_END();
	return rval;
}


static int s_pix_parm(struct dmw96cam_videodev *vdev,
					  struct v4l2_pix_format *wanted_pix,
					  struct v4l2_fract *wanted_ival)
{
	struct v4l2_streamparm a;
	struct v4l2_rect crect;
	struct v4l2_format fmt;
	struct v4l2_pix_format best_pix_ciu;
	struct v4l2_pix_format best_pix_sensor;
	unsigned int scaler_mode = 0;
	int rval = 0;

	CLEAR(a);
	CLEAR(crect);
	CLEAR(fmt);
	CLEAR(best_pix_ciu);
	CLEAR(best_pix_sensor);

	if (!vdev || !wanted_pix || !wanted_ival)
		return -1;

	CIU_START();
	
	/*Suggest the best common pixel format and size for the sensor CIU and user*/
    rval = try_pix_parm(vdev, wanted_pix, wanted_ival, &best_pix_sensor , &best_pix_ciu,  &scaler_mode, &crect);
	if (rval)
		return rval;
	
	/*Set the CIU HW and set the buffers size according to the best  pixel*/
	rval = ciu_s_fmt_cap(&best_pix_ciu, scaler_mode, &crect);
	if (rval)
		return rval;
	
	/*Set the sensor*/
    fmt.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	fmt.fmt.pix = best_pix_sensor;
	rval = vidioc_int_s_fmt_cap(vdev->vdev_sensor, &fmt);
	if (rval)
		return rval;
	
    a.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
	a.parm.capture.timeperframe = *wanted_ival;
	rval = vidioc_int_s_parm(vdev->vdev_sensor, &a);
	if (rval)
		return rval;
	
	CIU_END();
	return rval;
}

/**
 * vidioc_s_fmt_vid_cap - V4L2 set format capabilities IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @f: ptr to standard V4L2 format structure
 *
 * Attempts to set input format with the sensor driver (first) and then the
 * CIU.  Returns the return code from vidioc_g_fmt_vid_cap().
 */
static int vidioc_s_fmt_vid_cap(struct file *file, void *fh,
				struct v4l2_format *f)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	struct v4l2_fract timeperframe;
	int rval = 0;
	
	CIU_START();

	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	mutex_lock(&vdev->mutex);
	if (vdev->streaming) {
		rval = -EBUSY;
		goto out;
	}

	timeperframe = vdev->timeperframe;

	rval = s_pix_parm(vdev, &f->fmt.pix, &timeperframe);
 	if (!rval)
		vdev->pix = f->fmt.pix ;

	
out:
	mutex_unlock(&vdev->mutex);
	CIU_END();
	return rval;
}

/**
 * vidioc_try_fmt_vid_cap - V4L2 try format capabilities IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @f: ptr to standard V4L2 format structure
 *
 * Checks if the given format is supported by the sensor driver and
 * by the CIU.
 */
static int vidioc_try_fmt_vid_cap(struct file *file, void *fh,
				  struct v4l2_format *f)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	struct v4l2_pix_format best_pix;
	struct v4l2_pix_format best_pix_ciu;
	struct v4l2_fract timeperframe;
	int rval;
	unsigned int stab;
	struct v4l2_rect crect;

	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;
	CIU_START();
	mutex_lock(&vdev->mutex);

	timeperframe = vdev->timeperframe;

	rval = try_pix_parm(vdev, &f->fmt.pix, &timeperframe, &best_pix_ciu, &best_pix, &stab, &crect);
	
	mutex_unlock(&vdev->mutex);
	CIU_END();
	return rval;
}

/**
 * vidioc_reqbufs - V4L2 request buffers IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @b: ptr to standard V4L2 request buffers structure
 *
 * Attempts to get a buffer from the buffer queue associated with the
 * fh through the video buffer library API.
 */
static int vidioc_reqbufs(struct file *file, void *fh,
			  struct v4l2_requestbuffers *b)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval;
	CIU_START();
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	mutex_lock(&vdev->mutex);
	if (vdev->streaming) {
		mutex_unlock(&vdev->mutex);
		return -EBUSY;
	}

	rval = videobuf_reqbufs(&ofh->vbq, b);

	mutex_unlock(&vdev->mutex);

	/*
	 * Either videobuf_reqbufs failed or the buffers are not
	 * memory-mapped (which would need special attention).
	 */
	if (rval < 0 || b->memory != V4L2_MEMORY_MMAP)
		goto out;

out:
	CIU_END();
	return rval;
}

/**
 * vidioc_querybuf - V4L2 query buffer IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @b: ptr to standard V4L2 buffer structure
 *
 * Attempts to fill in the v4l2_buffer structure for the buffer queue
 * associated with the fh through the video buffer library API.
 */
static int vidioc_querybuf(struct file *file, void *fh, struct v4l2_buffer *b)
{
	struct dmw96cam_fh *ofh = fh;
	int rval=0;
	CIU_START();
	rval = videobuf_querybuf(&ofh->vbq, b);
	if (rval)
		goto out;
out:
	CIU_END();
	return rval; 
}

/**
 * vidioc_qbuf - V4L2 queue buffer IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @b: ptr to standard V4L2 buffer structure
 *
 * Attempts to queue the v4l2_buffer on the buffer queue
 * associated with the fh through the video buffer library API.
 */
static int vidioc_qbuf(struct file *file, void *fh, struct v4l2_buffer *b)
{
	struct dmw96cam_fh *ofh = fh;
	int rval=0;
	CIU_START();
	rval = videobuf_qbuf(&ofh->vbq, b);
	if (rval)
		goto out;
out:
	CIU_END();
	return rval;
}

/**
 * vidioc_dqbuf - V4L2 dequeue buffer IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @b: ptr to standard V4L2 buffer structure
 *
 * Attempts to dequeue the v4l2_buffer from the buffer queue
 * associated with the fh through the video buffer library API.  If the
 * buffer is a user space buffer, then this function will also requeue it,
 * as user does not expect to do this.
 */
static int vidioc_dqbuf(struct file *file, void *fh, struct v4l2_buffer *b)
{
	struct dmw96cam_fh *ofh = fh;
	int rval=0;
	CIU_START();

	rval = videobuf_dqbuf(&ofh->vbq, b, file->f_flags & O_NONBLOCK);

	CIU_END();
	return rval;
}

/**
 * vidioc_streamon - V4L2 streamon IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @i: V4L2 buffer type
 *
 * Attempts to start streaming by enabling the sensor interface and turning
 * on video buffer streaming through the video buffer library API.  Upon
 * success the function returns 0, otherwise an error code is returned.
 */
static int vidioc_streamon(struct file *file, void *fh, enum v4l2_buf_type i)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval = 0;
	CIU_START();
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	mutex_lock(&vdev->mutex);
	if (vdev->streaming) {
		rval = -EBUSY;
		goto out;
	}

	del_timer_sync(&vdev->poweroff_timer);
	
	rval = dmw96cam_slave_power_set(vdev, V4L2_POWER_ON,
					   DMW96CAM_SLAVE_POWER_SENSOR_LENS);
	if (rval) {
		dev_dbg(&vdev->vfd->dev, "dmw96cam_slave_power_set failed\n");
		goto out;
	}

	rval = videobuf_streamon(&ofh->vbq);
	if (rval) {
		dmw96cam_slave_power_set(
			vdev, V4L2_POWER_OFF,
			DMW96CAM_SLAVE_POWER_SENSOR_LENS);

		dmw96ciu_enb_set(0);
	} else {
		vdev->streaming = file;
		dmw96ciu_enb_set(1);
	}
	CIU_END();
out:
	mutex_unlock(&vdev->mutex);

	return rval;
}

/**
 * vidioc_streamoff - V4L2 streamoff IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @i: V4L2 buffer type
 *
 * Attempts to stop streaming by flushing all scheduled work, waiting on
 * any queued buffers to complete and then stopping the CIU and turning
 * off video buffer streaming through the video buffer library API.  Upon
 * success the function returns 0, otherwise an error code is returned.
 */
static int vidioc_streamoff(struct file *file, void *fh, enum v4l2_buf_type i)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	struct videobuf_queue *q = &ofh->vbq;
	int rval;
	CIU_START();
	mutex_lock(&vdev->mutex);

	rval = videobuf_streamoff(q);
	if ( (vdev->streaming == file) && !rval) {
		vdev->streaming = NULL;

		dmw96ciu_enb_set(0);

		dmw96cam_slave_power_suggest(vdev, V4L2_POWER_STANDBY,
						DMW96CAM_SLAVE_POWER_SENSOR_LENS);

		/*Reset the CIU in case of chaning resolution*/
		ciu_disable_block();
		mdelay(10);
		ciu_enable_block();
		ciu_buf_init();
	}
	CIU_END();
	mutex_unlock(&vdev->mutex);

	return rval;
}

/**
 * vidioc_enum_input - V4L2 enumerate input IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @inp: V4L2 input type information structure
 *
 * Fills in v4l2_input structure.  Returns 0.
 */
static int vidioc_enum_input(struct file *file, void *fh,
			     struct v4l2_input *inp)
{
	if (inp->index > 0)
		return -EINVAL;
	CIU_START();
	strlcpy(inp->name, "camera", sizeof(inp->name));
	inp->type = V4L2_INPUT_TYPE_CAMERA;
	CIU_END();
	return 0;
}

/**
 * vidioc_g_input - V4L2 get input IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @i: address to hold index of input supported
 *
 * Sets index to 0.
 */
static int vidioc_g_input(struct file *file, void *fh, unsigned int *i)
{
	*i = 0;

	return 0;
}

/**
 * vidioc_s_input - V4L2 set input IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @i: index of input selected
 *
 * 0 is only index supported.
 */
static int vidioc_s_input(struct file *file, void *fh, unsigned int i)
{
	if (i > 0)
		return -EINVAL;

	return 0;
}

/**
 * vidioc_queryctrl - V4L2 query control IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @a: standard V4L2 query control ioctl structure
 *
 * If the requested control is supported, returns the control information
 * in the v4l2_queryctrl structure.  Otherwise, returns -EINVAL if the
 * control is not supported.  If the sensor being used is a "smart sensor",
 * this request is passed to the sensor driver, otherwise the CIU is
 * queried and if it does not support the requested control, the request
 * is forwarded to the "raw" sensor driver to see if it supports it.
 */
static int vidioc_queryctrl(struct file *file, void *fh,
			    struct v4l2_queryctrl *a)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	struct v4l2_queryctrl a_tmp;
	int best_slave = -1;
	u32 best_ctrl = (u32)-1;
	int i;
	int res = 0;
	CIU_START();
	res = vidioc_int_queryctrl(vdev->vdev_sensor, a);
	if (res != -EINVAL)
		return res;
	

	/* No next flags: try othre slaves directly. */
	if (!(a->id & V4L2_CTRL_FLAG_NEXT_CTRL)) {
		for (i = 0; i <= DMW96CAM_SLAVE_FLASH; i++) {
			if (!vidioc_int_queryctrl(vdev->slave[i], a))
				return 0;
		}
		return -EINVAL;
	}

	/* Find slave with smallest next control id. */
	for (i = 0; i <= DMW96CAM_SLAVE_FLASH; i++) {
		a_tmp = *a;

		if (vidioc_int_queryctrl(vdev->slave[i], &a_tmp))
			continue;

		if (a_tmp.id < best_ctrl) {
			best_slave = i;
			best_ctrl = a_tmp.id;
		}
	}

	a_tmp = *a;
    
	if (best_slave == -1)
		return -EINVAL;

	a->id = best_ctrl;
	res = vidioc_int_queryctrl(vdev->slave[best_slave], a);
	CIU_END();
	return res;
}


static int vidioc_g_ctrl(struct file *file, void *fh,
			      struct v4l2_control *a)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval = 0;
	int i;
	CIU_START();
	mutex_lock(&vdev->mutex);
    
    rval = vidioc_int_g_ctrl(vdev->vdev_sensor, a);
    if (rval == -EINVAL) {
		/*Set other slaves*/	
		for (i = 0; i <= DMW96CAM_SLAVE_FLASH; i++) {
				rval = vidioc_int_g_ctrl(vdev->slave[i], a);
				if (!rval)
					break;
		}
		
	}
	CIU_END();
    mutex_unlock(&vdev->mutex);

	return rval;
}

static int vidioc_s_ctrl(struct file *file, void *fh,
			      struct v4l2_control *a)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval = 0;
	int i;
	CIU_START();
	mutex_lock(&vdev->mutex);
    
    rval = vidioc_int_s_ctrl(vdev->vdev_sensor, a);
	if (rval == -EINVAL) {
			
		for (i = 0; i <= DMW96CAM_SLAVE_FLASH; i++) {
				rval = vidioc_int_s_ctrl(vdev->slave[i], a);
				if (!rval)
					break;
		}
		
	}
	CIU_END();
    mutex_unlock(&vdev->mutex);

	return rval;

}

static int vidioc_g_ext_ctrls(struct file *file, void *fh,
			      struct v4l2_ext_controls *a)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int ctrl_idx, rval = 0;
	int i;
	CIU_START();
	mutex_lock(&vdev->mutex);
    
	for (ctrl_idx = 0; ctrl_idx < a->count; ctrl_idx++) {
		struct v4l2_control ctrl;
		memset (&ctrl , 0 , sizeof(struct v4l2_control));

		ctrl.id = a->controls[ctrl_idx].id;

		if (vidioc_int_s_ctrl(vdev->vdev_sensor, &ctrl) == -EINVAL) {

			for (i = 0; i <= DMW96CAM_SLAVE_FLASH; i++) {
				rval = vidioc_int_g_ctrl(vdev->slave[i], &ctrl);
				if (!rval)
					break;
			}
		}
        
		if (rval) {
			a->error_idx = ctrl_idx;
			break;
		}

		a->controls[ctrl_idx].value = ctrl.value;
	}
	CIU_END();
	mutex_unlock(&vdev->mutex);

	return rval;
}

static int vidioc_s_ext_ctrls(struct file *file, void *fh,
			      struct v4l2_ext_controls *a)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int ctrl_idx, rval = 0;
	int i;
	CIU_START();
	mutex_lock(&vdev->mutex);

	for (ctrl_idx = 0; ctrl_idx < a->count; ctrl_idx++) {
		struct v4l2_control ctrl;

		ctrl.id = a->controls[ctrl_idx].id;
		ctrl.value = a->controls[ctrl_idx].value;

		if (vidioc_int_s_ctrl(vdev->vdev_sensor, &ctrl) == -EINVAL) {
			for (i = 0; i <= DMW96CAM_SLAVE_FLASH; i++) {
				rval = vidioc_int_s_ctrl(vdev->slave[i], &ctrl);
				if (!rval)
					break;
			}
		}
		
		if (rval) {
			a->error_idx = ctrl_idx;
			break;
		}

		a->controls[ctrl_idx].value = ctrl.value;
	}
	CIU_END();
	mutex_unlock(&vdev->mutex);

	return rval;
}

/**
 * vidioc_g_parm - V4L2 get parameters IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @a: standard V4L2 stream parameters structure
 *
 * If request is for video capture buffer type, handles request by
 * forwarding to sensor driver.
 */
static int vidioc_g_parm(struct file *file, void *fh, struct v4l2_streamparm *a)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval;
	CIU_START();
	if (a->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
		return -EINVAL;

	mutex_lock(&vdev->mutex);
	rval = vidioc_int_g_parm(vdev->vdev_sensor, a);
	mutex_unlock(&vdev->mutex);
	CIU_END();
	return rval;
}

/**
 * vidioc_s_parm - V4L2 set parameters IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @a: standard V4L2 stream parameters structure
 *
 * If request is for video capture buffer type, handles request by
 * first getting current stream parameters from sensor, then forwarding
 * request to set new parameters to sensor driver.  It then attempts to
 * enable the sensor interface with the new parameters.  If this fails, it
 * reverts back to the previous parameters.
 */
static int vidioc_s_parm(struct file *file, void *fh, struct v4l2_streamparm *a)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval;
	CIU_START();
	if (a->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
		return -EINVAL;

	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	mutex_lock(&vdev->mutex);
	if (vdev->streaming) {
		rval = -EBUSY;
		goto out;
	}

	vdev->timeperframe = a->parm.capture.timeperframe;

	rval = s_pix_parm(vdev, &vdev->pix, &vdev->timeperframe);
out:
	mutex_unlock(&vdev->mutex);
	CIU_END();
	return rval;
}

/**
 * vidioc_cropcap - V4L2 crop capture IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @a: standard V4L2 crop capture structure
 *
 * If using a "smart" sensor, just forwards request to the sensor driver,
 * otherwise fills in the v4l2_cropcap values locally.
 */
static int vidioc_cropcap(struct file *file, void *fh, struct v4l2_cropcap *a)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	//struct v4l2_cropcap *cropcap = a;
	int rval  = 0;
	CIU_START();
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	mutex_lock(&vdev->mutex);

	/*That case the sensor will return its last configurable pixel size*/
	rval = vidioc_int_cropcap(vdev->vdev_sensor, a);

	/*That case the the sensor is not supporting the crop hence return the crop cap of the CIU*/
	if (rval == -EINVAL){
		PDEBUG("Not Able to get crop cap form sensor asking the CIU\n");
		rval = ciu_g_cropcap(a);	
    }

	PDEBUG("a->bounds.top = %d\n a->bounds.left = %d\n a->bounds.width = %d\n a->bounds.height = %d\n"
			"a->defrect.top = %d\n a->defrect.left = %d\n a->defrect.width = %d\n a->defrect.height = %d \n "
			"a->pixelaspect.numerator = %d\n a->pixelaspect.denominator = %d\n",
			a->bounds.top, a->bounds.left, a->bounds.width, a->bounds.height, 
			a->defrect.top, a->defrect.left, a->defrect.width, a->defrect.height, 
			a->pixelaspect.numerator,a->pixelaspect.denominator);

	mutex_unlock(&vdev->mutex);
	CIU_END();
	return rval;
}

/**
 * vidioc_g_crop - V4L2 get capture crop IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @a: standard V4L2 crop structure
 *
 * If using a "smart" sensor, just forwards request to the sensor driver,
 * otherwise calls the ciu functions to fill in current crop 
 * values. 
 */
static int vidioc_g_crop(struct file *file, void *fh, struct v4l2_crop *a)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval = 0;
	CIU_START();
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	mutex_lock(&vdev->mutex);

    rval = vidioc_int_g_crop(vdev->vdev_sensor, a);
	if (rval == -EINVAL) {// the cropping is not suported
		PDEBUG("Not Able to get crop form sensor asking the CIU\n");
		rval = ciu_g_crop(a);
	}
    
	mutex_unlock(&vdev->mutex);
	CIU_END();
	return rval;
}

/**
 * vidioc_s_crop - V4L2 set capture crop IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @a: standard V4L2 crop structure
 *
 * If using a "smart" sensor, just forwards request to the sensor driver,
 * otherwise calls the ciu functions to set the current crop 
 * values. 
 */
static int vidioc_s_crop(struct file *file, void *fh, struct v4l2_crop *a)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval = 0;
	CIU_START();
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	mutex_lock(&vdev->mutex);

    rval = vidioc_int_s_crop(vdev->vdev_sensor, a);
	if (rval == -EINVAL) {// the cropping is not suported
		PDEBUG("Not Able to set crop form sensor asking the CIU\n");
        rval = ciu_s_crop(a, &vdev->pix);
		if (rval)
			return rval;
	}
	/*Both sensor on CIU cropping might change the buffers size (the buffers size depend on vdev->pix.height and vdev->pix.width)*/
	if (!rval) {
		vdev->pix.height = a->c.height;
		vdev->pix.width = a->c.width;
	}
    
	mutex_unlock(&vdev->mutex);
	CIU_END();
	return rval;
}

static int vidioc_enum_framesizes(struct file *file, void *fh,
				  struct v4l2_frmsizeenum *frms)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval = 0;
	CIU_START();
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

    mutex_lock(&vdev->mutex);


	rval = vidioc_int_enum_framesizes(vdev->vdev_sensor, frms);

	/*TODO Dudi have to complete the cross between the sensor and the CIU

	// At this step we don't know if the sensor size support is discrete or stepwise 
	rval = vidioc_int_enum_framesizes(vdev->vdev_sensor, &frms_sensor);
	if (rval){
		PDEBUG("Unable to query frame format which is supported by camera sensor\n");	
		return rval;
	}

	
	// We assume that the CIU is stepwise 
	rval = ciu_enum_framesize(&frms_ciu);
	if (rval) {
		PDEBUG("unable to query frame sizes which is supported by CIU\n");	
		return rval;
	}

	if (frms_sensor.type == V4L2_FRMSIZE_TYPE_DISCRETE)
	{
		rval = get_discrate_ciu_sensor_user_dim(vdev,&frms_ciu,best_pix,wanted_pix);
		if (rval) {
            return rval;
		}

	} 
	else if (frms_sensor.type == V4L2_FRMSIZE_TYPE_STEPWISE)
	{
		rval = get_stepwise_ciu_sensor_user(&frms_ciu,&frms_sensor, best_pix ,wanted_pix);
		if (rval) {
            return rval;
		}

	}
	else{
		PDEBUG("unable to handle the frame size type which is supported by camera sensor\n");	
		return -1;
	}
	*/

	
	mutex_unlock(&vdev->mutex);

	CIU_END();
	return rval;
}

static int vidioc_enum_frameintervals(struct file *file, void *fh,
				      struct v4l2_frmivalenum *frmi)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval = 0;
	CIU_START();
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	mutex_lock(&vdev->mutex);
	rval = vidioc_int_enum_frameintervals(vdev->vdev_sensor, frmi);
	mutex_unlock(&vdev->mutex);
	CIU_END();
	return rval;
}

#ifdef CONFIG_VIDEO_ADV_DEBUG
static int vidioc_s_sensor_reg(struct file *file, void *fh, void *arg)
{
	struct dmw96cam_fh *ofh;
	struct dmw96cam_videodev *vdev;
	int rval;
	CIU_START();
	if (!file || !fh || !arg || !((struct dmw96cam_fh *)fh)->vdev )
		return -EINVAL;

	ofh = fh;
	vdev = ofh->vdev;
	
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;
	
	mutex_lock(&vdev->mutex);
	rval = vidioc_int_s_reg(vdev->vdev_sensor,arg);
	mutex_unlock(&vdev->mutex);
	CIU_END();
	return 0;
}

static int vidioc_g_sensor_reg(struct file *file, void *fh, void *arg)
{
	struct dmw96cam_fh *ofh;
	struct dmw96cam_videodev *vdev;
	int rval;
	CIU_START();
	if (!file || !fh || !arg || !((struct dmw96cam_fh *)fh)->vdev )
		return -EINVAL;

	ofh = fh;
	vdev = ofh->vdev;
	
	if (vdev->vdev_sensor == v4l2_int_device_dummy())
		return -EINVAL;

	mutex_lock(&vdev->mutex);
	rval = vidioc_int_g_reg(vdev->vdev_sensor,arg);
	mutex_unlock(&vdev->mutex);
	CIU_END();
	return 0;
}
#endif
static int vidioc_g_physical_request(void *fh, struct v4l2_buffer *buf)
{
	struct dmw96cam_fh *ofh = (struct dmw96cam_fh*)fh;
    struct videobuf_buffer *vb;
	CIU_START();
	if (!fh || !buf || !ofh )
		return -EINVAL;
		
	if (buf->index >= VIDEO_MAX_FRAME) return -1;

    if ((vb = ofh->vbq.bufs[buf->index]) == NULL) return -1;

	buf->reserved = (unsigned int)videobuf_to_dma_contig(vb);
	if (!buf->reserved) return -1;
	CIU_END();
	return 0;
}


static int vidioc_g_chip_ident(struct file *file, void *fh,
		struct v4l2_dbg_chip_ident *chip)
{
	//struct dmw96cam_fh *ofh = (struct dmw96cam_fh*)fh;

	return 0;
}


#ifdef CONFIG_VIDEO_ADV_DEBUG
static int vidioc_g_register (struct file *file, void *fh,
			      struct v4l2_dbg_register *reg)
{
	
	//struct dmw96cam_fh *ofh = (struct dmw96cam_fh*)fh;
	
	//if (!v4l2_chip_match_host(&reg->match))
	//	return -EINVAL;

	vidioc_g_sensor_reg(file, fh, reg);

	//reg->val = 
	//reg->size = 1;
	
	return 0;
}

static int vidioc_s_register (struct file *file, void *fh,
				struct v4l2_dbg_register *reg)
{
	//struct dmw96cam_fh *ofh = (struct dmw96cam_fh*)fh;

	//if (!v4l2_chip_match_host(&reg->match))
	//	return -EINVAL;

	//v4l2_chip_ident_i2c_client()

	vidioc_s_sensor_reg(file, fh, reg);
	
	return 0;
}
#endif

/*
static int vidioc_enum_slaves(struct file *file, void *fh,
			      struct v4l2_slave_info *si)
{
	struct dmw96cam_fh *ofh = fh;
	struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval;

	if (si->index > DMW96CAM_SLAVE_FLASH)
		return -ENODEV;

	mutex_lock(&vdev->mutex);
	if (vdev->slave[si->index] == v4l2_int_device_dummy()) {
		rval = -EBUSY;
		goto out;
	}

	rval = vidioc_int_enum_slaves(vdev->slave[si->index], si);
	if (rval == -ENOIOCTLCMD)
		return -EINVAL;

out:
	mutex_unlock(&vdev->mutex);

	return rval;
}
*/
/**
 * vidioc_default - private IOCTL handler
 * @file: ptr. to system file structure
 * @fh: ptr to hold address of dmw96cam_fh struct (per-filehandle data)
 * @cmd: ioctl cmd value
 * @arg: ioctl arg value
 *
 * If the sensor being used is a "smart sensor", this request is returned to
 * caller with -EINVAL err code.  Otherwise if the control id is the private
 * VIDIOC_PRIVATE_CIU_AEWB_REQ to update the analog gain or exposure,
 * then this request is forwared directly to the sensor to incorporate the
 * feedback. The request is then passed on to the CIU private IOCTL handler,
 * ciu_handle_private()
 */
static long vidioc_default(struct file *file, void *fh, bool valid_prio,
                           int cmd, void *arg)
{
	//struct dmw96cam_fh *ofh = file->private_data;
	//struct dmw96cam_videodev *vdev = ofh->vdev;
	int rval = 0;
	CIU_START();
	switch (cmd) {
		
	    case VIDIOC_G_PHYSICAL:
			rval = vidioc_g_physical_request(fh, (struct v4l2_buffer*)arg);
			goto out;
		

		// TODO Dudi have to remove
		//case VIDIOC_ENUM_SLAVES:
		//	rval = vidioc_enum_slaves(file, fh, arg);
		//	goto out;

		/*
		case VIDIOC_PRIVATE_CIU_AEWB_REQ:
		{
			// Need to update sensor first 
			struct ciu3a_aewb_data *data;
			struct v4l2_control vc;

			data = (struct ciu3a_aewb_data *) arg;
			if (data->update & SET_EXPOSURE) {
				dev_info(&vdev->vfd->dev, "using "
					 "VIDIOC_PRIVATE_CIU_AEWB_REQ to set "
					 "exposure is deprecated!\n");
				vc.id = V4L2_CID_EXPOSURE;
				vc.value = data->shutter;
				mutex_lock(&vdev->mutex);
				rval = vidioc_int_s_ctrl(vdev->vdev_sensor,
							 &vc);
				mutex_unlock(&vdev->mutex);
				if (rval)
					goto out;
			}
			if (data->update & SET_ANALOG_GAIN) {
				dev_info(&vdev->vfd->dev, "using "
					 "VIDIOC_PRIVATE_CIU_AEWB_REQ to set "
					 "gain is deprecated!\n");
				vc.id = V4L2_CID_GAIN;
				vc.value = data->gain;
				mutex_lock(&vdev->mutex);
				rval = vidioc_int_s_ctrl(vdev->vdev_sensor,
							 &vc);
				mutex_unlock(&vdev->mutex);
				if (rval)
					goto out;
			}

		}
		break;
		
		case VIDIOC_PRIVATE_CIU_AF_REQ: {
			// Need to update lens first 
			struct ciu_af_data *data;
			struct v4l2_control vc;

			if (!vdev->vdev_lens) {
				rval = -EINVAL;
				goto out;
			}
			data = (struct ciu_af_data *) arg;
			if (data->update & LENS_DESIRED_POSITION) {
				dev_info(&vdev->vfd->dev, "using "
					 "VIDIOC_PRIVATE_CIU_AF_REQ to set "
					 "lens position is deprecated!\n");
				vc.id = V4L2_CID_FOCUS_ABSOLUTE;
				vc.value = data->desired_lens_direction;
				mutex_lock(&vdev->mutex);
				rval = vidioc_int_s_ctrl(vdev->vdev_lens, &vc);
				mutex_unlock(&vdev->mutex);
				if (rval)
					goto out;
			}
			if (data->update & REQUEST_STATISTICS) {
				dev_info(&vdev->vfd->dev, "using "
					 "VIDIOC_PRIVATE_CIU_AF_REQ to set "
					 "lens position is deprecated!\n");
				vc.id = V4L2_CID_FOCUS_ABSOLUTE;
				mutex_lock(&vdev->mutex);
				rval = vidioc_int_g_ctrl(vdev->vdev_lens, &vc);
				mutex_unlock(&vdev->mutex);
				if (rval)
					goto out;
				data->xtrastats.lens_position = vc.value;
			}
		}
		break;
		*/
		default:
			return 0;
	}

	/*
	mutex_lock(&vdev->mutex);
	rval = ciu_handle_private(cmd, arg);
	mutex_unlock(&vdev->mutex);
	*/
	
out:
	CIU_END();
	return rval;
}

/*
 *
 * File operations.
 *
 */

static long dmw96cam_unlocked_ioctl(struct file *file, unsigned int cmd,
				       unsigned long arg)
{
	//return (long)video_ioctl2(file->f_dentry->d_inode, file, cmd, arg);
	return (long)video_ioctl2(file, cmd, arg);
}

/**
 * dmw96cam_poll - file operations poll handler
 * @file: ptr. to system file structure
 * @wait: system poll table structure
 *
 */
static unsigned int dmw96cam_poll(struct file *file,
				     struct poll_table_struct *wait)
{
	struct dmw96cam_fh *fh = file->private_data;
	unsigned int res;
	CIU_START();
	res = videobuf_poll_stream(file,&fh->vbq,wait);
	CIU_END();
	return res;
}

/**
 * dmw96cam_mmap - file operations mmap handler
 * @file: ptr. to system file structure
 * @vma: system virt. mem. area structure
 *
 * Maps a virtual memory area via the video buffer API
 */
static int dmw96cam_mmap(struct file *file, struct vm_area_struct *vma)
{
	
	struct dmw96cam_fh *fh = file->private_data;
	return videobuf_mmap_mapper(&fh->vbq, vma);
}

/**
 * dmw96cam_open - file operations open handler
 * @inode: ptr. to system inode structure
 * @file: ptr. to system file structure
 *
 * Allocates and initializes the per-filehandle data (dmw96cam_fh),
 * enables the sensor, opens/initializes the CIU interface and the
 * video buffer queue.  Note that this function will allow multiple
 * file handles to be open simultaneously, however only the first
 * handle opened will initialize the CIU.  It is the application
 * responsibility to only use one handle for streaming and the others
 * for control only.
 * This function returns 0 upon success and -ENODEV upon error.
 */
static int dmw96cam_open(struct file *file)
{
	struct video_device *vfd = video_devdata(file);
	struct dmw96cam_videodev *vdev = NULL;
	struct dmw96cam_fh *fh;
	int i;
	CIU_START();

	BUG_ON(!vfd);

	vdev = video_get_drvdata(vfd);

	if (!vdev || !vdev->vfd)
		return -ENODEV;

	fh = kzalloc(sizeof(*fh), GFP_KERNEL);
	if (fh == NULL)
		return -ENOMEM;

	mutex_lock(&vdev->mutex);
	for (i = 0; i <= DMW96CAM_SLAVE_FLASH; i++) {
		if (vdev->slave[i] != v4l2_int_device_dummy()
		    && !try_module_get(vdev->slave[i]->module)) {
			mutex_unlock(&vdev->mutex);
			goto out_try_module_get;
		}
	}

	if (atomic_inc_return(&vdev->users) == 1) {
		ciu_enable_block();
		if (dmw96cam_slave_power_set(vdev, V4L2_POWER_ON,DMW96CAM_SLAVE_POWER_ALL))
			goto out_slave_power_set_standby;

		dmw96cam_slave_power_set(vdev, V4L2_POWER_STANDBY,DMW96CAM_SLAVE_POWER_SENSOR);

		dmw96cam_slave_power_suggest(vdev, V4L2_POWER_STANDBY,DMW96CAM_SLAVE_POWER_LENS);
	}

	fh->vdev = vdev;
	
	if (vdev->vdev_sensor != v4l2_int_device_dummy()) {
		if (s_pix_parm(vdev, &(vdev->pix), &(vdev->timeperframe ) )) {
				dev_err(&vdev->vfd->dev,
					"ciu doesn't like the sensor!\n");
				goto out_slave_power_set_standby;
		}
		
		PDEBUG("h=%d w=%d size=%d format=%s \n",vdev->pix.height , vdev->pix.width , vdev->pix.sizeimage , format_to_string(vdev->pix.pixelformat));
		
	}

	mutex_unlock(&vdev->mutex);

	file->private_data = fh;

	spin_lock_init(&fh->vbq_lock);

	videobuf_queue_dma_contig_init(&fh->vbq, &dmw96cam_vbq_ops, dmw96ciu->dev,
				&fh->vbq_lock, V4L2_BUF_TYPE_VIDEO_CAPTURE,
				V4L2_FIELD_NONE,
				sizeof(struct videobuf_buffer), fh,NULL);
	CIU_END();
	return 0;

out_slave_power_set_standby:
	dmw96cam_slave_power_set(vdev, V4L2_POWER_OFF,
				    DMW96CAM_SLAVE_POWER_ALL);

	ciu_disable_block();
	atomic_dec(&vdev->users);
	mutex_unlock(&vdev->mutex);

out_try_module_get:
	for (i--; i >= 0; i--)
		if (vdev->slave[i] != v4l2_int_device_dummy())
			module_put(vdev->slave[i]->module);

	kfree(fh);

	return -ENODEV;
}

/**
 * dmw96cam_release - file operations release handler
 * @inode: ptr. to system inode structure
 * @file: ptr. to system file structure
 *
 * Complement of dmw96cam_open.  This function will flush any scheduled
 * work, disable the sensor, close the CIU interface, stop the
 * video buffer queue from streaming and free the per-filehandle data
 * (dmw96cam_fh).  Note that because multiple open file handles
 * are allowed, this function will only close the CIU and disable the
 * sensor when the last open file handle (by count) is closed.
 * This function returns 0.
 */
static int dmw96cam_release(struct file *file)
{
	struct dmw96cam_fh *fh = file->private_data;
	struct dmw96cam_videodev *vdev = fh->vdev;
	int i;
	CIU_START();
	mutex_lock(&vdev->mutex);
	if (vdev->streaming == file) {
        ciu_buf_init();
		videobuf_streamoff(&fh->vbq);
		dmw96cam_slave_power_set(
			vdev, V4L2_POWER_STANDBY,
			DMW96CAM_SLAVE_POWER_SENSOR);
		dmw96cam_slave_power_suggest(
			vdev, V4L2_POWER_STANDBY,
			DMW96CAM_SLAVE_POWER_LENS);
		vdev->streaming = NULL;
	}

	if (atomic_dec_return(&vdev->users) == 0) {
		dmw96cam_slave_power_set(vdev, V4L2_POWER_OFF , DMW96CAM_SLAVE_POWER_ALL);
        ciu_disable_block();
	}
	mutex_unlock(&vdev->mutex);

	file->private_data = NULL;

	for (i = 0; i <= DMW96CAM_SLAVE_FLASH; i++)
		if (vdev->slave[i] != v4l2_int_device_dummy())
			module_put(vdev->slave[i]->module);

	kfree(fh);
	CIU_END();
	return 0;
}

static struct v4l2_file_operations dmw96cam_fops = {
	.owner = THIS_MODULE,
	.unlocked_ioctl = dmw96cam_unlocked_ioctl,
	.poll = dmw96cam_poll,
	.mmap = dmw96cam_mmap,
	.open = dmw96cam_open,
	.release = dmw96cam_release,
};

static void dmw96cam_vfd_name_update(struct dmw96cam_videodev *vdev)
{
	struct video_device *vfd = vdev->vfd;
	int i;

	strlcpy(vfd->name, CAM_NAME, sizeof(vfd->name));
	for (i = 0; i <= DMW96CAM_SLAVE_FLASH; i++) {
		strlcat(vfd->name, "/", sizeof(vfd->name));
		if (vdev->slave[i] == v4l2_int_device_dummy())
			continue;
		strlcat(vfd->name, vdev->slave[i]->name, sizeof(vfd->name));
	}
	dev_info(&vdev->vfd->dev, "video%d is now %s\n", vfd->minor, vfd->name);
}

/**
 * dmw96cam_device_unregister - V4L2 detach handler
 * @s: ptr. to standard V4L2 device information structure
 *
 * Detach sensor and unregister and release the video device.
 */
static void dmw96cam_device_unregister(struct v4l2_int_device *s)
{
	struct dmw96cam_videodev *vdev = s->u.slave->master->priv;
	struct dmw96cam_hw_config hwc;

	BUG_ON(vidioc_int_g_priv(s, &hwc) < 0);

	mutex_lock(&vdev->mutex);

	if (vdev->slave[hwc.dev_type] != v4l2_int_device_dummy()) {
		vdev->slave[hwc.dev_type] = v4l2_int_device_dummy();
		vdev->slaves--;
		dmw96cam_vfd_name_update(vdev);
	}

	if (vdev->slaves == 0 && vdev->vfd) {
		if (vdev->vfd->minor == -1) {
			/*
			 * The device was never registered, so release the
			 * video_device struct directly.
			 */
			video_device_release(vdev->vfd);
		} else {
			/*
			 * The unregister function will release the
			 * video_device struct as well as
			 * unregistering it.
			 */
			video_unregister_device(vdev->vfd);
		}
		vdev->vfd = NULL;
	}

	mutex_unlock(&vdev->mutex);
}

static const struct v4l2_ioctl_ops dmw96cam_ioctl_ops = {
	.vidioc_querycap	 = vidioc_querycap,
	.vidioc_enum_fmt_vid_cap = vidioc_enum_fmt_vid_cap,
	.vidioc_g_fmt_vid_cap	 = vidioc_g_fmt_vid_cap,
	.vidioc_s_fmt_vid_cap	 = vidioc_s_fmt_vid_cap,
	.vidioc_try_fmt_vid_cap	 = vidioc_try_fmt_vid_cap,
	.vidioc_reqbufs		 = vidioc_reqbufs,
	.vidioc_querybuf	 = vidioc_querybuf,
	.vidioc_qbuf		 = vidioc_qbuf,
	.vidioc_dqbuf		 = vidioc_dqbuf,
	.vidioc_streamon	 = vidioc_streamon,
	.vidioc_streamoff	 = vidioc_streamoff,
	.vidioc_enum_input	 = vidioc_enum_input,
	.vidioc_g_input		 = vidioc_g_input,
	.vidioc_s_input		 = vidioc_s_input,
	.vidioc_queryctrl	 = vidioc_queryctrl,
	.vidioc_g_ext_ctrls	 = vidioc_g_ext_ctrls,
	.vidioc_s_ext_ctrls	 = vidioc_s_ext_ctrls,
	.vidioc_g_ctrl		= vidioc_g_ctrl,
	.vidioc_s_ctrl		= vidioc_s_ctrl,
	.vidioc_g_parm		 = vidioc_g_parm,
	.vidioc_s_parm		 = vidioc_s_parm,
	.vidioc_cropcap		 = vidioc_cropcap,
	.vidioc_g_crop		 = vidioc_g_crop,
	.vidioc_s_crop		 = vidioc_s_crop,
	.vidioc_enum_frameintervals		= vidioc_enum_frameintervals,
	.vidioc_enum_framesizes			= vidioc_enum_framesizes,
	.vidioc_g_chip_ident			= vidioc_g_chip_ident,
#ifdef CONFIG_VIDEO_ADV_DEBUG
	.vidioc_g_register				= vidioc_g_register,
	.vidioc_s_register				= vidioc_s_register,
#endif
	.vidioc_default		 = vidioc_default,
};

/**
 * dmw96cam_device_register - V4L2 attach handler
 * @s: ptr. to standard V4L2 device information structure
 *
 * Allocates and initializes the V4L2 video_device structure, initializes
 * the sensor, and finally
 registers the device with V4L2 based on the
 * video_device structure.
 *
 * Returns 0 on success, otherwise an appropriate error code on
 * failure.
 */
static int dmw96cam_device_register(struct v4l2_int_device *s)
{
	struct dmw96cam_videodev *vdev = s->u.slave->master->priv;
	struct dmw96cam_hw_config hwc;
	struct video_device *vfd;
	int rval = 0;

	//return 0;

    /* We need to check rval just once. The place is here. */
	if (vidioc_int_g_priv(s, &hwc))
		return -ENODEV;

	if (vdev->index != hwc.dev_index)
		return -ENODEV;

	if (hwc.dev_type < 0 || hwc.dev_type > DMW96CAM_SLAVE_FLASH)
		return -EINVAL;

	if (vdev->slave[hwc.dev_type] != v4l2_int_device_dummy())
		return -EBUSY;

    mutex_lock(&vdev->mutex);
	if (atomic_read(&vdev->users)) {
		printk("we're open (%d), can't register\n",
			atomic_read(&vdev->users));
		mutex_unlock(&vdev->mutex);
		return -EBUSY;
	}

	vdev->slaves++;
	vdev->slave[hwc.dev_type] = s;
	vdev->slave_config[hwc.dev_type] = hwc;

	if (hwc.dev_type == DMW96CAM_SLAVE_SENSOR) {
        rval = ciu_enable_block();
		if (rval < 0) {
			printk("can't get CIU, sensor init failed\n");
			goto err;
		}
	}

    rval = dmw96cam_slave_power_set(vdev, V4L2_POWER_ON, 1 << hwc.dev_type);
  
	if (rval)
		goto err_dmw96cam_slave_power_set;
	if (hwc.dev_type == DMW96CAM_SLAVE_SENSOR) {
		struct v4l2_format format;
		struct v4l2_streamparm a;

		format.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		rval = vidioc_int_g_fmt_cap(vdev->vdev_sensor, &format);

		a.type = V4L2_BUF_TYPE_VIDEO_CAPTURE;
		rval |= vidioc_int_g_parm(vdev->vdev_sensor, &a);
		if (rval)
			rval = -EBUSY;

		vdev->pix = format.fmt.pix;
		vdev->timeperframe = a.parm.capture.timeperframe;
	}
	
	dmw96cam_slave_power_set(vdev, V4L2_POWER_OFF, 1 << hwc.dev_type);
    
	if (hwc.dev_type == DMW96CAM_SLAVE_SENSOR){
            ciu_disable_block();
	}

	if (rval)
		goto err;
    
    // Are we the first slave? 
	if (vdev->slaves == 1) {
		// initialize the video_device struct 
		vdev->vfd = video_device_alloc();
        vfd = vdev->vfd;
		if (!vfd) {
			dev_err(&vfd->dev,
				"could not allocate video device struct\n");
			return -ENOMEM;
		}
        vfd->release	= video_device_release;
		vfd->minor	= -1;
		vfd->fops	= &dmw96cam_fops;  	
		vfd->ioctl_ops	= &dmw96cam_ioctl_ops;
		video_set_drvdata(vfd, vdev);
        if (video_register_device(vfd, VFL_TYPE_GRABBER,hwc.dev_minor) < 0) {
			dev_err(&vfd->dev,
				"could not register V4L device\n");
			vfd->minor = -1;
			rval = -EBUSY;
			goto err;
		}
	} else {
		vfd = vdev->vfd;
	}
	
	dmw96cam_vfd_name_update(vdev);
    
	mutex_unlock(&vdev->mutex);

	PDEBUG("\n\n\n register slave	 s u c c e s s   ! ! !\n\n\n");

	return 0;

err_dmw96cam_slave_power_set:
	if (hwc.dev_type == DMW96CAM_SLAVE_SENSOR){
        ciu_disable_block();
	}

err:
	if (s == vdev->slave[hwc.dev_type]) {
		vdev->slave[hwc.dev_type] = v4l2_int_device_dummy();
		vdev->slaves--;
	}

	mutex_unlock(&vdev->mutex);
	dmw96cam_device_unregister(s);

	return rval;
}

static struct v4l2_int_master dmw96cam_master = {
	.attach = dmw96cam_device_register,
	.detach = dmw96cam_device_unregister,
};

/*
 *
 * Module initialisation and deinitialisation
 *
 */


#ifdef CONFIG_PM
/*
static int dmw96cam_suspend(struct platform_device *pdev, pm_message_t state)
{
	
	struct dmw96cam_device *cam = platform_get_drvdata(pdev);

	if (atomic_read(&cam->users) == 0)
		return 0;

	if (!atomic_read(&cam->reset_disable))
		dmw96cam_captute_stop(cam);

	dmw96cam_sensor_disable(cam);
	dmw96cam_poweron_reset(cam);
	

	return 0;
}

static int dmw96cam_resume(struct platform_device *pdev)
{
	
	struct dmw96cam_device *cam = platform_get_drvdata(pdev);

	if (atomic_read(&cam->users) == 0)
		return 0;

	dmw96cam_hwinit(cam);
	dmw96cam_sensor_enable(cam);

	if (!atomic_read(&cam->reset_disable))
		dmw96cam_capture_cont(cam);

	
	return 0;
}
*/
#endif /* CONFIG_PM */

/*
static int dmw96cam_remove(struct platform_device *pdev)
{
	struct dmw96cam_device *cam = dmw96cam;
	int i;

	if (!cam)
		return 0;

	for (i = 0; i < DMW96CAM_VIDEODEVS; i++) {
		if (cam->vdevs[i].cam == NULL)
			continue;

		v4l2_int_device_unregister(&cam->vdevs[i].master);
		cam->vdevs[i].cam = NULL;
	}

	dmw96cam = NULL;

	kfree(cam);

	return 0;
}
*/

/*
 *
 * Module initialisation and deinitialisation
 *
 */

static void dmw96cam_exit(void)
{
	struct dmw96cam_device *cam = dmw96cam;
	int i;

	if (!cam)
		return;

	for (i = 0; i < DMW96CAM_VIDEODEVS; i++) {
		if (cam->vdevs[i].cam == NULL)
			continue;

		v4l2_int_device_unregister(&cam->vdevs[i].master);
		cam->vdevs[i].cam = NULL;
	}

	dmw96cam = NULL;

	kfree(cam);
}



static int __init dmw96cam_init(void)
{
	struct dmw96cam_device *cam;
	int i;

    cam = kzalloc(sizeof(*cam), GFP_KERNEL);
	if (!cam) {
		printk(KERN_ERR "%s: could not allocate memory\n", __func__);
		goto err;
	}

	dmw96cam = cam;

	for (i = 0; i < DMW96CAM_VIDEODEVS; i++) {
		struct dmw96cam_videodev *vdev = &cam->vdevs[i];
		struct v4l2_int_device *m = &vdev->master;

		m->module       = THIS_MODULE;
		strlcpy(m->name, CAM_NAME, sizeof(m->name));
		m->type         = v4l2_int_type_master;
		m->u.master     = &dmw96cam_master;
		m->priv		= vdev;

		mutex_init(&vdev->mutex);
		vdev->index             = i;
		vdev->cam               = cam;
		vdev->vdev_sensor =
			vdev->vdev_lens =
			vdev->vdev_flash = v4l2_int_device_dummy();
#ifdef DMW96CAM_POWEROFF_DELAY
		setup_timer(&vdev->poweroff_timer,
			    dmw96cam_slave_power_timer, (unsigned long)vdev);
		INIT_WORK(&vdev->poweroff_work, dmw96cam_slave_power_work);
#endif /* DMW96CAM_POWEROFF_DELAY */

		if (v4l2_int_device_register(m))
			goto err;
	}

	PDEBUG("\n\n\n P r o b i n g  CIU success ! ! !\n\n\n");

	return 0;

err:
	dmw96cam_exit();
	return -ENODEV;
}


MODULE_AUTHOR("Dudi Dolev <dudi.dolev@dspg.com>");
MODULE_DESCRIPTION("DMW96 Video for Linux camera driver");
MODULE_LICENSE("GPL");


late_initcall(dmw96cam_init);
module_exit(dmw96cam_exit);

