/*
 *  linux/drivers/serial/dw_uart.c
 *
 *  Driver for DSPG DW style serial ports
 *
 *  Based on drivers/char/serial.c, by Linus Torvalds, Theodore Ts'o.
 *
 *  Copyright (C) 2009 DSP Group Inc.
 *  Copyright (C) 2010 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#if defined(CONFIG_SERIAL_DW_CONSOLE) && defined(CONFIG_MAGIC_SYSRQ)
#define SUPPORT_SYSRQ
#endif

#include <linux/module.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <linux/console.h>
#include <linux/sysrq.h>
#include <linux/device.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/serial_core.h>
#include <linux/serial.h>
#include <linux/platform_device.h>
#include <linux/resource.h>

#include <linux/clk.h>

#include <asm/io.h>
#include <asm/sizes.h>
#include <asm/uaccess.h>

#include <mach/hardware.h>

#define UART_NR				14
#define UART_DUMMY_RSR_RX		256
#define UART_FIFO_SIZE			128

#ifdef CONFIG_SERIAL_DW_TTYDW

#define SERIAL_DW_MAJOR			254
#define SERIAL_DW_MINOR			74
#define SERIAL_DW_NR			UART_NR
#define SERIAL_DW_NAME			"ttyDW"

#else

#define SERIAL_DW_MAJOR			TTY_MAJOR
#define SERIAL_DW_MINOR			64
#define SERIAL_DW_NR			UART_NR
#define SERIAL_DW_NAME			"ttyS"

#endif
#ifdef CONFIG_ARCH_DMW
#define NR_PORTS			3
#else
#define NR_PORTS			2
#endif

#define TCSETRXWM			0x9901
#define TCSETTXWM			0x9902
#define TCGETRXWM			0x9911
#define TCGETTXWM			0x9912

#if CONFIG_SERIAL_DW_REVISION == 1
# define REGOFFSET			0x04
#else
# define REGOFFSET			0x00
#endif

#define DW_UART_CTL			0x00
#define DW_UART_CFG			0x04
#define DW_UART_DIV			0x08

/* these are only available on revision 1 */
#define DW_UART_INT_DIV			0x08
#define DW_UART_FRAC_DIV		0x0C

#define DW_UART_TX_WM			(0x0C + REGOFFSET)
#define DW_UART_RX_WM			(0x10 + REGOFFSET)
#define DW_UART_FIFO_ISR		(0x14 + REGOFFSET)
#define DW_UART_FIFO_IER		(0x18 + REGOFFSET)
#define DW_UART_FIFO_ICR		(0x1C + REGOFFSET)
#define DW_UART_TX_LVL			(0x20 + REGOFFSET)
#define DW_UART_RX_LVL			(0x24 + REGOFFSET)
#define DW_UART_TX_DATA			(0x28 + REGOFFSET)
#define DW_UART_RX_DATA			(0x2C + REGOFFSET)
#define DW_UART_STAT			(0x30 + REGOFFSET)

#define DW_UART_CTL_UART_EN		(1 << 0)	/* Enable the UART block operation. */
#define DW_UART_CTL_UART_DIS		(0 << 0)	/* Disable the UART block operation. */
#define DW_UART_RESET			(1 << 7)	/* Reset the UART block operation. */

#define DW_UART_CFG_DATA_LENGTH_MASK	((1 << 2) | (1 << 3))
#define DW_UART_CFG_DATA_LENGTH_8_VAL	(0 << 2)	/* Data is 8 bits length. */
#define DW_UART_CFG_DATA_LENGTH_7_VAL	(1 << 2)	/* Data is 7 bits length. */
#define DW_UART_CFG_DATA_LENGTH_9_VAL	(1 << 3)	/* Data is 9 bits length. */
#define DW_UART_CFG_STOP_CFG		(1 << 4)	/* Number of stop bits. 0=1bit, 1=2bits */
#define DW_UART_CFG_PAR_OD_EVEN		(1 << 5)	/* Parity. 0=even, 1=odd */
#define DW_UART_CFG_PAR_EN		(1 << 6)	/* Parity enabled/disabled. */
#define DW_UART_CFG_SW_RST		(1 << 7)
#define DW_UART_CFG_OD_EN		(1 << 8)
#define DW_UART_CFG_RTSN_EN		(1 << 9)	/* RTSN flow control enabled/disabled. */
#define DW_UART_CFG_CTSN_EN		(1 << 10)	/* CTSN flow control enabled/disabled. */
#define DW_UART_CFG_BREAK_EN		(1 << 11)	/* BREAK detection logic enabled/disabled. */
#define DW_UART_CFG_IRDA_DTX_INV	(1 << 13)
#define DW_UART_CFG_IRDA_DRX_INV	(1 << 14)
#define DW_UART_CFG_IRDA_UART		(1 << 15)

#define DW_UART_DIV_FRAC_RATE_LOC	(12)
#define DW_UART_DIV_INT_RATE_MASK	(0x3FF)

#define DW_UART_TX_FIFO_OVER_ISR	(1 << 0)	/* Tx FIFO over-run interrupt status. */
#define DW_UART_TX_FIFO_WTR_ISR		(1 << 1)	/* Tx FIFO watermark interrupt status. */
#define DW_UART_RX_FIFO_UNDER_ISR	(1 << 8)	/* Rx FIFO under-run interrupt status. */
#define DW_UART_RX_FIFO_OVER_ISR	(1 << 9)	/* Rx FIFO over-run interrupt status. */
#define DW_UART_RX_FIFO_WTR_ISR		(1 << 10)	/* Rx FIFO watermark interrupt status. */
#define DW_UART_RX_FIFO_TIMEOUT_ISR	(1 << 11)	/* Rx FIFO timeout interrupt status. */
#define DW_UART_RX_BREAK_ISR		(1 << 12)	/* Rx FIFO BREAK interrupt status. */

#define DW_UART_TX_FIFO_OVER_IER	(1 << 0)	/* Tx FIFO over-run interrupt enable. */
#define DW_UART_TX_FIFO_WTR_IER		(1 << 1)	/* Tx FIFO watermark interrupt enable. */
#define DW_UART_RX_FIFO_UNDER_IER	(1 << 8)	/* Rx FIFO under-run interrupt enable. */
#define DW_UART_RX_FIFO_OVER_IER	(1 << 9)	/* Rx FIFO over-run interrupt enable. */
#define DW_UART_RX_FIFO_WTR_IER		(1 << 10)	/* Rx FIFO watermark interrupt enable. */
#define DW_UART_RX_FIFO_TIMEOUT_IER	(1 << 11)	/* Rx FIFO timeout interrupt enable. */
#define DW_UART_RX_BREAK_IER		(1 << 12)	/* Rx FIFO BREAK interrupt enable. */
#define DW_UART_STARTUP_IER		(DW_UART_RX_FIFO_OVER_IER | \
					 DW_UART_RX_FIFO_WTR_IER | \
					 DW_UART_RX_FIFO_TIMEOUT_IER | \
					 DW_UART_RX_BREAK_IER)

#define DW_UART_TX_IER			(DW_UART_TX_FIFO_OVER_IER|DW_UART_TX_FIFO_WTR_IER)
#define DW_UART_RX_OE_BE_IER		(DW_UART_RX_FIFO_OVER_IER|DW_UART_RX_BREAK_IER)

#define DW_UART_TX_FIFO_OVER_ICR	(1 << 0)	/* Tx FIFO over-run interrupt clear.*/
#define DW_UART_RX_FIFO_UNDER_ICR	(1 << 8)	/* Rx FIFO under-run interrupt clear.*/
#define DW_UART_RX_FIFO_OVER_ICR	(1 << 9)	/* Rx FIFO over-run interrupt clear.*/
#define DW_UART_RX_BREAK_ICR		(1 << 12)	/* Rx BREAK interrupt clear.*/
#define DW_UART_CLR_ICR			(DW_UART_TX_FIFO_OVER_ICR | \
					 DW_UART_RX_FIFO_UNDER_ICR | \
					 DW_UART_RX_FIFO_OVER_ICR | \
					 DW_UART_RX_BREAK_ICR)

#define DW_UART_TX_FIFO_LEVEL_MASK	(0x1F)		/* Tx FIFO current level bits. Spreads on bits [0 : Log2(fifo_depth)]. */
#define DW_UART_RX_FIFO_LEVEL_MASK	(0x1F)		/* Rx FIFO current level bits. Spreads on bits [0 : Log2(fifo_depth)]. */

#define DW_UART_RX_DATA_WORD		(0x1FF)		/* UART word from Rx FIFO - received word. */
#define DW_UART_RX_DATA_PE		(0x200)		/* UART word from Rx FIFO - Parity error bit. */
#define DW_UART_RX_DATA_FE		(0x400)		/* UART word from Rx FIFO - frame error bit. */
#define DW_UART_RX_PE_FE		(DW_UART_RX_DATA_PE | DW_UART_RX_DATA_FE)

#define DW_UART_STAT_TX_EMPTY		(1 << 0)	/* Tx FIFO completely empty. */
#define DW_UART_STAT_TX_FULL		(1 << 1)	/* Tx FIFO completely full. */
#define DW_UART_STAT_RTS		(1 << 2)	/* The inverted(!) values of the RTS port. */
#define DW_UART_STAT_CTS		(1 << 3)	/* The inverted(!) values of the CTS port. */
#define DW_UART_STAT_RX_EMPTY		(1 << 8)	/* Rx FIFO completely empty. */
#define DW_UART_STAT_RX_FULL		(1 << 9)	/* Rx FIFO completely full. */
#define DW_UART_STAT_BREAK_DETECTED	(1 << 10)	/* Rx BREAK was detected. */

#define DW_UART_STAT_MASK_LOC_0		(9)
#define DW_UART_STAT_MASK_LOC_1		(11)
#define DW_UART_STAT_MASK_LOC_2_3	(7)

struct dw_port {
	struct uart_port port;
	struct clk *clk;
	struct tasklet_struct tasklet;
};

struct dw_baudrate {
    unsigned int intgr;
    unsigned int frac;
};

static struct dw_port dw_ports[UART_NR];

static void dw_uart_get_div(struct uart_port *port, int *intgr, int *frac)
{
#if CONFIG_SERIAL_DW_REVISION == 1
	*intgr = readw(port->membase + DW_UART_INT_DIV);
	*frac  = readw(port->membase + DW_UART_FRAC_DIV) & 0xf;
#else
	*intgr = readw(port->membase + DW_UART_DIV) & 0x3ff;
	*frac  = readw(port->membase + DW_UART_DIV) >> 12;
#endif
}

static void dw_uart_set_div(struct uart_port *port, int intgr, int frac)
{
#if CONFIG_SERIAL_DW_REVISION == 1
	writew(intgr & 0xffff, port->membase + DW_UART_INT_DIV);
	writew(frac  & 0x000f, port->membase + DW_UART_FRAC_DIV);
#else
	writew((intgr & 0x3ff) | ((frac & 0xf) << 12),
	       port->membase + DW_UART_DIV);
#endif
}

static void dw_uart_stop_tx(struct uart_port *port)
{
	struct dw_port *uap = (struct dw_port *)port;
	unsigned int ier;

	ier = readw(uap->port.membase + DW_UART_FIFO_IER);
	ier &= ~(DW_UART_TX_FIFO_OVER_IER|DW_UART_TX_FIFO_WTR_IER);
	writew(ier, uap->port.membase + DW_UART_FIFO_IER);
}

static void dw_uart_start_tx(struct uart_port *port)
{
	struct dw_port *uap = (struct dw_port *)port;
	unsigned int ier;

	ier = readw(uap->port.membase + DW_UART_FIFO_IER);
	ier |= (DW_UART_TX_FIFO_OVER_IER|DW_UART_TX_FIFO_WTR_IER);
	writew(ier, uap->port.membase + DW_UART_FIFO_IER);
}

static void dw_uart_stop_rx(struct uart_port *port)
{
	struct dw_port *uap = (struct dw_port *)port;
	unsigned int mask;

	mask = readw(uap->port.membase + DW_UART_FIFO_IER);
	mask &= ~(DW_UART_RX_FIFO_UNDER_IER | DW_UART_RX_FIFO_OVER_IER |
	          DW_UART_RX_FIFO_WTR_IER | DW_UART_RX_FIFO_TIMEOUT_IER |
	          DW_UART_RX_BREAK_IER);
	writew(mask, uap->port.membase + DW_UART_FIFO_IER);
}

static void dw_uart_enable_ms(struct uart_port *port)
{
	return;
}


static void
#ifdef SUPPORT_SYSRQ
dw_uart_rx_chars(struct dw_port *uap, struct pt_regs *regs)
#else
dw_uart_rx_chars(struct dw_port *uap)
#endif
{
	struct tty_struct *tty = uap->port.state->port.tty;
	unsigned int status, ch, flag, rx_err, rx_err_m, pf_err, pf_err_m;
	unsigned int max_count = 256;

	status = readw(uap->port.membase + DW_UART_STAT);
	while ((status & DW_UART_STAT_RX_EMPTY) == 0 && max_count--) {
		ch = readw(uap->port.membase + DW_UART_RX_DATA);
		flag = TTY_NORMAL;
		uap->port.icount.rx++;

		/*
		 * Note that the error handling code is
		 * out of the main execution path
		 */
		rx_err = readw(uap->port.membase + DW_UART_FIFO_ISR);
		rx_err &= DW_UART_RX_OE_BE_IER;

		pf_err = ch & DW_UART_RX_PE_FE;

		/*
		 * Join error indicaion bits to one word:
		 *   IER - over-run (bit 9), break (bit 12)
		 *   RX_DATA - parity (bit 9), frame (bit 10)
		 *   rx_err_m -   X  |   X  |break|over-run
		 *                3  |   2  |  1  |    0
		 *   pf_err_m - frame|parity|  X  |    X
		 *                3  |   2  |  1  |    0
		 */
		rx_err_m = (rx_err & DW_UART_RX_FIFO_OVER_IER) >>
				DW_UART_STAT_MASK_LOC_0;

		rx_err_m |= (rx_err & DW_UART_RX_BREAK_IER) >>
				DW_UART_STAT_MASK_LOC_1;

		rx_err_m &= uap->port.read_status_mask;

		pf_err_m = (pf_err & DW_UART_RX_PE_FE) >>
				DW_UART_STAT_MASK_LOC_2_3;

		pf_err_m &= uap->port.read_status_mask;

		if (unlikely(rx_err || pf_err)) {
			if (rx_err & DW_UART_RX_BREAK_IER) {
				pf_err &= ~DW_UART_RX_PE_FE;
				uap->port.icount.brk++;
				if (uart_handle_break(&uap->port))
					goto ignore_char;
			} else if (pf_err & DW_UART_RX_DATA_PE)
				uap->port.icount.parity++;
			else if (pf_err & DW_UART_RX_DATA_FE)
				uap->port.icount.frame++;
			if (rx_err & DW_UART_RX_FIFO_OVER_IER)
				uap->port.icount.overrun++;

			if (rx_err_m & (DW_UART_RX_BREAK_IER >>
						DW_UART_STAT_MASK_LOC_1))
				flag = TTY_BREAK;

			else if (pf_err_m & (DW_UART_RX_DATA_PE >>
						DW_UART_STAT_MASK_LOC_2_3))
				flag = TTY_PARITY;

			else if (pf_err_m & (DW_UART_RX_DATA_FE >>
						DW_UART_STAT_MASK_LOC_2_3))
				flag = TTY_FRAME;
		}

		if (uart_handle_sysrq_char(&uap->port, ch & 0xff))
			goto ignore_char;

		if ((((rx_err_m & uap->port.ignore_status_mask) & ~DW_UART_RX_FIFO_OVER_IER)
		         && ((pf_err_m & uap->port.ignore_status_mask) & ~DW_UART_RX_FIFO_OVER_IER)) == 0) {
			tty_insert_flip_char(tty, ch, flag);
		}

		if (((rx_err_m & ~uap->port.ignore_status_mask) & DW_UART_RX_FIFO_OVER_IER)
		        && ((pf_err_m & ~uap->port.ignore_status_mask) & DW_UART_RX_FIFO_OVER_IER)) {
			/*
			 * Overrun is special, since it's reported
			 * immediately, and doesn't affect the current
			 * character
			 */
			tty_insert_flip_char(tty, 0, TTY_OVERRUN);
		}

ignore_char:
		status = readw(uap->port.membase + DW_UART_STAT);
	}
	tty_flip_buffer_push(tty);
}

static void dw_uart_tx_chars(struct dw_port *uap)
{
	struct circ_buf *xmit = &uap->port.state->xmit;
	int count;

	if (uap->port.x_char) {
		writew(uap->port.x_char, uap->port.membase + DW_UART_TX_DATA);
		uap->port.icount.tx++;
		uap->port.x_char = 0;
		return;
	}

	if (uart_circ_empty(xmit) || uart_tx_stopped(&uap->port)) {
		dw_uart_stop_tx(&uap->port);
		return;
	}

	count = uap->port.fifosize >> 1;
	do {
		writew(xmit->buf[xmit->tail],
		       uap->port.membase + DW_UART_TX_DATA);
		xmit->tail = (xmit->tail + 1) & (UART_XMIT_SIZE - 1);
		uap->port.icount.tx++;
		if (uart_circ_empty(xmit))
			break;
	} while (--count > 0);

	if (uart_circ_chars_pending(xmit) < WAKEUP_CHARS)
		uart_write_wakeup(&uap->port);

	if (uart_circ_empty(xmit))
		dw_uart_stop_tx(&uap->port);
}

static void dw_uart_tasklet_func(unsigned long data)
{
	struct dw_port *uap = (struct dw_port *)data;
	unsigned int status, pass_counter = 256;

	if ((readw(uap->port.membase + DW_UART_STAT)) & 0xc) {
		printk("\nCTS-RTS port is LOW\n");
	}

	spin_lock_irq(&uap->port.lock);

	status = readw(uap->port.membase + DW_UART_FIFO_ISR);
	status &= readw(uap->port.membase + DW_UART_FIFO_IER);

	while (status) {
		/*
		 * rx characters:
		 *  1) above watermark
		 *  2) time out reading from FIFO
		 *  3) over-run
		 *  4) BREAK detection
		 */
		if (status & (DW_UART_RX_FIFO_WTR_IER |
		              DW_UART_RX_FIFO_TIMEOUT_IER |
		              DW_UART_RX_FIFO_OVER_IER |
		              DW_UART_RX_BREAK_IER)) {
#ifdef SUPPORT_SYSRQ
			dw_uart_rx_chars(uap, regs);
#else
			dw_uart_rx_chars(uap);
#endif
			if (status & DW_UART_RX_FIFO_OVER_IER)
				writew(DW_UART_RX_FIFO_OVER_ICR,
			               uap->port.membase + DW_UART_FIFO_ICR);

			if (status & DW_UART_RX_BREAK_IER)
				writew(DW_UART_RX_BREAK_ICR,
				       uap->port.membase + DW_UART_FIFO_ICR);
		}

		/*
		 * tx characters:
		 *  1) below watermark
		 *  2) over-run
		 */
		if (status & (DW_UART_TX_FIFO_WTR_IER |
		              DW_UART_TX_FIFO_OVER_IER)) {
			dw_uart_tx_chars(uap);

			if (status & DW_UART_TX_FIFO_OVER_IER)
				writew(DW_UART_TX_FIFO_OVER_ICR,
				       uap->port.membase + DW_UART_FIFO_ICR);
		}

		if (pass_counter-- == 0)
			break;

		status = readw(uap->port.membase + DW_UART_FIFO_ISR);
		status &= readw(uap->port.membase + DW_UART_FIFO_IER);
	}

	spin_unlock_irq(&uap->port.lock);
	enable_irq(uap->port.irq);

	return;
}


static irqreturn_t dw_uart_interrupt(int irq, void *dev_id)
{
	struct dw_port *uap = dev_id;

	disable_irq_nosync(uap->port.irq);
	tasklet_schedule(&uap->tasklet);

	return IRQ_HANDLED;
}

static unsigned int dw_uart_tx_empty(struct uart_port *port)
{
	struct dw_port *uap = (struct dw_port *)port;

	unsigned int status = readw(uap->port.membase + DW_UART_STAT);
	return (status & DW_UART_STAT_TX_EMPTY) ? TIOCSER_TEMT : 0;
}

static unsigned int dw_uart_get_mctrl(struct uart_port *port)
{
	struct dw_port *uap = (struct dw_port *)port;
	unsigned int result = 0;
	unsigned int status = readw(uap->port.membase + DW_UART_STAT);

#define DW_UART_BIT(uartbit, tiocmbit) \
	if (status & uartbit)          \
		result |= tiocmbit

	DW_UART_BIT(!DW_UART_STAT_RTS, TIOCM_RTS);
	DW_UART_BIT(!DW_UART_STAT_CTS, TIOCM_CTS);
#undef DW_UART_BIT

	return result;
}

static void dw_uart_set_mctrl(struct uart_port *port, unsigned int mctrl)
{
	struct dw_port *uap = (struct dw_port *)port;
	unsigned int cr;

	cr = readw(uap->port.membase + DW_UART_CFG);

#define DW_UART_BIT(tiocmbit, uartbit) \
	if (mctrl & tiocmbit)          \
		cr |= uartbit;         \
	else                           \
		cr &= ~uartbit

	//printk("\ncfg %x\n", cr);
	//DW_UART_BIT(TIOCM_RTS, DW_UART_CFG_RTSN_EN);
	//DW_UART_BIT(TIOCM_CTS, DW_UART_CFG_CTSN_EN);
	//printk("\ncfg %x\n", cr);
#undef DW_UART_BIT

	writew(cr, uap->port.membase + DW_UART_CFG);
}

static void dw_uart_break_ctl(struct uart_port *port, int break_state)
{
	return;
}

static int dw_uart_startup(struct uart_port *port)
{
	struct dw_port *uap = (struct dw_port *)port;
	struct tty_struct *tty = port->state->port.tty;
	int retval;

	uap->port.uartclk = clk_get_rate(uap->clk);

	/*
	 * Allocate the IRQ
	 */
	retval = request_irq(uap->port.irq, dw_uart_interrupt, 0,
	                     tty ? tty->name : "dw-uart", uap);
	if (retval)
		goto out;

	writew(DW_UART_CTL_UART_DIS, uap->port.membase + DW_UART_CTL);
	writew(DW_UART_CFG_BREAK_EN, uap->port.membase + DW_UART_CFG);

	/* Don't touch the divisor at startup, use the setting from u-boot.
	 * This can be changed later with dw_uart_set_termios(). */

	writew(DW_UART_CLR_ICR, uap->port.membase + DW_UART_FIFO_ICR);
	writew(UART_FIFO_SIZE/2, uap->port.membase + DW_UART_TX_WM);
	writew(UART_FIFO_SIZE/2, uap->port.membase + DW_UART_RX_WM);
	writew(DW_UART_CTL_UART_EN, uap->port.membase + DW_UART_CTL);

	return 0;

out:
	return retval;
}

static void dw_uart_shutdown(struct uart_port *port)
{
	struct dw_port *uap = (struct dw_port *)port;

	/*
	 * disable all interrupts
	 */
	spin_lock_irq(&uap->port.lock);
	writew(0, uap->port.membase + DW_UART_FIFO_IER);
	writew(DW_UART_CLR_ICR, uap->port.membase + DW_UART_FIFO_ICR);
	spin_unlock_irq(&uap->port.lock);

	/*
	 * Free the interrupt
	 */
	free_irq(uap->port.irq, uap);

	/*
	 * disable the port
	 */
	writew(DW_UART_CTL_UART_DIS, uap->port.membase + DW_UART_CTL);
}

static void
dw_uart_set_termios(struct uart_port *port, struct ktermios *termios,
                    struct ktermios *old)
{
	unsigned int cfg;
	unsigned long flags;
	unsigned int baud;
	struct dw_baudrate quot;

	/*
	 * Ask the core to calculate the divisor for us.
	 */
	baud = uart_get_baud_rate(port, termios, old, 0, port->uartclk/16);

	quot.intgr = (port->uartclk/baud) >> 4;
	quot.frac = (port->uartclk/baud) & 0x000f;

	/* byte size */
	switch (termios->c_cflag & CSIZE) {
	case CS5:
		cfg = 0;
		break;
	case CS6:
		cfg = 0;
		break;
	case CS7:
		cfg = DW_UART_CFG_DATA_LENGTH_7_VAL;
		break;
	default:
		cfg = DW_UART_CFG_DATA_LENGTH_8_VAL;
		break;
	}

	/* stop bit */
	if (termios->c_cflag & CSTOPB)
		cfg |= DW_UART_CFG_STOP_CFG;

	/* parity */
	if (termios->c_cflag & PARENB) {
		cfg |= DW_UART_CFG_PAR_EN;
		if (termios->c_cflag & PARODD)
			cfg |= DW_UART_CFG_PAR_OD_EVEN;
	}

	/* hardware handshake */
	if (termios->c_cflag & CRTSCTS) {
		cfg |= (DW_UART_CFG_RTSN_EN|DW_UART_CFG_CTSN_EN);
		//cfg |= DW_UART_CFG_RTSN_EN;
	}

	if (!(termios->c_iflag & IGNBRK))
		cfg |= DW_UART_CFG_BREAK_EN;
	else
		cfg &= ~DW_UART_CFG_BREAK_EN;


	spin_lock_irqsave(&port->lock, flags);

	/*
	 * Update the per-port timeout.
	 */
	uart_update_timeout(port, termios->c_cflag, baud);

	port->read_status_mask =
		/* bit 9 in IER -> bit 0 */
		(DW_UART_RX_FIFO_OVER_IER >> DW_UART_STAT_MASK_LOC_0);

	if (termios->c_iflag & INPCK)
		port->read_status_mask |=
			/* bits 9-10 in RX_DATA -> bits 2-3 */
			(DW_UART_RX_PE_FE >> DW_UART_STAT_MASK_LOC_2_3);

	if (termios->c_iflag & (BRKINT | PARMRK))
		port->read_status_mask |=
			/* bit 12 in IER -> bit 1 */
			(DW_UART_RX_BREAK_IER >> DW_UART_STAT_MASK_LOC_1);

	/*
	 * Characters to ignore
	 */
	port->ignore_status_mask = 0;
	if (termios->c_iflag & IGNPAR)
		port->ignore_status_mask |=
			(DW_UART_RX_PE_FE >> DW_UART_STAT_MASK_LOC_2_3);

	if (termios->c_iflag & IGNBRK) {
		port->ignore_status_mask |=
			(DW_UART_RX_BREAK_IER >> DW_UART_STAT_MASK_LOC_1);

		/*
		 * If we're ignoring parity and break indicators,
		 * ignore overruns too (for real raw support).
		 */
		if (termios->c_iflag & IGNPAR)
			port->ignore_status_mask |=
				(DW_UART_RX_FIFO_OVER_IER >>
					DW_UART_STAT_MASK_LOC_0);
	}

	if (UART_ENABLE_MS(port, termios->c_cflag))
		dw_uart_enable_ms(port);

	dw_uart_set_div(port, quot.intgr, quot.frac);
	writew(cfg, port->membase + DW_UART_CFG);
	writew(DW_UART_STARTUP_IER, port->membase + DW_UART_FIFO_IER);
	writew(DW_UART_CTL_UART_EN, port->membase + DW_UART_CTL);

	spin_unlock_irqrestore(&port->lock, flags);
}

static const char *dw_uart_type(struct uart_port *port)
{
	struct dw_port *sport = (struct dw_port *)port;

	return sport->port.type == PORT_DW ? "DW_UART" : NULL;
}

/*
 * Release the memory region(s) being used by 'port'
 */
static void dw_uart_release_port(struct uart_port *port)
{
	release_mem_region(port->mapbase, SZ_4K);
}

/*
 * Request the memory region(s) being used by 'port'
 */
static int dw_uart_request_port(struct uart_port *port)
{
	if (!request_mem_region(port->mapbase, SZ_4K, "DW_UART")) {
		dev_err(port->dev, "Memory region busy\n");
		return -EBUSY;
	}

	port->membase = ioremap(port->mapbase, SZ_4K);
	if (!port->membase) {
		dev_err(port->dev, "Unable to map registers\n");
		release_mem_region(port->mapbase, SZ_4K);
		return -EBUSY;
	}

	return 0;
}

/*
 * Configure/autoconfigure the port.
 */
static void dw_uart_config_port(struct uart_port *port, int flags)
{
	struct dw_port *sport = (struct dw_port *)port;

	if (flags & UART_CONFIG_TYPE &&
	    dw_uart_request_port(&sport->port) == 0)
		sport->port.type = PORT_DW;
}

/*
 * verify the new serial_struct (for TIOCSSERIAL).
 */
static int
dw_uart_verify_port(struct uart_port *port, struct serial_struct *ser)
{
	int ret = 0;

	if (ser->type != PORT_UNKNOWN && ser->type != PORT_DW)
		ret = -EINVAL;
	if (ser->irq < 0 || ser->irq >= NR_IRQS)
		ret = -EINVAL;
	if (ser->baud_base < 9600)
		ret = -EINVAL;
	return ret;
}

static int
dw_uart_ioctl(struct uart_port *port, unsigned int cmd, unsigned long arg)
{
	unsigned long data;

	switch (cmd) {
	case TCSETRXWM:
		if (!get_user(data, (unsigned long*)arg)) {
			writew(data, port->membase + DW_UART_RX_WM);
			return 0;
		} else {
			return -EINVAL;
		}
	case TCSETTXWM:
		if (!get_user(data, (unsigned long*)arg)) {
			writew(data, port->membase + DW_UART_TX_WM);
			return 0;
		} else {
			return -EINVAL;
		}
	case TCGETRXWM:
		data = readw(port->membase + DW_UART_RX_WM);
		return put_user(data, (unsigned long __user *)arg);
	case TCGETTXWM:
		data = readw(port->membase + DW_UART_TX_WM);
		return put_user(data, (unsigned long __user *)arg);
	default:
		return -ENOIOCTLCMD;
	}
}

static void
dw_uart_pm(struct uart_port *port, unsigned int state, unsigned int oldstate)
{
	struct dw_port *sport = (struct dw_port *)port;

	if (!state)
		clk_enable(sport->clk);
	else
		clk_disable(sport->clk);
}

static struct uart_ops dw_uart_pops = {
	.pm		= dw_uart_pm,
	.tx_empty	= dw_uart_tx_empty,
	.set_mctrl	= dw_uart_set_mctrl,
	.get_mctrl	= dw_uart_get_mctrl,
	.stop_tx	= dw_uart_stop_tx,
	.start_tx	= dw_uart_start_tx,
	.stop_rx	= dw_uart_stop_rx,
	.enable_ms	= dw_uart_enable_ms,
	.break_ctl	= dw_uart_break_ctl,
	.startup	= dw_uart_startup,
	.shutdown	= dw_uart_shutdown,
	.set_termios	= dw_uart_set_termios,
	.type		= dw_uart_type,
	.release_port	= dw_uart_release_port,
	.request_port	= dw_uart_request_port,
	.config_port	= dw_uart_config_port,
	.verify_port	= dw_uart_verify_port,
	.ioctl		= dw_uart_ioctl,
};

#ifdef CONFIG_SERIAL_DW_CONSOLE

static void dw_uart_console_putchar(struct uart_port *port, int ch)
{
	struct dw_port *uap = (struct dw_port *)port;

	while (readw(uap->port.membase + DW_UART_STAT) & DW_UART_STAT_TX_FULL)
		barrier();
	writew(ch, uap->port.membase + DW_UART_TX_DATA);
}

static void
dw_uart_console_write(struct console *co, const char *s, unsigned int count)
{
	struct dw_port *uap = &dw_ports[co->index];
	unsigned int status, old_cr, new_cr, old_ier, new_ier, old_ctl;

	clk_enable(uap->clk);

	/*
	 *	First save the CFG then disable the interrupts
	 */
	old_ctl = readw(uap->port.membase + DW_UART_CTL);
	writew(DW_UART_CTL_UART_EN, uap->port.membase + DW_UART_CTL);

	old_cr = readw(uap->port.membase + DW_UART_CFG);
	new_cr = old_cr & ~DW_UART_CFG_CTSN_EN;
	writew(new_cr, uap->port.membase + DW_UART_CFG);

	old_ier = readw(uap->port.membase + DW_UART_FIFO_IER);
	new_ier = old_ier & ~DW_UART_TX_IER;
	writew(new_ier, uap->port.membase + DW_UART_FIFO_IER);

	uart_console_write(&uap->port, s, count, dw_uart_console_putchar);

	/*
	 *	Finally, wait for transmitter to become empty,
	 *	restore the CFG and enable the interrupts
	 */
	do {
		status = readw(uap->port.membase + DW_UART_STAT);
	} while (!(status & DW_UART_STAT_TX_EMPTY));
	writew(old_cr, uap->port.membase + DW_UART_CFG);
	writew(old_ier, uap->port.membase + DW_UART_FIFO_IER);
	writew(old_ctl, uap->port.membase + DW_UART_CTL);

	clk_disable(uap->clk);
}

static void __init
dw_uart_console_get_options(struct dw_port *uap, int *baud, int *parity,
                            int *bits)
{
	unsigned int cfg, ibrd, fbrd;

	cfg = readw(uap->port.membase + DW_UART_CTL);

	if (!(cfg & DW_UART_CTL_UART_EN))
		return;

	cfg = readw(uap->port.membase + DW_UART_CFG);

	*parity = 'n';
	if (cfg & DW_UART_CFG_PAR_EN) {
		if (cfg & DW_UART_CFG_OD_EN)
			*parity = 'o';
		else
			*parity = 'e';
	}

	*bits = 0;
	if (cfg & DW_UART_CFG_DATA_LENGTH_7_VAL)
		*bits = 7;
	else if (cfg & DW_UART_CFG_DATA_LENGTH_8_VAL)
		*bits = 8;
	else if (cfg & DW_UART_CFG_DATA_LENGTH_9_VAL)
		*bits = 9;

	dw_uart_get_div(&uap->port, &ibrd, &fbrd);
	*baud = uap->port.uartclk * 4 / (64 * ibrd + fbrd);
}

static int __init dw_uart_console_setup(struct console *co, char *options)
{
	struct dw_port *uap;
	int baud = 38400;
	int bits = 8;
	int parity = 'n';
	int flow = 'n';

	/*
	 * Check whether an invalid uart number has been specified, and
	 * if so, search for the first available port that does have
	 * console support.
	 */
	if (co->index >= UART_NR)
		co->index = 0;
	uap = &dw_ports[co->index];

	/* Has the device been initialized yet? */
	if (!uap->port.mapbase) {
		pr_debug("console on ttyDW%i not present\n", co->index);
		return -ENODEV;
	}

	/* not initialized yet? */
	if (!uap->port.membase) {
		if (dw_uart_request_port(&uap->port))
			return -ENODEV;
	}

	uap->port.uartclk = clk_get_rate(uap->clk);

	if (options)
		uart_parse_options(options, &baud, &parity, &bits, &flow);
	else
		dw_uart_console_get_options(uap, &baud, &parity, &bits);

	return uart_set_options(&uap->port, co, baud, parity, bits, flow);
}

static struct uart_driver dw_uart;

static struct console dw_console = {
	.name		= SERIAL_DW_NAME,
	.write		= dw_uart_console_write,
	.device		= uart_console_device,
	.setup		= dw_uart_console_setup,
	.flags		= CON_PRINTBUFFER,
	.index		= -1,
	.data		= &dw_uart,
};

static int __init dw_rs_console_init(void)
{
	register_console(&dw_console);
	return 0;
}
console_initcall(dw_rs_console_init);

#define DW_CONSOLE	(&dw_console)
#else
#define DW_CONSOLE	NULL
#endif

static struct uart_driver dw_uart = {
	.owner		= THIS_MODULE,
	.driver_name	= "dw-uart",
	.dev_name	= SERIAL_DW_NAME,
	.major		= SERIAL_DW_MAJOR,
	.minor		= SERIAL_DW_MINOR,
	.nr		= UART_NR,
	.cons		= DW_CONSOLE,
};

static int __devinit dw_uart_probe(struct platform_device *pdev)
{
	struct resource *mem_res, *irq_res;
	struct dw_port *uap;
	int ret = 0;
	int id;
	char clkname[6];

	mem_res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!mem_res)
		return -ENODEV;

	irq_res = platform_get_resource(pdev, IORESOURCE_IRQ, 0);
	if (!irq_res)
		return -ENODEV;

	id = pdev->id;
	if (id < 0) {
		for (id = 0; id < NR_PORTS; id++) {
			if (dw_ports[id].port.mapbase == 0)
				break;
		}
	}
	if (id < 0 || id >= NR_PORTS) {
		dev_err(&pdev->dev, "number of supported ports exceeded\n");
		return -EINVAL;
	}

	uap = &dw_ports[id];

	if (uap->port.mapbase && uap->port.mapbase != mem_res->start) {
		dev_err(&pdev->dev, "port %d already in use\n", id);
		return -EBUSY;
	}

	snprintf(clkname, sizeof(clkname), "uart%d", id + 1);
	uap->clk = clk_get(&pdev->dev, clkname);
	if (IS_ERR(uap->clk)) {
		ret = PTR_ERR(uap->clk);
		return ret;
	}

	tasklet_init(&uap->tasklet, dw_uart_tasklet_func, (unsigned long)uap);

	uap->port.dev =         &pdev->dev;
	uap->port.mapbase =     mem_res->start;
	uap->port.membase =     NULL;
	uap->port.iotype =      UPIO_MEM;
	uap->port.irq =         irq_res->start;
	uap->port.fifosize =    128;
	uap->port.ops =         &dw_uart_pops;
	uap->port.flags =       UPF_BOOT_AUTOCONF;
	uap->port.line =        id;

	ret = uart_add_one_port(&dw_uart, &uap->port);
	if (ret) {
		dev_err(&pdev->dev, "adding port failed; err=%d\n", ret);
		uap->port.mapbase = 0;
		clk_put(uap->clk);
		return ret;
	}
	platform_set_drvdata(pdev, uap);

	return ret;
}

static int dw_uart_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct dw_port *sport = platform_get_drvdata(pdev);

	return uart_suspend_port(&dw_uart, &sport->port);
}

static int dw_uart_resume(struct platform_device *pdev)
{
	struct dw_port *sport = platform_get_drvdata(pdev);

	return uart_resume_port(&dw_uart, &sport->port);
}

static int __devexit dw_uart_remove(struct platform_device *pdev)
{
	struct dw_port *sport = platform_get_drvdata(pdev);

	platform_set_drvdata(pdev, NULL);

	if (sport)
		uart_remove_one_port(&dw_uart, &sport->port);

	return 0;
}

static struct platform_driver dw_uart_driver = {
	.probe		= dw_uart_probe,
	.remove		= __devexit_p(dw_uart_remove),
	.suspend	= dw_uart_suspend,
	.resume		= dw_uart_resume,
	.driver		= {
		.owner	= THIS_MODULE,
		.name	= "dw-uart",
	},
};

static int __init dw_uart_init(void)
{
	volatile int ret;

	ret = uart_register_driver(&dw_uart);
	if (ret)
		return ret;

	ret = platform_driver_register(&dw_uart_driver);
	if (ret)
		uart_unregister_driver(&dw_uart);

	return ret;
}


static void __exit dw_uart_exit(void)
{
	platform_driver_unregister(&dw_uart_driver);
	uart_unregister_driver(&dw_uart);

}

module_init(dw_uart_init);
module_exit(dw_uart_exit);

MODULE_AUTHOR("DSP Group");
MODULE_DESCRIPTION("DW serial port driver");
MODULE_LICENSE("GPL");

