/*
 * TODO:
 * 	- dynamically setup var and fix on probe
 * 	- switch backlight on/off
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/mm.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/semaphore.h>
#include <linux/delay.h>
#include <linux/backlight.h>
#include <linux/proc_fs.h>
#include <mach/reset.h>
#include <asm/uaccess.h> /*needed for copy_form_user*/
#include <linux/ktime.h>

#ifndef CONFIG_MACH_VERSATILE_BROADTILE
#include <asm/gpio.h>
#endif

#include <video/dw74fb.h>

MODULE_DESCRIPTION("DW74 LCD Controller framebuffer driver");
MODULE_LICENSE("GPL");

static unsigned underruns = 0;
module_param(underruns, uint, 0644);
MODULE_PARM_DESC(underruns, "Number of underruns that happened.");

static unsigned unhandled_ints = 0;
module_param(unhandled_ints, uint, 0644);
MODULE_PARM_DESC(unhandled_ints, "Number of unhandled interrupts.");

static int mem = 12150;
module_param(mem, int, 0644);
MODULE_PARM_DESC(mem, "Video buffer size (in kB); default is 12150kB for full-HD, 24-bit, double buffering.");

#define DW74FB_NAME "dw74fb"
static char dw74fb_name[] = DW74FB_NAME;

#ifdef CONFIG_FB_DW74_LOW_POWER
#define DW74FB_NAME_LOW_POWER "dw74fb_low_power"
#endif

static struct dw74fb_panel *panel = NULL;
static struct dw74fb *g_dw74fb = NULL;

static int lcdc_mode = DWFB_MODE_LCD;

const struct fb_videomode dw74fb_modedb[] = {
	{ "480p60a",  60,  720,  480, 37037,  60,  16, 30, 9, 62, 6, // 27.00 MHz
	  0, FB_VMODE_NONINTERLACED, FB_MODE_IS_ANDROID },
	{ "720p50a",  50, 1280,  720, 13468, 440, 220, 5, 20, 40, 5, // 74.25 MHz
	  FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED, FB_MODE_IS_ANDROID },
	{ "720p60a",  60, 1280,  720, 13468, 110, 220, 5, 20, 40, 5, // 74.25 MHz
	  FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED, FB_MODE_IS_ANDROID },
	{ "1080p24a",  24, 1920, 1080, 13468, 638, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED, FB_MODE_IS_ANDROID },
	{ "1080p25a",  25, 1920, 1080, 13468, 528, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED, FB_MODE_IS_ANDROID },
	{ "1080p30a",  30, 1920, 1080, 13468,  88, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED, FB_MODE_IS_ANDROID },
	{ "480p60",   60,  720,  480, 37037,  60,  16, 30, 9, 62, 6, // 27.00 MHz
	  0, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "720p50",   50, 1280,  720, 13468, 440, 220, 5, 20, 40, 5, // 74.25 MHz
	  FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "720p60",   60, 1280,  720, 13468, 110, 220, 5, 20, 40, 5, // 74.25 MHz
	  FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1080p24",  24, 1920, 1080, 13468, 638, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1080p25",  25, 1920, 1080, 13468, 528, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1080p30",  30, 1920, 1080, 13468,  88, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_HOR_HIGH_ACT | FB_SYNC_VERT_HIGH_ACT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1440x480p60",   60, 1440, 480, 18500,  32, 120, 9, 30, 124, 6, // 54.054 MHz
	  0, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1440x480p60a",  60, 1440, 480, 18500,  32, 120, 9, 30, 124, 6, // 54.054 MHz
	  0, FB_VMODE_NONINTERLACED, FB_MODE_IS_ANDROID },
};

#define DWFB_MODE_MAX (DWFB_MODE_HDMI + ARRAY_SIZE(dw74fb_modedb) - 1)

static int bpp_to_info(struct fb_var_screeninfo *var, int bits_per_pixel)
{
	int bpp = -1;

	if (bits_per_pixel == 16) {
		var->bits_per_pixel = 16;
		var->red.offset = 0;
		var->red.length = 5;
		var->green.offset = 5;
		var->green.length = 6;
		var->blue.offset = 11;
		var->blue.length = 5;
		bpp = 0;
	} else if ((bits_per_pixel == 18) || (bits_per_pixel == 24)) {
		var->bits_per_pixel = 24;
		var->red.offset = 0;
		var->red.length = 8;
		var->green.offset = 8;
		var->green.length = 8;
		var->blue.offset = 16;
		var->blue.length = 8;
		bpp = 1;
	}

	return bpp;
}

static void dw74fb_hw_param_update(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_PARUP, 1);
	if (panel->is_cpu_type)
		dw74fb_writel(dw74fb, LCDC_REG_CDISPUPR, 1);
}

static unsigned int calc_line_stride(unsigned int line_length_bytes)
{
	/* LCDC needs each line to be aligned to 64bytes for best bus utilization */
	/* aligne to next 64 bytes (OSDM is doing the same logic) */
	line_length_bytes += 63;
	line_length_bytes &= ~63;
	
	return line_length_bytes;
}

static int set_display_bpp(struct dw74fb *dw74fb, struct fb_info *info)
{
	unsigned int bpp = 0;

	if (info->var.bits_per_pixel == 16) {
		bpp = 0x0;
		/* This specify that this is a video*/
		if (lcdc_mode == DWFB_MODE_VIDEO_PAL || lcdc_mode == DWFB_MODE_VIDEO_NTSC)
			bpp = 0x2;
	} else if (info->var.bits_per_pixel == 24) {
		bpp = 0x1;
	} else if (info->var.bits_per_pixel == 2) {
		bpp = 0x3;
	} else {
		/* TODO: find a way to define YUV422 */
		printk(KERN_ERR "Fail to set bit per pixel value %d is not valid\n", info->var.bits_per_pixel);
		return -1;
	}

	dw74fb_writel(dw74fb,LCDC_REG_INTMR, bpp);

	/* input data transfer sizes */
	dw74fb_writel(dw74fb, LCDC_REG_INDTR, (( info->var.xres * info->var.bits_per_pixel ) / 32 ) - 1);

	dw74fb_writel(dw74fb, LCDC_REG_OFFAR0, calc_line_stride(info->var.xres * info->var.bits_per_pixel / 8));
	dw74fb_writel(dw74fb, LCDC_REG_OFFAR1, calc_line_stride(info->var.xres * info->var.bits_per_pixel / 8));

	return 0;
}

static void
dw74fb_set_display_plane(struct dw74fb *dw74fb)
{
	int xpos = 0, ypos = 0, xend, yend;

	xend = panel->xres_full;
	yend = panel->yres_full;
	if (panel->xres_full > panel->xres)
		xpos = (panel->xres_full - panel->xres)/2;
	if (panel->yres_full > panel->yres)
		ypos = (panel->yres_full - panel->yres)/2;
	if (panel->xres_full > panel->xres)
		xend = panel->xres_full - xpos;
	if (panel->yres_full > panel->yres)
		yend = panel->yres_full - ypos;

	/* display position */
	dw74fb_writel(dw74fb, LCDC_REG_DISPXSPOSR,  xpos);
	dw74fb_writel(dw74fb, LCDC_REG_DISPXEPOSR,  xend - 1);
	dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS1R, ypos);
	dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS1R, yend - 1);
}

static int dw74fb_backlight_update(struct backlight_device *bl);

static int
dw74fb_set_videomode(struct dw74fb *dw74fb, int mode)
{
	struct fb_info *info = dw74fb->dwfb_info.info;
	struct fb_var_screeninfo var = info->var;
	const struct fb_videomode *hdmi_mode = NULL;
	int ret, xres, yres;
	unsigned int addr;
	int bpp = 1;
	int xpos = 0, ypos = 0, xend = panel->xres, yend = panel->yres;
	int android = 0;
	int dispir = 0x0001;
	int pancsr = 0x0007;
	unsigned long clk_rate = 0;
	int panel_active = 0;
	int disploc_set = (dw74fb->dwfb_info.location.x_end || dw74fb->dwfb_info.location.y_end);

	if ((mode < DWFB_MODE_LCD) || (mode > DWFB_MODE_MAX))
		return -1;

	clk_enable(dw74fb->clk);

	if (mode == DWFB_MODE_LCD) {
		/* restore original settings */
		hdmi_mode = &panel->videomode;
		dispir    =  panel->dispir;
		pancsr    =  panel->pancsr;
		clk_rate  =  panel->clock_rate;
		bpp = bpp_to_info(&info->var, panel->bits_per_pixel);
	} else {
		hdmi_mode = &dw74fb_modedb[mode - DWFB_MODE_HDMI];
		xres = hdmi_mode->hsync_len + hdmi_mode->left_margin +
		       hdmi_mode->xres + hdmi_mode->right_margin;
		yres = hdmi_mode->vsync_len + hdmi_mode->upper_margin +
		       hdmi_mode->yres + hdmi_mode->lower_margin;
		clk_rate = xres * yres * hdmi_mode->refresh;
		bpp = bpp_to_info(&info->var, 24);
		if (hdmi_mode->sync & FB_SYNC_VERT_HIGH_ACT)
			pancsr &= ~0x2;
		if (hdmi_mode->sync & FB_SYNC_HOR_HIGH_ACT)
			pancsr &= ~0x4;
	}
	info->mode = (struct fb_videomode *)hdmi_mode;

	if (panel->suspend && panel->resume) {
		char *mode, *token;
		mode = kmalloc(strlen(dw74fb->dwfb_info.info->mode->name) + 1, GFP_KERNEL);
		if (!mode) {
			clk_disable(dw74fb->clk);
			return -1;
		}
		strcpy(mode, dw74fb->dwfb_info.info->mode->name);
		while ((token = strsep(&mode, ",")) != NULL) {
			if (!strcmp(token, panel->name)) {
				panel_active = 1;
				break;
			}
		}
		kfree(mode);
	}

	if (!panel_active) {
		if (panel->suspend)
			panel->suspend(panel);
		if (dw74fb->backlight) {
			dw74fb_writel(dw74fb, LCDC_REG_RBCR, 128);
			dw74fb_writel(dw74fb, LCDC_REG_GBCR, 128);
			dw74fb_writel(dw74fb, LCDC_REG_BBCR, 128);
			dw74fb_hw_param_update(dw74fb);
		}
	}

	if ((mode != DWFB_MODE_LCD) && (hdmi_mode->flag & FB_MODE_IS_ANDROID))
		android = 1;

	dw74fb_writel(dw74fb, LCDC_REG_LCDCCR, 0);
	dw74fb_writel(dw74fb, LCDC_REG_DISPCR, 0);

	fb_videomode_to_var(&var, hdmi_mode);
	if (!android) {
		if (disploc_set) {
			int xres = dw74fb->dwfb_info.location.x_end - dw74fb->dwfb_info.location.x_start + 1;
			int yres = dw74fb->dwfb_info.location.y_end - dw74fb->dwfb_info.location.y_start + 1;

			var.xres = xres;
			var.yres = yres;
			var.yres_virtual = yres;

	#ifdef CONFIG_FB_DW74_DOUBLEBUF
			var.yres_virtual *= 2;
	#endif
			info->fix.line_length = calc_line_stride(xres * (info->var.bits_per_pixel/8));
			info->screen_size = info->fix.line_length * var.yres_virtual;
		}
		else {
			/* hdmi_mode describes the actual image we are displaying */
	#ifdef CONFIG_FB_DW74_DOUBLEBUF
			var.yres_virtual = hdmi_mode->yres*2;
	#endif
			info->fix.line_length = calc_line_stride(hdmi_mode->xres * (info->var.bits_per_pixel/8));
			info->screen_size = info->fix.line_length * var.yres_virtual;
		}

		info->fix.smem_len = info->screen_size;

		var.activate = FB_ACTIVATE_FORCE | FB_ACTIVATE_NOW;
		ret = fb_set_var(info, &var);
		if (ret < 0) {
			printk("set_videomode failed %d\n", ret);
			clk_disable(dw74fb->clk);
			return -1;
		}
	}
	clk_rate = clk_round_rate(dw74fb->clk, clk_rate);
	clk_set_rate(dw74fb->clk, clk_rate);

	dw74fb_writel(dw74fb, LCDC_REG_DISPIR, dispir);
	dw74fb_writel(dw74fb, LCDC_REG_PANCSR, pancsr);

	dw74fb_writel(dw74fb, LCDC_REG_INTMR,  bpp);

	dw74fb_writel(dw74fb, LCDC_REG_VSTR,   hdmi_mode->vsync_len - 1);
	dw74fb_writel(dw74fb, LCDC_REG_VFTR,   hdmi_mode->upper_margin - 1);
	dw74fb_writel(dw74fb, LCDC_REG_VATR,   hdmi_mode->yres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_VETR,   hdmi_mode->lower_margin - 1);

	dw74fb_writel(dw74fb, LCDC_REG_HSTR,   hdmi_mode->hsync_len - 1);
	dw74fb_writel(dw74fb, LCDC_REG_HFTR,   hdmi_mode->left_margin - 1);
	dw74fb_writel(dw74fb, LCDC_REG_HADSTR, hdmi_mode->xres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_HAPWR,  hdmi_mode->xres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_HETR,   hdmi_mode->right_margin - 1);

	if (mode == DWFB_MODE_LCD) {
		dw74fb_writel(dw74fb, LCDC_REG_VATR,   panel->yres_full - 1);
		dw74fb_writel(dw74fb, LCDC_REG_HADSTR, panel->xres_full - 1);
		dw74fb_writel(dw74fb, LCDC_REG_HAPWR,  panel->xres_full - 1);
	}

	if (!android) {
		if (disploc_set)
			dw74fb_writel(dw74fb, LCDC_REG_INDTR, ((var.xres * info->var.bits_per_pixel) / 32 ) - 1);
		else
			dw74fb_writel(dw74fb, LCDC_REG_INDTR, ((hdmi_mode->xres * info->var.bits_per_pixel) / 32 ) - 1);
	}


	if (disploc_set) {
		dw74fb_writel(dw74fb, LCDC_REG_INDXSR, var.xres - 1);
		dw74fb_writel(dw74fb, LCDC_REG_INDYSR, var.yres - 1);
	}
	else {
		dw74fb_writel(dw74fb, LCDC_REG_INDXSR, hdmi_mode->xres - 1);
		dw74fb_writel(dw74fb, LCDC_REG_INDYSR, hdmi_mode->yres - 1);
	}

	if (mode == DWFB_MODE_LCD) {
		dw74fb_set_display_plane(dw74fb);
	}
	else {
		if (android && (hdmi_mode->xres > panel->xres))
			xpos = (hdmi_mode->xres - panel->xres)/2;
		else if (disploc_set)
			xpos = dw74fb->dwfb_info.location.x_start;

		if (android && (hdmi_mode->yres > panel->yres))
			ypos = (hdmi_mode->yres - panel->yres)/2;
		else if (disploc_set)
			ypos = dw74fb->dwfb_info.location.y_start;

		if (android && (hdmi_mode->xres > panel->xres))
			xend = hdmi_mode->xres - xpos - 1;
		else if (disploc_set)
			xend = dw74fb->dwfb_info.location.x_end;
		else
			xend = hdmi_mode->xres - 1;

		if (android && (hdmi_mode->yres > panel->yres))
			yend = hdmi_mode->yres - ypos - 1;
		else if (disploc_set)
			yend = dw74fb->dwfb_info.location.y_end;
		else
			yend = hdmi_mode->yres - 1;

		dw74fb_writel(dw74fb, LCDC_REG_DISPXSPOSR,  xpos);
		dw74fb_writel(dw74fb, LCDC_REG_DISPXEPOSR,  xend);
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS1R, ypos);
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS1R, yend);
	}

	if (!android) {
		/* input buffer */
		dw74fb_writel(dw74fb, LCDC_REG_OFFAR0, info->fix.line_length);
#ifdef CONFIG_FB_DW74_DOUBLEBUF
		dw74fb_writel(dw74fb, LCDC_REG_OFFAR1, info->fix.line_length);
#endif
	}

	addr = info->fix.smem_start + (info->screen_size / 2);
	dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA1R, addr >> 16);
	dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA1R, addr & 0xffff);

	dw74fb_writel(dw74fb, LCDC_REG_DISPIDXR, 0);

	dw74fb_hw_param_update(dw74fb);

	dw74fb_writel(dw74fb, LCDC_REG_INTR,   0x0001);
	dw74fb_writel(dw74fb, LCDC_REG_DISPCR, 0x0001);
	dw74fb_writel(dw74fb, LCDC_REG_LCDCCR, 0x0001);

	if (panel_active) {
		if (panel->resume)
			panel->resume(panel);
		if (dw74fb->backlight)
			dw74fb_backlight_update(dw74fb->backlight);
	}

	clk_disable(dw74fb->clk);

	return 0;
}

/*
 * hardware specifics
 */
unsigned long
dw74fb_readl(struct dw74fb *dw74fb, unsigned long addr)
{
	return readl(dw74fb->regs + addr);
}

void
dw74fb_writel(struct dw74fb *dw74fb, unsigned long addr, unsigned long val)
{
	writel(val, dw74fb->regs + addr);
}

static void dw74fb_hw_enable(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_LCDCCR, 1);
}

static void dw74fb_hw_disable(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_LCDCCR, 0);
}

static void
dw74fb_run_sequence(struct dw74fb *dw74fb, struct dw74fb_seq *seq)
{
	struct dw74fb_seq *cmd = seq;

	while (cmd->addr != LCDC_SEQ_END) {
		dw74fb_writel(dw74fb, cmd->addr, cmd->value);
		cmd++;
	}
}

void lcdc_own_hw(void)
{
	if (g_dw74fb != NULL){
		if (down_interruptible(&(g_dw74fb->lcdc_hw_busy)) != 0){
				printk(KERN_ERR "ili9325 LCD operation since it was signaled while waiting for frame to be finished\n");
		}
	}
}

void lcdc_release_hw(void)
{
	if (g_dw74fb != NULL)
		up(&g_dw74fb->lcdc_hw_busy);
}

#ifdef FB_DW74_PANEL_ILI9325
static struct platform_device dw_ili9325_device = {
	.name		= "dw-ili9325",
	.dev		= {
		.platform_data     = &g_dw74fb,
	},
};
#endif

static void set_hw_init_seq(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_INTER,  0x05);  /* enable underrun & fifo empty irqs */
	dw74fb_writel(dw74fb, LCDC_REG_GCER,   0);     /* disable gamma correction */
	dw74fb_writel(dw74fb, LCDC_REG_CLPER,  0);     /* disable clipping */
	dw74fb_writel(dw74fb, LCDC_REG_BACKCPR, 0);    /* black background */
	/* Background color and brightness */
	dw74fb_writel(dw74fb, LCDC_REG_YCLPCR,   0xff00);
	dw74fb_writel(dw74fb, LCDC_REG_CCLPCR,   0xff00);
	dw74fb_writel(dw74fb, LCDC_REG_DISPIDXR, 0x0000);
	dw74fb_writel(dw74fb, LCDC_REG_GPSELR, 0); /* legacy mode */
	dw74fb_writel(dw74fb, LCDC_REG_PARUP,  1); /* update parameters on next frame */
}

static void set_hw_init_seq_cputype_ili9325(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_INTER,  0x02);  /* frame display done */
	dw74fb_writel(dw74fb, LCDC_REG_GCER,   0);     /* disable gamma correction */
	dw74fb_writel(dw74fb, LCDC_REG_CLPER,  0);     /* disable clipping */
	dw74fb_writel(dw74fb, LCDC_REG_BACKCPR, 0);    /* black background */
	/* Background color and brightness */
	dw74fb_writel(dw74fb, LCDC_REG_YCLPCR,   0xff00);
	dw74fb_writel(dw74fb, LCDC_REG_CCLPCR,   0xff00);
	dw74fb_writel(dw74fb, LCDC_REG_DISPIDXR, 0x0000);
	dw74fb_writel(dw74fb, LCDC_REG_LCDCCR, 0x02); /* FIF0 Reset */
	dw74fb_writel(dw74fb, LCDC_REG_DISPIR, 0x24); /* Set CPU Type, 80 Type, 16 bit RGB */
	dw74fb_writel(dw74fb, LCDC_REG_CTLTR0, 0x1000); /*Set RS Polarity and Data Format, Note: RS is active LOW for ili9325 */
	dw74fb_writel(dw74fb, LCDC_REG_CMDFSR, 0x06); /* Command Fifo Size */
	dw74fb_writel(dw74fb, LCDC_REG_GPSELR, 0); /* legacy mode */
	dw74fb_writel(dw74fb, LCDC_REG_PARUP,  1); /* update parameters on next frame */
}

static void set_display_location(struct dw74fb *dw74fb, struct dwfb_location* location)
{
	/* display position */
	dw74fb_writel(dw74fb, LCDC_REG_DISPXSPOSR,  location->x_start);
	dw74fb_writel(dw74fb, LCDC_REG_DISPXEPOSR,  location->x_end - 1);

	if (lcdc_mode == DWFB_MODE_VIDEO_PAL){
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS1R, 23 + (location->y_start / 2));
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS2R, 336 + location->y_start / 2 );
	} else if (lcdc_mode == DWFB_MODE_VIDEO_NTSC){
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS1R, 20 + (location->y_start / 2));
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS2R, 283 + location->y_start / 2 );
	} else  {
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS1R, location->y_start);
	}

	if (lcdc_mode == DWFB_MODE_VIDEO_PAL){
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS1R, 23 + location->y_end / 2  - 1);
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS2R, 336 + location->y_end / 2  - 1);
	}
	else if (lcdc_mode == DWFB_MODE_VIDEO_NTSC) {
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS1R, 20 + location->y_end / 2  - 1);
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS2R, 283 + location->y_end / 2  - 1);
	}
	else{
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS1R, location->y_end - 1);
	}
}

static void set_display_background(struct dw74fb *dw74fb, unsigned short background_color)
{
	dw74fb_writel(dw74fb,LCDC_REG_BACKCPR, background_color);
}

static void display_palette_set(struct dw74fb *dw74fb, const struct fb_cmap* pal){

	dw74fb_writel(dw74fb,LCDC_REG_PALR0 , get_rgb565(pal->red[0],pal->green[0], pal->blue[0]) );
	dw74fb_writel(dw74fb,LCDC_REG_PALR1 , get_rgb565(pal->red[1],pal->green[1], pal->blue[1]) );
	dw74fb_writel(dw74fb,LCDC_REG_PALR2 , get_rgb565(pal->red[2],pal->green[2], pal->blue[2]) );
	dw74fb_writel(dw74fb,LCDC_REG_PALR3 , get_rgb565(pal->red[3],pal->green[3], pal->blue[3]) );
}

#ifdef CONFIG_FB_DMW96_CURSOR
static void cursor_palette_set(struct dw74fb *dw74fb, const struct fb_cmap* pal){

	dw74fb_writel(dw74fb,LCDC_REG_CURPALR1 , get_rgb565(pal->red[0],pal->green[0], pal->blue[0]) );
	dw74fb_writel(dw74fb,LCDC_REG_CURPALR2 , get_rgb565(pal->red[1],pal->green[1], pal->blue[1]) );
	dw74fb_writel(dw74fb,LCDC_REG_CURPALR3 , get_rgb565(pal->red[2],pal->green[2], pal->blue[2]) );

	/*
	printk (KERN_ERR "%s(%d) 0x%x 0x%x 0x%x \n",__FUNCTION__, __LINE__, pal->red[0] ,pal->green[0], pal->blue[0]);
	printk (KERN_ERR "%s(%d) 0x%x 0x%x 0x%x \n",__FUNCTION__, __LINE__, pal->red[1] ,pal->green[1], pal->blue[1]);
	printk (KERN_ERR "%s(%d) 0x%x 0x%x 0x%x \n",__FUNCTION__, __LINE__, pal->red[2], pal->green[2], pal->blue[2]);
	*/
}

static void lcdc_cursor_bitmap_set(struct dw74fb *dw74fb, const struct cursor_bitmap* bitmap)
{
	int i = 0;

	for (i = 0 ; i < LCDC_CURSOR_BITMAP_ARRAY_SIZE ; i++ ) {
		dw74fb_writel(dw74fb, LCDC_REG_CURBITMAP + (i * 4) , bitmap->bitmap_array[i] );
	}
	dw74fb_writel(dw74fb, LCDC_REG_CURCR1, bitmap->bitmap_size);
	/*
	for (i = 0 ; i < LCDC_CURSOR_BITMAP_ARRAY_SIZE ; i++ ) {
		printk (KERN_ERR "%s(%d) \n" ,__FUNCTION__, __LINE__);
		printk (KERN_ERR "bitmap[%i] = 0x%x \n" , i , bitmap->bitmap_array[i] );
	}
	printk (KERN_ERR "%s(%d) bitmap size %s\n" ,__FUNCTION__, __LINE__ , bitmap->bitmap_size ? "64x64" : "32x32");
	*/
}

static void lcdc_cursor_bitmap_get(struct dw74fb *dw74fb, struct cursor_bitmap* bitmap)
{
	int i = 0;
	for (i = 0 ; i < LCDC_CURSOR_BITMAP_ARRAY_SIZE ; i++ ) {
		bitmap->bitmap_array[i] = dw74fb_readl(dw74fb, LCDC_REG_CURBITMAP + (i * 4));
	}
	bitmap->bitmap_size = dw74fb_readl(dw74fb, LCDC_REG_CURCR1);
}

static void lcdc_cursor_plane_location_set(struct dw74fb *dw74fb, struct dwfb_location* location){

	unsigned short y_start = location->y_start & 0xFFFE;
	dw74fb_writel(dw74fb, LCDC_REG_CURXSPR, location->x_start);
	dw74fb_writel(dw74fb, LCDC_REG_CURYSPR, y_start);
	/*printk (KERN_ERR "%s(%d) x = %d y = %d\n",__FUNCTION__, __LINE__,location->x_start , y_start);*/
}

static void lcdc_cursor_plane_location_get(struct dw74fb *dw74fb, struct dwfb_location* location)
{
	location->x_start = dw74fb_readl(dw74fb, LCDC_REG_CURXSPR);
	location->y_start = dw74fb_readl(dw74fb, LCDC_REG_CURYSPR);
}

static void lcdc_cursor_plane_set(struct dw74fb *dw74fb, const struct fb_info *cursor)
{
	dw74fb_writel(dw74fb, LCDC_REG_CURCR0, (cursor->var.nonstd)? 1 : 0 );
	/*printk (KERN_ERR "%s(%d) Cursor status = %s \n",__FUNCTION__, __LINE__,(cursor->var.nonstd)? "On" : "Off" );*/
}

#endif

static void lcdc_set_panel_size(struct dw74fb *dw74fb, const struct fb_info *info)
{
	dw74fb_writel(dw74fb, LCDC_REG_INDXSR, info->var.xres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_INDYSR, info->var.yres - 1);
}

static void dw74fb_hw_init(struct dw74fb *dw74fb)
{
	struct fb_info *info = dw74fb->dwfb_info.info;
	dma_addr_t addr = info->fix.smem_start;

#ifdef CONFIG_MACH_VERSATILE_BROADTILE
	addr -= 0x80000000;
#endif

	/* input data transfer sizes */
	dw74fb_writel(dw74fb, LCDC_REG_INDXSR, panel->xres_full - 1);
	dw74fb_writel(dw74fb, LCDC_REG_INDYSR, panel->yres_full - 1);

	/* display position */
	dw74fb_set_display_plane(dw74fb);

	/* input buffer */
	dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA0R, addr >> 16);
	dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA0R, addr & 0xffff);

#ifdef CONFIG_FB_DW74_DOUBLEBUF
	addr += (info->var.yres * info->fix.line_length);
	dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA1R, addr >> 16);
	dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA1R, addr & 0xffff);
#else
	dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA1R, 0);
	dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA1R, 0);
#endif

	/* if cpu type, there must be special initialization in panel->init_seq; else, run default init sequence */
	if (!panel->is_cpu_type) {
		set_hw_init_seq(dw74fb);
	}

	if (panel->init_seq) {
		dw74fb_run_sequence(dw74fb, panel->init_seq);
	}

	dw74fb_hw_param_update(dw74fb);

	set_display_bpp(dw74fb, info);
	dw74fb_set_videomode(dw74fb, DWFB_MODE_LCD);
	dw74fb_hw_disable(dw74fb);

	dw74fb_hw_param_update(dw74fb);
	dw74fb_hw_enable(dw74fb);
	
	/* run panel init */
	if (panel->cputype_desc) {
		panel->cputype_desc->init_regs(dw74fb);
	}

#ifdef FB_DW74_PANEL_ILI9325
	platform_device_register(&dw_ili9325_device);
#endif
}

static void dw74fb_hw_param_update_sync(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_PARUP, 1);
	mb();
	dw74fb->is_update_sync = 1;

	if (panel->is_cpu_type)
		printk("Error should happen\n");

	/* wait for this interrupt to be sure that the lcdc updated the registers */
	if (down_timeout(&dw74fb->parameter_update_sync, 4) != 0)
		printk(KERN_WARNING "[%s:%s] parameters update timeout\n", __FILE__, __FUNCTION__);

	if (dw74fb_readl(dw74fb, LCDC_REG_PARUP) & 0x1)
		printk(KERN_WARNING "[%s:%s] lcdc parameters not updated yet!\n", __FILE__, __FUNCTION__);
}

static int get_bpp(struct dw74fb *dw74fb)
{
	int bpp = 0;

	switch (dw74fb_readl(dw74fb, LCDC_REG_INTMR)) {
		case 0:
			bpp = 16;
			break;
		case 1:
			bpp = 24;
			break;
		/* YUV422 and 2bpp are not supported */
	}

	return bpp;
}

static void convert_16bpp_to_24bpp(unsigned char *dest, unsigned short *src, unsigned int xres, unsigned int yres, unsigned int stride)
{
	int r;
	int g;
	int b;

	while (yres--) {
		unsigned int num_pixels = xres;
		unsigned char *orig_dest = dest;
		while (num_pixels--) {
			b = *src & 0x001f;
			g = (*src & 0x07e0) >> 5;
			r = (*src & 0xf800) >> 11;
	
			dest[0] = b << 3;
			dest[1] = g << 2;
			dest[2] = r << 3;
	
			src++;
			dest += 3;
		}
		dest = orig_dest + stride;
	}
}

static void update_refresh_address(struct dw74fb *dw74fb, int index, dma_addr_t addr, int start_refresh_new_addr)
{
	struct fb_info *info = dw74fb->dwfb_info.info;

	if (index == 0) {
		dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA0R, addr >> 16);
		dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA0R, addr & 0xffff);
		dw74fb_writel(dw74fb, LCDC_REG_OFFAR0,    info->fix.line_length);
	} else {
		dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA1R, addr >> 16);
		dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA1R, addr & 0xffff);
		dw74fb_writel(dw74fb, LCDC_REG_OFFAR1,    info->fix.line_length);
	}

	if (start_refresh_new_addr) {
		dw74fb_writel(dw74fb, LCDC_REG_DISPIDXR, index);
		if (panel->is_cpu_type)
			dw74fb_hw_param_update(dw74fb);
		else
			dw74fb_hw_param_update_sync(dw74fb);
	}
}

static void copy_with_stride(void *dst, const void *src, unsigned int xsize, unsigned int yres, unsigned int stride)
{
	unsigned int dst_int = (unsigned int)dst;
	unsigned int src_int = (unsigned int)src;
			
	while (yres) {
		memcpy((void*)dst_int, (const void*)src_int, xsize);
		dst_int += stride;
		src_int += xsize;
		yres--;
	}
}

static int dw74fb_hw_try_resume(struct dw74fb *dw74fb)
{
	struct fb_info *info = dw74fb->dwfb_info.info;
	dma_addr_t addr1 = info->fix.smem_start + (info->var.yres * info->fix.line_length);
	dma_addr_t old;
	int running = dw74fb_readl(dw74fb, LCDC_REG_LCDCCR) & 1;
	void *old_virt;
	int old_bpp = get_bpp(dw74fb);
	int new_bpp = info->var.bits_per_pixel == 16 ? 16 : 24;
	int xres, yres;
	
	if (!running)
		return 0;

	/* same screen resolution? */
	xres = dw74fb_readl(dw74fb, LCDC_REG_HADSTR) + 1;
	yres = dw74fb_readl(dw74fb, LCDC_REG_VATR) + 1;
	if ((xres != panel->xres) || (yres != panel->yres))
		return 0;

	/* copy over the contents of the framebuffer to the new location */
	old = dw74fb_readl(dw74fb, LCDC_REG_LSBAHBA0R) |
	      dw74fb_readl(dw74fb, LCDC_REG_MSBAHBA0R) << 16;
	old_virt = ioremap(old, info->screen_size/2);
	if (!old_virt)
		return 0;

	if (old_bpp == new_bpp) {
		/* simple copy. u-boot has only one buffer, so we copy it to both our buffers */
		copy_with_stride(info->screen_base, old_virt, xres*new_bpp/8, yres, info->fix.line_length);
#ifdef CONFIG_FB_DW74_DOUBLEBUF
		copy_with_stride(info->screen_base + info->screen_size/2, old_virt, xres*new_bpp/8, yres, info->fix.line_length);
#endif
	} else	if (new_bpp == 24 && old_bpp == 16) {
		/* convert the image while copying */
		convert_16bpp_to_24bpp(info->screen_base, old_virt, xres, yres ,info->fix.line_length);
#ifdef CONFIG_FB_DW74_DOUBLEBUF
		convert_16bpp_to_24bpp(info->screen_base + info->screen_size/2, old_virt, xres, yres, info->fix.line_length);
#endif
	} else {
		memset(info->screen_base, 0, info->screen_size);
	}

	iounmap(old_virt);

	/*
	 * the next code is ugly, but with a reason :-)
	 * PURAP mechanism is not working on OFFARX, MSBAHBAXR and LSBAHBXR registers!
	 * Thus, writing to any of these registers will cause a flickering on the screen (while refreshing from address X)
	 * Solution:
	 *		If we are refreshing from address 0:
	 *			Update address1, and switch to it
	 *			Then update address0, and switch back to it
	 *		If we are refreshing from address 1:
	 *			Update address0, and switch to it
	 *			Then update address1, and don't switch back to it!
	 *
	 * Note: rest of the code assumes we are on address0 at boot time!
	 */
	dw74fb_writel(dw74fb,LCDC_REG_INTMR, new_bpp == 24 ? 1 : 0);
	dw74fb_writel(dw74fb, LCDC_REG_INDTR, ((info->var.xres * info->var.bits_per_pixel ) / 32 ) - 1);

	if (dw74fb_readl(dw74fb, LCDC_REG_DISPIDXR) == 0) {
		update_refresh_address(dw74fb, 1, addr1, 1);
		update_refresh_address(dw74fb, 0, info->fix.smem_start, 1);
	}
	else {
		update_refresh_address(dw74fb, 0, info->fix.smem_start, 1);
		update_refresh_address(dw74fb, 1, addr1, 0);
	}

	return 1;
}

/*
 * FIXME: uglyness, quick integration into DW branch for DW74 chip
 */
#include <mach/hardware.h>
#ifdef CONFIG_FB_DMW96_OSDM
void lcdc_enable_vsync_int_once(void)
{
	g_dw74fb->osdm_vsync_enb = 1;
}

void lcdc_disable_vsync_interrupt(void)
{
	g_dw74fb->osdm_vsync_enb = 0;
}

#define CPUTYPE_VSYNC_TIME 70
static void cputype_vsync_callback( unsigned long data )
{
	int ret = 0;
	struct dw74fb *dw74fb = (struct dw74fb *)data;

	mod_timer( &dw74fb->cputype_vsync_timer, jiffies + msecs_to_jiffies(CPUTYPE_VSYNC_TIME) );
#ifdef CONFIG_FB_DMW96_OSDM
	if (dw74fb->osdm_vsync_enb && dw74fb->osdm_vsync_cb) {
		lcdc_disable_vsync_interrupt();
		dw74fb->osdm_vsync_cb();
	}
#endif
	if (ret)
		printk( "FAIL restarting timer\n");
}

int get_osdm_parameters_from_lcdc(struct fb_info* display , int (*osdm_vsync_cb)(void))
{
	int res =0;
	int non_active_index = 0;

	display->var.xres = g_dw74fb->dwfb_info.info->var.xres;
	display->var.yres = g_dw74fb->dwfb_info.info->var.yres;
	display->var.bits_per_pixel = g_dw74fb->dwfb_info.info->var.bits_per_pixel;
#ifdef CONFIG_FB_DMW96_OSDM
	display->var.reserved[0] = panel->osdm_config;
#endif

	if (g_dw74fb != NULL) {
		clk_enable(g_dw74fb->clk);
		non_active_index = dw74fb_readl(g_dw74fb, LCDC_REG_DISPIDXR) == 1 ? 0 : 1;
		clk_disable(g_dw74fb->clk);
		display->fix.smem_start = g_dw74fb->dwfb_info.info->fix.smem_start +
			(non_active_index * g_dw74fb->dwfb_info.info->var.yres * g_dw74fb->dwfb_info.info->fix.line_length);
		#ifdef CONFIG_MACH_VERSATILE_BROADTILE
		display->fix.smem_start = 0x60000000;
		#endif
	}
	else {
		printk("%s unable to hand display info to OSDM devide was not initiated\n", __FUNCTION__);
		res = -ENOMEM;
	}

	if (osdm_vsync_cb)
		g_dw74fb->osdm_vsync_cb = osdm_vsync_cb;
	else
		res = -ENOMEM;

	printk("LCDC(0)=0x%lx LCDC(1)=0x%lx non_active_index=%d\n",g_dw74fb->dwfb_info.info->fix.smem_start, display->fix.smem_start , non_active_index);
	return res;
}


int switch_osdm_and_lcdc_address(struct fb_info* display)
{
	unsigned long index_lcdc=0x0;

	if (panel->is_cpu_type && (!down_trylock(&g_dw74fb->lcdc_hw_busy))) {
		return 0;
	}

	/* if screen should be prepared for next */
	if (g_dw74fb->panel->cputype_desc) {
		g_dw74fb->panel->cputype_desc->prepare_screen(g_dw74fb);
	}

	index_lcdc = dw74fb_readl(g_dw74fb, LCDC_REG_DISPIDXR);

	/*Switch the OSDM address to LCDC address*/
	display->fix.smem_start = g_dw74fb->dwfb_info.info->fix.smem_start + (index_lcdc * g_dw74fb->dwfb_info.info->fix.smem_len / 2);
	
	if (index_lcdc == 0x1)
		index_lcdc = 0x0;
	else
		index_lcdc = 0x1;

	dw74fb_writel(g_dw74fb,LCDC_REG_DISPIDXR, index_lcdc );

	dw74fb_hw_param_update(g_dw74fb);
	
	return 0;
}


EXPORT_SYMBOL(get_osdm_parameters_from_lcdc);
EXPORT_SYMBOL(switch_osdm_and_lcdc_address);
EXPORT_SYMBOL(lcdc_enable_vsync_int_once);
#endif

/*
 * driver implementation
 */
#ifdef CONFIG_FB_DW74_LOW_POWER
static void preper_to_pan(struct fb_var_screeninfo *var, struct fb_info *info)
{
	struct dw74fb *dw74fb = info->par;
	struct dwfb_info *l_dwfb_info = NULL;

	if (dw74fb->dwfb_info.info == info) {
		l_dwfb_info = &(dw74fb->dwfb_info);
	}
	else if (dw74fb->dwfb_info_low_power.info == info){
		l_dwfb_info = &(dw74fb->dwfb_info_low_power);
	}
	else {
		printk(KERN_ERR "Fail to handle IOCT invalide frame buffer\n");
		return;
	}

	set_display_location(dw74fb, &(l_dwfb_info->location));
	set_display_background(dw74fb, l_dwfb_info->background_color);
	set_display_bpp(dw74fb, info);
}
#endif

/*
 * this function will only be called if CONFIG_FB_DW74_DOUBLEBUF is set
 */
static int
dw74fb_pan_display(struct fb_var_screeninfo *var, struct fb_info *info)
{
#ifdef CONFIG_FB_DMW96_OSDM
	/* we don't support panning when OSDM is used */
	return 0;
#else
	struct dw74fb *dw74fb = info->par;
	dma_addr_t addr;
	dma_addr_t offs = var->yoffset * info->fix.line_length;

#ifdef CONFIG_FB_DW74_LOW_POWER
	preper_to_pan(var,info);
#endif

	if (panel->is_cpu_type)
		lcdc_own_hw();

	addr = info->fix.smem_start + offs;

	dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA0R, addr >> 16);
	dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA0R, addr & 0x0000ffff);

	/* Attention if not defined that we are working with v-sync which is connected to external interrupts
	then we need to update the parameters of the LCDC other wise we need will update them in the interrupt
	handler of the v-sync.
	*/
	if (!panel->is_cpu_type)
		dw74fb_hw_param_update_sync(dw74fb);
	else
		dw74fb_hw_param_update(dw74fb);

	return 0;
#endif
}

static u32 pseudo_palette[16];
#ifdef CONFIG_FB_DW74_LOW_POWER
static u32 pseudo_palette_low_power[16];
#endif

static int dw74fb_setcolreg(unsigned regno, unsigned red, unsigned green,
                          unsigned blue, unsigned transp, struct fb_info *info)
{
	u32 *palette = info->pseudo_palette;

	if (regno >= 16)
		return -EINVAL;

	/* only FB_VISUAL_TRUECOLOR supported */

	red >>= 16 - info->var.red.length;
	green >>= 16 - info->var.green.length;
	blue >>= 16 - info->var.blue.length;
	transp >>= 16 - info->var.transp.length;

	palette[regno] = (red << info->var.red.offset) |
	  (green << info->var.green.offset) |
	  (blue << info->var.blue.offset) |
	  (transp << info->var.transp.offset);

	return 0;
}

static int
dw74fb_check_var(struct fb_var_screeninfo *var, struct fb_info *info)
{
	/* For now, don't check anything... */
	return 0;
}

static int dw74fb_ioctl(struct fb_info *info, unsigned int cmd,unsigned long arg)
{
	void __user *argp = (void __user *)arg;
	struct dw74fb *dw74fb = info->par;
	struct dwfb_info *l_dwfb_info = NULL;
	unsigned int clock_rate = 0;
#ifdef CONFIG_FB_DMW96_CURSOR
	struct cursor_bitmap cursor_bit;
#endif

	if (dw74fb->dwfb_info.info == info) {
		l_dwfb_info = &(dw74fb->dwfb_info);
	}
#ifdef CONFIG_FB_DW74_LOW_POWER
	else if (dw74fb->dwfb_info_low_power.info == info){
		l_dwfb_info = &(dw74fb->dwfb_info_low_power);
	}
#endif
#ifdef CONFIG_FB_DMW96_CURSOR
	else if (dw74fb->dwfb_info_cursor.info == info){
		l_dwfb_info = &(dw74fb->dwfb_info_cursor);
	}
#endif
	else {
		printk(KERN_ERR "Fail to handle IOCT invalide frame buffer\n");
		return -1;
	}

	switch (cmd) {
		case FBIOSET_BACKGROUND:
			if (copy_from_user(&(l_dwfb_info->background_color), argp, sizeof(unsigned short)) )
				return -EFAULT;
			set_display_background(dw74fb,l_dwfb_info->background_color);
			break;

		case FBIOSET_DISPLAY_LOCATION:
			if (copy_from_user(&(l_dwfb_info->location), argp, sizeof(struct dwfb_location)) )
				return -EFAULT;
			if (l_dwfb_info->location.x_start > l_dwfb_info->location.x_end ||
				l_dwfb_info->location.y_start > l_dwfb_info->location.y_end ) {

				printk(KERN_ERR "Fail to handle IOCT invalide location\n");
				return -EFAULT;
			}
			set_display_location(dw74fb,&(l_dwfb_info->location));
			break;

		case FBIOSET_DISPLAY_CLOCK_RATE:
			if (copy_from_user(&(clock_rate), argp, sizeof(unsigned int)) )
				return -EFAULT;

			clk_set_rate(dw74fb->clk, clk_round_rate(dw74fb->clk, clock_rate));
			break;

		case FBIOSET_DISPLAY_MODE:
			if (copy_from_user(&(lcdc_mode), argp, sizeof(unsigned int)) )
				return -EFAULT;
			dw74fb_set_videomode(dw74fb, lcdc_mode);

			dw74fb->event.info = dw74fb->dwfb_info.info;
			if (fb_notifier_call_chain(FB_EVENT_MODE_CHANGE_ALL, &dw74fb->event) < 0)
				return -EFAULT;

			break;

#ifdef CONFIG_FB_DMW96_CURSOR
		case FBIOSET_LCDC_CURSOR_BITMAP:
			if (copy_from_user(&cursor_bit, argp, sizeof(struct cursor_bitmap)) )
				return -EFAULT;
			lcdc_cursor_bitmap_set(dw74fb, &cursor_bit);
			break;
		case FBIOGET_LCDC_CURSOR_BITMAP:
			lcdc_cursor_bitmap_get(dw74fb, &cursor_bit);
			if (copy_to_user(argp, &cursor_bit , sizeof(struct cursor_bitmap)) )
				return -EFAULT;
			break;
		case FBIOSET_LCDC_CURSOR_LOCATION:
			if (copy_from_user(&(l_dwfb_info->location), argp, sizeof(struct dwfb_location)) )
				return -EFAULT;
			lcdc_cursor_plane_location_set(dw74fb,&(l_dwfb_info->location));
			break;
		case FBIOGET_LCDC_CURSOR_LOCATION:
			lcdc_cursor_plane_location_get(dw74fb,&(l_dwfb_info->location));
			if (copy_to_user(argp, &(l_dwfb_info->location) , sizeof(struct dwfb_location)) )
				return -EFAULT;
			break;
#endif
	}

	return 0;
}

static int dw74fb_cmap(struct fb_cmap *cmap, struct fb_info *info)
{
	int res = 0;
	struct dw74fb *dw74fb = info->par;

#ifdef CONFIG_FB_DMW96_CURSOR
	if (dw74fb->dwfb_info_cursor.info == info) {
		cursor_palette_set(dw74fb, cmap);
	}
#endif

	display_palette_set(dw74fb, cmap);
	return res;
}

int dw74fb_set_par(struct fb_info *info)
{
	int res = 0;
	struct dw74fb *dw74fb = info->par;

	#ifdef CONFIG_FB_DMW96_CURSOR
	if (info == dw74fb->dwfb_info_cursor.info) {
		lcdc_cursor_plane_set(dw74fb,info);
	}
	#endif

	/* only cursor is manipulated when in HDMI mode */
	if (lcdc_mode >= DWFB_MODE_HDMI)
		return res;

	#ifdef CONFIG_FB_DMW96_CURSOR
	if (info != dw74fb->dwfb_info_cursor.info)
	#endif
		res = set_display_bpp(dw74fb, info);

	if (info == dw74fb->dwfb_info.info) {
		lcdc_set_panel_size(dw74fb,info);
		if (lcdc_mode == DWFB_MODE_VIDEO_PAL) {

			dw74fb_writel(dw74fb, LCDC_REG_DISPIR,  0x0061);
			dw74fb_writel(dw74fb, LCDC_REG_VATR, (unsigned int) 624);
			dw74fb_writel(dw74fb, LCDC_REG_HSTR,(unsigned int) 279);

		} else if (lcdc_mode == DWFB_MODE_VIDEO_NTSC) {

			dw74fb_writel(dw74fb, LCDC_REG_DISPIR,  0x00e1);
			dw74fb_writel(dw74fb, LCDC_REG_VATR, (unsigned int) 524);
			dw74fb_writel(dw74fb, LCDC_REG_HSTR,(unsigned int) 267);
		} else if (lcdc_mode == DWFB_MODE_LCD) {

			/* if cpu type, there must be special initialization in panel->init_seq; else, run default init sequence */
			if (!panel->is_cpu_type) {
				set_hw_init_seq(dw74fb);
			}

			if (panel->init_seq) {
				dw74fb_run_sequence(dw74fb, panel->init_seq);
			}
		}
	}

	return res;
}

static struct fb_ops dw74fb_ops = {
	.owner          = THIS_MODULE,
	.fb_pan_display = dw74fb_pan_display,
	.fb_check_var   = dw74fb_check_var,
	.fb_ioctl       = dw74fb_ioctl,
	.fb_fillrect    = cfb_fillrect,
	.fb_copyarea    = cfb_copyarea,
	.fb_imageblit   = cfb_imageblit,
	.fb_setcolreg   = dw74fb_setcolreg,
	.fb_setcmap     = dw74fb_cmap,
	.fb_set_par     = dw74fb_set_par,
};

static irqreturn_t dw74fb_irq(int irq, void *priv)
{
	struct dw74fb *dw74fb = priv;
	int stat;

	/* read & clear active interrupts */
	stat = dw74fb_readl(dw74fb, LCDC_REG_INTSR);
	dw74fb_writel(dw74fb, LCDC_REG_INTSR, stat);

	if (panel->is_cpu_type) {
		if (stat & 0x2) {
			/*Frame display done*/
			lcdc_release_hw();
		}
	} else {
		if (stat & 0x1) {
			/* release parameters update semaphore and disable the interrupt */
			//val = dw74fb_readl(dw74fb, LCDC_REG_INTER);
			//dw74fb_writel(dw74fb, LCDC_REG_INTER, val & (~0x1));
#ifdef CONFIG_FB_DMW96_OSDM
			if (dw74fb->osdm_vsync_enb && dw74fb->osdm_vsync_cb) {
				//printk("%s(%d) \n",__FUNCTION__,__LINE__);
				lcdc_disable_vsync_interrupt();
				dw74fb->osdm_vsync_cb();
			}
#endif
			if (dw74fb->is_update_sync) {
				dw74fb->is_update_sync = 0;
				up(&dw74fb->parameter_update_sync);
			}

			return IRQ_HANDLED;
		}
	}

	/* handle interrupts */
	if (stat & 0x4) {
		underruns++;
		return IRQ_HANDLED;
	}

	unhandled_ints++;

	return IRQ_HANDLED;
}

#ifdef CONFIG_FB_DW74_LOW_POWER
int release_low_power_fb(struct dw74fb *dw74fb)
{
	struct dw74fb *dw74fb = info->par;
	struct fb_info *info = dw74fb->dwfb_info_low_power.info;

	dma_free_coherent(NULL, (int)info->screen_size, info->screen_base, (dma_addr_t)info->fix.smem_start);
	fb_dealloc_cmap(&info->cmap);
	framebuffer_release(info);

	return 0;
}

int configure_low_power_fb(struct dw74fb *dw74fb)
{
	struct fb_info *info;
	int ret = 0;

	info = framebuffer_alloc(sizeof(u32) * 256, NULL);
	if (!info)
		goto err_free_dw74fb_low_power;

	info->pseudo_palette = pseudo_palette_low_power;
	info->par = dw74fb;
	info->flags = FBINFO_FLAG_DEFAULT;

	strcpy(info->fix.id, DW74FB_NAME_LOW_POWER);
	info->fix.type = FB_TYPE_PACKED_PIXELS;
	info->fix.visual = FB_VISUAL_MONO10;
	info->fix.accel = FB_ACCEL_NONE;
	info->fix.line_length = panel->xres / 4 ;
	info->var.xres = panel->xres;
	info->var.yres = panel->yres;
	info->var.xres_virtual = panel->xres;
	info->var.yres_virtual = panel->yres;
	info->var.xoffset = 0;
	info->var.yoffset = 0;
	info->var.bits_per_pixel = 2;
	info->var.red.offset = 0;
	info->var.red.length = 0;
	info->var.green.offset = 0;
	info->var.green.length = 0;
	info->var.blue.offset = 0;
	info->var.blue.length = 0;
	info->var.transp.offset = 0;
	info->var.transp.length = 0;
	info->var.vmode = FB_VMODE_NONINTERLACED;
	info->var.height = 0;
	info->var.width = 0;

	#ifdef CONFIG_FB_DW74_DOUBLEBUF
	info->fix.ypanstep = 1;
	info->var.yres_virtual = panel->yres * 2;
	#endif

	info->screen_size = info->fix.line_length * info->var.yres_virtual;
	info->fix.smem_len = info->screen_size;
	info->fbops = &dw74fb_ops;

	/* location = {0,0,0,0} means "not set" */
	dw74fb->dwfb_info_low_power.location.x_start = 0;
	dw74fb->dwfb_info_low_power.location.x_end = 0;
	dw74fb->dwfb_info_low_power.location.y_start = 0;
	dw74fb->dwfb_info_low_power.location.y_end = 0;

	/*Set the backgrount of the display*/
	dw74fb->dwfb_info_low_power.background_color = 0;

	ret = fb_alloc_cmap(&info->cmap, 256, 0);
	if (ret < 0)
		goto err_free_info_low_power;

	/* attach info to driver data */
	dw74fb->dwfb_info_low_power.info = info;

	dw74fb->dwfb_info_low_power.vidmem = dma_alloc_coherent(NULL, info->screen_size,
										(dma_addr_t *)&info->fix.smem_start,
										GFP_DMA);

	if (!dw74fb->dwfb_info_low_power.vidmem) {
		//dev_err(dev, "could not allocate framebuffer (%lu bytes)\n",
		//			 info->screen_size);
		goto err_free_cmap_low_power;
	}

	info->screen_base = dw74fb->dwfb_info_low_power.vidmem;
	memset(info->screen_base, 0x00, info->screen_size);

	/* register framebuffer in userspace */
	ret = register_framebuffer(info);
	if (ret < 0)
		goto err_free_allocate_low_power;

	return 0;

err_free_allocate_low_power:
	dma_free_coherent(NULL, (int)info->screen_size, info->screen_base, (dma_addr_t)info->fix.smem_start);
err_free_cmap_low_power:
	fb_dealloc_cmap(&info->cmap);
err_free_info_low_power:
	framebuffer_release(info);
err_free_dw74fb_low_power:
	return -1;
}
#endif

#ifdef CONFIG_FB_DMW96_CURSOR
const char* lcdc_cursor_name = "dmw96_cursor";
static int configure_cursor_fb(struct dw74fb *dw74fb)
{
	struct fb_info *info;
	int ret = -ENOMEM;

	info = framebuffer_alloc(0 , NULL);
	if (!info)
		return -ENOMEM;

	info->par = dw74fb;
	info->var.nonstd = 0;

	info->screen_base = NULL;
	info->fbops = &dw74fb_ops;
	info->flags = FBINFO_FLAG_DEFAULT;

	strcpy(info->fix.id, lcdc_cursor_name);
	info->fix.type = FB_TYPE_PACKED_PIXELS;
	info->fix.visual = FB_VISUAL_TRUECOLOR;
	info->fix.accel = FB_ACCEL_NONE;
	info->fix.line_length = 0;
	info->fix.smem_start = 0;
	info->fix.smem_len = 0;
	info->screen_size = info->fix.smem_len;

	ret = fb_alloc_cmap(&info->cmap, CURSOR_PALETTE_SIZE , 0);
	if (ret < 0)
		goto err_free_info_cursor;

	/* register framebuffer in userspace */
	ret = register_framebuffer(info);
	if (ret < 0)
		goto err_free_cmap_cursor;

	/* attach info to driver data */
	dw74fb->dwfb_info_cursor.info = info;

	printk(KERN_INFO "fb%d: %s frame buffer device, %dx%d, %d bpp, %luk\n",
	       info->node,
	       info->fix.id,
	       info->var.xres,
	       info->var.yres,
	       info->var.bits_per_pixel,
	       info->screen_size >> 10);

	return 0;

err_free_cmap_cursor:
	fb_dealloc_cmap(&info->cmap);
err_free_info_cursor:
	framebuffer_release(info);

	return ret;
}
#endif /*CONFIG_FB_DMW96_CURSOR*/

static int dw74fb_backlight_update(struct backlight_device *bl)
{
	struct dw74fb* dw74fb = dev_get_drvdata(&bl->dev);
	struct backlight_properties *props = &bl->props;
	unsigned int brightness = props->brightness;

	dev_dbg(&bl->dev, "update: brightness = %d, reg_value = 0x%lx\n", brightness,
	                  dw74fb_readl(dw74fb , LCDC_REG_RBCR));

	if ((props->power != FB_BLANK_UNBLANK) ||
	    (props->state & BL_CORE_SUSPENDED))
		brightness = 0;

	dw74fb_writel(dw74fb, LCDC_REG_RBCR, brightness);
	dw74fb_writel(dw74fb, LCDC_REG_GBCR, brightness);
	dw74fb_writel(dw74fb, LCDC_REG_BBCR, brightness);
	dw74fb_hw_param_update(dw74fb);

	return 0;
}

static int dw74fb_backlight_get_brightness(struct backlight_device *bl)
{
	struct backlight_properties *props = &bl->props;
	return props->brightness;
}

static struct backlight_ops dw74fb_backlight_ops = {
	.get_brightness = dw74fb_backlight_get_brightness,
	.update_status  = dw74fb_backlight_update,
};

static void dw74fb_backlight_init(struct dw74fb *dw74fb)
{
	struct backlight_properties props;
	struct backlight_device *bl;
	struct device *parent = &dw74fb->pdev->dev;

	memset(&props, 0, sizeof(props));
	props.type = BACKLIGHT_RAW;
	props.max_brightness = 255;

	bl = backlight_device_register("dw74fb-backlight", parent, dw74fb,
	                               &dw74fb_backlight_ops, &props);
	if (IS_ERR(bl)) {
		dev_err(parent, "could not register backlight\n");
		return;
	}

	bl->props.brightness = 128;
	bl->props.max_brightness = 255;
	bl->props.power = FB_BLANK_UNBLANK;

	dw74fb->backlight = bl;
	dw74fb_backlight_update(bl);
}

static int
dw74fb_hdmi_current_read_proc(char *page, char **start, off_t off, int count,
                              int *eof, void *data)
{
	char *out = page;
	int len = 0;
	int size = count;

	out += snprintf(out, size, "%d\n", lcdc_mode);
	len = out - page - off;
	if (len >= count)
		return -EFAULT;
	*out = '\0';
	*start = page + off;

	return len;
}

static int
dw74fb_hdmi_modes_read_proc(char *page, char **start, off_t off, int count,
                            int *eof, void *data)
{
	char *out = page;
	int len = 0, i;
	int size = count;

	out += snprintf(out, size, "%s=%d\n", "lcd", DWFB_MODE_LCD);
	for (i = DWFB_MODE_HDMI; i <= DWFB_MODE_MAX; i++) {
		if (dw74fb_modedb[i - DWFB_MODE_HDMI].flag & FB_MODE_IS_ANDROID)
			continue;
		out += snprintf(out, size, "%s=%d\n",
		                dw74fb_modedb[i - DWFB_MODE_HDMI].name, i);
		len = out - page - off;
		if (len >= count)
			return -EFAULT;
	}
	*out = '\0';
	*start = page + off;

	return len;
}

static int
dw74fb_hdmi_android_modes_read_proc(char *page, char **start, off_t off,
                                    int count, int *eof, void *data)
{
	char *out = page;
	int len = 0, i;
	int size = count;

	for (i = DWFB_MODE_HDMI; i <= DWFB_MODE_MAX; i++) {
		if (!(dw74fb_modedb[i - DWFB_MODE_HDMI].flag & FB_MODE_IS_ANDROID))
			continue;
		out += snprintf(out, size, "%s=%d\n",
		                dw74fb_modedb[i - DWFB_MODE_HDMI].name, i);
		len = out - page - off;
		if (len >= count)
			return -EFAULT;
	}
	*out = '\0';
	*start = page + off;

	return len;
}

static int
dw74fb_hdmi_create_procfs(struct dw74fb *fb)
{
	fb->hdmi_root = proc_mkdir("hdmi", 0);
	if (!fb->hdmi_root)
		return -EPERM;

	fb->hdmi_entry_modes = create_proc_entry("modes", S_IFREG|S_IRUGO|S_IRUGO|S_IROTH, fb->hdmi_root);
	if (!fb->hdmi_entry_modes) {
		remove_proc_entry(fb->hdmi_root->name, NULL);
		return -EPERM;
	}

	fb->hdmi_entry_modes->data = (void *)NULL;
	fb->hdmi_entry_modes->read_proc = dw74fb_hdmi_modes_read_proc;

	fb->hdmi_entry_android_modes = create_proc_entry("android_modes", S_IFREG|S_IRUGO|S_IRUGO|S_IROTH, fb->hdmi_root);
	if (!fb->hdmi_entry_android_modes) {
		remove_proc_entry(fb->hdmi_entry_modes->name, fb->hdmi_root);
		remove_proc_entry(fb->hdmi_root->name, NULL);
		return -EPERM;
	}

	fb->hdmi_entry_android_modes->data = (void *)NULL;
	fb->hdmi_entry_android_modes->read_proc = dw74fb_hdmi_android_modes_read_proc;

	fb->hdmi_entry_current = create_proc_entry("current_mode", S_IFREG|S_IRUGO|S_IRUGO|S_IROTH, fb->hdmi_root);
	if (!fb->hdmi_entry_current) {
		remove_proc_entry(fb->hdmi_entry_android_modes->name, fb->hdmi_root);
		remove_proc_entry(fb->hdmi_entry_modes->name, fb->hdmi_root);
		remove_proc_entry(fb->hdmi_root->name, NULL);
		return -EPERM;
	}

	fb->hdmi_entry_current->data = (void *)NULL;
	fb->hdmi_entry_current->read_proc = dw74fb_hdmi_current_read_proc;

	return 0;
}

static int
dw74fb_hdmi_remove_procfs(struct dw74fb *fb)
{
	if (fb->hdmi_entry_android_modes)
		remove_proc_entry(fb->hdmi_entry_android_modes->name, fb->hdmi_root);
	if (fb->hdmi_entry_modes)
		remove_proc_entry(fb->hdmi_entry_modes->name, fb->hdmi_root);
	if (fb->hdmi_entry_current)
		remove_proc_entry(fb->hdmi_entry_current->name, fb->hdmi_root);

	if (fb->hdmi_root)
		remove_proc_entry(fb->hdmi_root->name, NULL);

	return 0;
}

static void dw74fb_suspend(struct dw74fb *dw74fb)
{
	if (panel->suspend)
		panel->suspend(panel);

	dw74fb_writel(dw74fb, LCDC_REG_DISPCR, 0);
	dw74fb_hw_param_update_sync(dw74fb);
	dw74fb_hw_param_update_sync(dw74fb);

	disable_irq(dw74fb->irq);

	clk_disable(dw74fb->clk);
}

static void dw74fb_resume(struct dw74fb *dw74fb)
{
	clk_enable(dw74fb->clk);

	enable_irq(dw74fb->irq);

	dw74fb_writel(dw74fb, LCDC_REG_DISPCR, 1);
	dw74fb_hw_param_update_sync(dw74fb);

	/* Update backlight after enabling clocks so that changes done to the
	 * brightness level before or during resume are reflected in the LCDC
	 * registers. */
	if (dw74fb->backlight)
		dw74fb_backlight_update(dw74fb->backlight);

	if (panel->resume)
		panel->resume(panel);
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void dw74fb_early_suspend(struct early_suspend *es)
{
	struct dw74fb *dw74fb = container_of(es, struct dw74fb, early_suspend);
	dw74fb_suspend(dw74fb);
}

static void dw74fb_late_resume(struct early_suspend *es)
{
	struct dw74fb *dw74fb = container_of(es, struct dw74fb, early_suspend);
	dw74fb_resume(dw74fb);
}
#endif

void hx8352_reset(void);

static int __devinit dw74fb_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct dw74fb *dw74fb;
	struct fb_info *info;
	struct resource *res;
	int ret;

	printk("%s()\n", __FUNCTION__);

	if (!dev->platform_data)
		return -EINVAL;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "need a memory resource\n");
		return -ENXIO;
	}

	if (!request_mem_region(res->start, res->end - res->start,
	                        "dw74fb regs")) {
		dev_err(dev, "failed to request memory region\n");
		return -EBUSY;
	}

	/* init driver data */
	ret = -ENOMEM;
	dw74fb = kzalloc(sizeof(*dw74fb), GFP_KERNEL);
	if (!dw74fb)
		goto err_rel_mem;

	sema_init(&dw74fb->parameter_update_sync, 0);
	panel = dev->platform_data;
	dw74fb->panel = panel;

	dw74fb->irq = platform_get_irq(pdev, 0);
	if (dw74fb->irq < 0) {
		dev_err(&pdev->dev, "need an irq resource\n");
		ret = dw74fb->irq;
		goto err_free_dw74fb;
	}

	dw74fb->clk = clk_get(&pdev->dev, "lcdc");
	if (IS_ERR(dw74fb->clk)) {
		dev_err(&pdev->dev, "could not get clock\n");
		ret = PTR_ERR(dw74fb->clk);
		goto err_free_dw74fb;
	}

	if (panel->clock_rate)
		clk_set_rate(dw74fb->clk, clk_round_rate(dw74fb->clk, panel->clock_rate));
	clk_enable(dw74fb->clk);

	/* init frame buffer */
	info = framebuffer_alloc(sizeof(u32) * 256, dev);
	if (!info)
		goto err_free_dw74fb;

	info->pseudo_palette = pseudo_palette;
	info->par = dw74fb;
	info->flags = FBINFO_FLAG_DEFAULT;

	strcpy(info->fix.id, DW74FB_NAME);
	info->fix.type = FB_TYPE_PACKED_PIXELS;
	info->fix.visual = FB_VISUAL_TRUECOLOR;
	info->fix.accel = FB_ACCEL_NONE;
	info->fix.smem_start = 0xe0000000;
	info->fix.smem_len = 0x04000000;

	/* in case of small resolution emulation - allocate minimal amount of memory */
	if (panel->xres_full || panel->yres_full)
		mem = 0;

	if (!panel->xres_full)
		panel->xres_full = panel->xres;
	if (!panel->yres_full)
		panel->yres_full = panel->yres;

	info->var.xres = panel->xres;
	info->var.yres = panel->yres;
	info->var.xres_virtual = panel->xres;
	info->var.yres_virtual = panel->yres;
	info->var.xoffset = 0;
	info->var.yoffset = 0;

	/*
	 * if panel supports only 16bits, then frame buffer will hold rgb565
	 * if panel supports rgb666 or rgb888, then frame buffer will hold rgb888
	 * (in case of panel which supports rgb666, the LCDC will drop the 2
	 * least significat bits of each color component))
	 */
	if (bpp_to_info(&info->var, panel->bits_per_pixel) < 0) {
		printk("dw74fb: panel->bits_per_pixel is not supported!");
		return -EBUSY;
	}

	info->fix.line_length = calc_line_stride(panel->xres * info->var.bits_per_pixel / 8);
	
	info->var.transp.offset = 0;
	info->var.transp.length = 0;
	info->var.vmode = FB_VMODE_NONINTERLACED;
	info->var.width = panel->width_mm;
	info->var.height = panel->height_mm;

#ifdef CONFIG_FB_DW74_DOUBLEBUF
	info->fix.ypanstep = 1;
	info->var.yres_virtual = panel->yres * 2;
#endif

	info->screen_size = info->fix.line_length * info->var.yres_virtual;
	info->fix.smem_len = info->screen_size;
	info->fbops = &dw74fb_ops;

	ret = fb_alloc_cmap(&info->cmap, 256, 0);
	if (ret < 0)
		goto err_free_info;

	/* location = {0,0,0,0} means "not set" */
	dw74fb->dwfb_info.location.x_start = 0;
	dw74fb->dwfb_info.location.x_end = 0;
	dw74fb->dwfb_info.location.y_start = 0;
	dw74fb->dwfb_info.location.y_end = 0;

	/*Set the backgrount of the display*/
	dw74fb->dwfb_info.background_color = 0;

	/* attach info to driver data */
	dw74fb->dwfb_info.info = info;

	/* setup memory */
#ifndef CONFIG_MACH_VERSATILE_BROADTILE
	if (!mem)
		mem = info->screen_size;
	else
		mem *= 1024;

	dw74fb->dwfb_info.vidmem = dma_alloc_coherent(&pdev->dev, mem,
	                                    (dma_addr_t *)&info->fix.smem_start,
	                                    GFP_DMA);
#else
	if (!request_mem_region(info->fix.smem_start, info->screen_size ,dw74fb_name)) {
		printk("cannot request mem region for %s \n",dw74fb_name);
		return -EBUSY;
	}

	dw74fb->dwfb_info.vidmem  = ioremap_nocache(info->fix.smem_start, info->screen_size );
	if (!dw74fb->dwfb_info.vidmem) {
		printk("cannot map sdram for %s\n",dw74fb_name);
		return -EBUSY;
	}
#endif

	if (!dw74fb->dwfb_info.vidmem) {
		dev_err(dev, "could not allocate framebuffer (%lu bytes)\n",
		             info->screen_size);
		goto err_free_cmap;
	}

	info->fix.smem_len = info->screen_size;
	info->screen_base = dw74fb->dwfb_info.vidmem;

	dw74fb->pdev = pdev;

	/* init hardware */
	ret = -ENOMEM;
	dw74fb->regs = ioremap(res->start, res->end - res->start);
	if (!dw74fb->regs) {
		dev_err(dev, "unable to map registers\n");
		ret = -ENOMEM;
		goto err_free_cmap;
	}

	/*This ugly global pointer is used to access dma address from OSDM driver*/
	g_dw74fb = dw74fb;

	/* if cputype LCD, register and associate */
	if (panel->cputype_desc) {
		if (platform_device_register(panel->cputype_desc->pdev)) {
			goto err_cputype_pdev_register;
		}

		panel->cputype_desc->associate(dw74fb);
	}

	if (!dw74fb_hw_try_resume(dw74fb) ) {
		memset(info->screen_base, 0x00, info->screen_size);

		dw74fb_hw_init(dw74fb);
		dw74fb_hw_enable(dw74fb);
	}

	/* attach driver data to the device */
	dev_set_drvdata(dev, dw74fb);

	/* disable update int, and clear all pending interrupts (may exist from u-boot) */
	dw74fb_writel(dw74fb, LCDC_REG_INTER, dw74fb_readl(dw74fb, LCDC_REG_INTER) | 0x1);
	dw74fb_writel(dw74fb, LCDC_REG_INTSR, dw74fb_readl(dw74fb, LCDC_REG_INTSR));

	ret = request_irq(dw74fb->irq, dw74fb_irq, 0, dw74fb_name, dw74fb);
	if (ret < 0) {
		dev_err(dev, "request irq %d failed\n", dw74fb->irq);
		goto err_free_cmap;
	}

	/* init panel after the LCDC has been setup */
	if (panel->init)
		panel->init(panel);

	/* register framebuffer in userspace */
	ret = register_framebuffer(info);
	if (ret < 0)
		goto err_free_irq;

	printk(KERN_INFO "fb%d: %s frame buffer, %s (%dx%d, %d bpp, %luk)\n",
	       info->node,
	       info->fix.id,
	       panel->name,
	       info->var.xres,
	       info->var.yres,
	       info->var.bits_per_pixel,
	       info->screen_size >> 10);

#ifdef CONFIG_HAS_EARLYSUSPEND
	dw74fb->early_suspend.suspend = dw74fb_early_suspend;
	dw74fb->early_suspend.resume = dw74fb_late_resume;
	dw74fb->early_suspend.level = EARLY_SUSPEND_LEVEL_DISABLE_FB;

	register_early_suspend(&dw74fb->early_suspend);
#endif

	/*We need that semaphore in case we are in CPU type panel and we
	want to make sure that there will not be any updates from the external
	controller while the DMA of the external controller was NOT finished*/
	if (panel->is_cpu_type) {
		setup_timer( &dw74fb->cputype_vsync_timer, &cputype_vsync_callback, (unsigned long)dw74fb );
		ret = mod_timer( &dw74fb->cputype_vsync_timer, jiffies + msecs_to_jiffies(CPUTYPE_VSYNC_TIME) );
		if (ret) {
			printk("Error setting cputype timer\n");
			goto err_free_irq;
		}

		sema_init(&(dw74fb->lcdc_hw_busy), 1);
	}

#ifdef CONFIG_FB_DW74_LOW_POWER
	ret = configure_low_power_fb(dw74fb);
	if (ret < 0)
		goto err_free_irq;
#endif

#ifdef CONFIG_FB_DMW96_CURSOR
	ret = configure_cursor_fb(dw74fb);
	#ifdef CONFIG_FB_DW74_LOW_POWER
		if (ret < 0)
			release_low_power_fb(dw74fb);
	#endif
	if (ret < 0)
		goto err_free_irq;
#endif

	dw74fb_hdmi_create_procfs(dw74fb);

	if (panel->no_dimmable_backlight)
		dw74fb_backlight_init(dw74fb);

	return 0;

err_free_irq:
	free_irq(dw74fb->irq, dw74fb);

err_cputype_pdev_register:
	if (dw74fb->panel->cputype_desc) {
		platform_device_unregister(dw74fb->panel->cputype_desc->pdev);
	}

err_free_cmap:
	fb_dealloc_cmap(&info->cmap);

err_free_info:
	framebuffer_release(dw74fb->dwfb_info.info);

err_free_dw74fb:
	kfree(dw74fb);

err_rel_mem:
	release_mem_region(res->start, res->end - res->start);
	return ret;
}

static int __exit dw74fb_remove(struct platform_device *pdev)
{
	struct dw74fb *dw74fb = dev_get_drvdata(&pdev->dev);

	dw74fb_hdmi_remove_procfs(dw74fb);

	g_dw74fb = NULL;

	/* TODO: free everything, including backlight */
	return 0;
}

#if defined(CONFIG_PM) && !defined(CONFIG_HAS_EARLYSUSPEND)
static int dw74fb_dev_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct dw74fb *dw74fb = dev_get_drvdata(&pdev->dev);
	dw74fb_suspend(dw74fb);
	return 0;
}

static int dw74fb_dev_resume(struct platform_device *pdev)
{
	struct dw74fb *dw74fb = dev_get_drvdata(&pdev->dev);
	dw74fb_resume(dw74fb);
	return 0;
}
#else
#define dw74fb_dev_suspend NULL
#define dw74fb_dev_resume NULL
#endif

static struct platform_driver dw74fb_driver = {
	.remove = __exit_p(dw74fb_remove),
	.suspend = dw74fb_dev_suspend,
	.resume = dw74fb_dev_resume,
	.driver = {
		.name = dw74fb_name,
		.owner = THIS_MODULE,
	},
};

static int __init dw74fb_init(void)
{
	return platform_driver_probe(&dw74fb_driver, dw74fb_probe);
}

static void __exit dw74fb_exit(void)
{
	platform_driver_unregister(&dw74fb_driver);
}


module_init(dw74fb_init);
module_exit(dw74fb_exit);
