//**************************************************************************************************
//                                          SysScan.cpp                                            *
//                                         -------------                                           *
// Started     : 2020-09-27                                                                        *
// Last Update : 2023-05-27                                                                        *
// Copyright   : (C) 2020-2023 MSWaters                                                            *
//**************************************************************************************************

//**************************************************************************************************
//                                                                                                 *
//      This program is free software; you can redistribute it and/or modify it under the          *
//      terms of the GNU General Public License as published by the Free Software Foundation;      *
//      either version 3 of the License, or (at your option) any later version.                    *
//                                                                                                 *
//**************************************************************************************************

#include "SysScan.hpp"

//**************************************************************************************************
// Allocate storage for static data members.

wxString       SysScan::m_osPathInstall;
wxString       SysScan::m_osPathDoc;
ArrayFileName  SysScan::m_oaBinFiles;

//**************************************************************************************************
// Constructor.

SysScan::SysScan( void )
// The wxTheApp object exists but the variables argv and argc are empty

{
  InitPathVars( );
}

//**************************************************************************************************
// Destructor.

SysScan::~SysScan( )
{
}

//**************************************************************************************************
// Determine the host application installation and documentation paths.
//
// Possible installation paths are :
//   /usr/share/gspiceui/
//   /usr/local/share/gspiceui/
//   /home/user/gspiceui/
//
// Possible documentation paths are :
//   /usr/share/doc/gspiceui/
//   /usr/local/share/doc/gspiceui/
//   /home/user/gspiceui/doc/
//
// Possible binary paths are :
//   /usr/bin/
//   /usr/local/bin/
//   /home/user/gspiceui/bin/

void  SysScan::InitPathVars( void )
{
  wxPathList  opl1;
  wxFileName  ofn1;
  wxString    os1;

  // This function only needs to be called once to initialize the static class attributes
  if( !m_osPathInstall.IsEmpty( ) && !m_osPathDoc.IsEmpty( ) ) return;

  // Get the command used to envoke this process eg. "./gspiceui"
  ofn1 = wxTheApp->argv[ 0 ];

  // Get the path to the host application binary eg. "/usr/bin/"
  if( ! ofn1.GetPath( ).IsEmpty( ) )
    ofn1.Normalize( wxPATH_NORM_DOTS | wxPATH_NORM_TILDE | wxPATH_NORM_ABSOLUTE |
                    wxPATH_NORM_LONG | wxPATH_NORM_SHORTCUT );  // Expand "..", "~', etc.
  if( ! ofn1.FileExists( ) )
  { // Search environment variable PATH for the first occurrence of the host application name
    opl1.AddEnvList( "PATH" );
    os1 = opl1.FindAbsoluteValidPath( ofn1.GetFullName( ) );
    ofn1 = os1;
  }

  // Test and trim the path
  if( ofn1.GetPath( ).AfterLast( '/' ) != "bin" )
  {
    // If debug mode is enabled send the process command line to the console
    if( g_bDebug )
      std::cout << "DEBUG : SysScan::InitHostPaths( ) : Couldn't determine host application paths\n";
    return;
  }
  os1 = ofn1.GetPath( ).BeforeLast( '/' );
  ofn1.SetPath( os1 );

  // Derive the host application paths
  if( ofn1.GetPath( ).AfterLast( '/' ) != ofn1.GetName( ) )
  {
    m_osPathInstall = ofn1.GetPath( ) + "/share/"     + ofn1.GetFullName( );
    m_osPathDoc     = ofn1.GetPath( ) + "/share/doc/" + ofn1.GetFullName( );
  }
  else
  {
    m_osPathInstall = ofn1.GetPath( );
    m_osPathDoc     = ofn1.GetPath( ) + "/doc";
  }
}

//**************************************************************************************************
// Search for a binary file item in the list of binary files and return it's index.
//
// Argument List :
//   rosFileBin - The binary name to be found
//
// Return Values :
//   Success - the binary file index within the binary file list ( >=0 )
//   Failure - wxNOT_FOUND

int  SysScan::iSearchBinList( const wxString & rosFileBin )
{
  size_t  sz1;

  // Search the class binary file list for the requested binary file name
  if( ! m_oaBinFiles.IsEmpty( ) )
  {
    for( sz1=0; sz1<m_oaBinFiles.GetCount( ); sz1++ )
      if( m_oaBinFiles.Item( sz1 ).GetFullName( ) == rosFileBin )
        return( (int) sz1 );
  }

  return( wxNOT_FOUND );
}

//**************************************************************************************************
// Find a binary using the user's PATH environment variable given the binary name.
//
// Note : This function will fail for (bash) shell builtin commands like "cd" since these commands
//        do not have a associated binary file. Builtin commands can be listed by entering "help" at
//        bash command prompt, or for a simple list of command names enter "compgen -b".
//
// Argument List :
//   ofnBin - The binary name without the path
//
// Return Values :
//   true  - Success (add the path to the function argument rofnFileBin)
//   false - Failure (function argument rofnFileBin is unchanged)

bool  SysScan::bFindBinFile( wxFileName & rofnFileBin )
{
  wxPathList  opl1;
  wxFileName  ofn1;

  // Initial checks
  if( ! rofnFileBin.IsOk( ) ) return( false );

  // Search environment variable PATH for the first occurrence of the binary
  opl1.AddEnvList( "PATH" );
  ofn1 = opl1.FindAbsoluteValidPath( rofnFileBin.GetFullName( ) );

  // Check whether the binary was successfully found
  if( !ofn1.IsOk( ) || !ofn1.FileExists( ) )
  {
    // If debug mode is enabled send the error message to the console
    if( g_bDebug ) std::cout << "DEBUG : SysScan::bFindBinFile( ) : Didn't find binary "
                             << rofnFileBin.GetFullPath( ) << '\n';
    return( false );
  }
  else
  {
    // If debug mode is enabled send the error message to the console
    if( g_bDebug ) std::cout << "DEBUG : SysScan::bFindBinFile( ) : Found "
                             << ofn1.GetFullPath( ) << '\n';
  }

  rofnFileBin = ofn1;

  return( true );
}

//**************************************************************************************************
// Attempt to add a binary file to the list of binary files.
//
// Argument List :
//   rosFileBin - The binary name to be added.
//
// Return Values :
//   Success - true
//   Failure - false

bool  SysScan::bAddBinFile( const wxString & rosFileBin )
{
  wxFileName  ofn1;

  // Check that the binary file name is valid
  ofn1.SetFullName( rosFileBin );
  if( ! ofn1.IsOk( ) )                              return( false );

  // Is the binary file already in the list?
  if( iSearchBinList( rosFileBin ) != wxNOT_FOUND ) return( true );

  // Binary file isn't in the binary file list so search for it on the system and add it to the list
  ofn1.SetFullName( rosFileBin );
  bFindBinFile( ofn1 );      // Add the path to the binary if it is currently installed
  m_oaBinFiles.Add( ofn1 );  // Add the binary file name to the list

  return( true );
}

//**************************************************************************************************
// Check if a binary is currently installed on the system.
//
// This class maintains a list of binaries which have been searched for previously. If a binary is
// installed, the list element contains the binary name and full path. If a binary isn't installed
// the associated list element path is left empty. The class binary list is first searched, if the
// binary isn't present the system is then searched and a new item is added to the binary list.
//
// Argument List :
//   rosFileBin - The binary name to search for.
//
// Return Values :
//   true  - Success
//   false - Failure

bool  SysScan::bIsInstalled( const wxString & rosFileBin )
{
  int  i1;

  // Search the class binary file list for the requested binary file name
  if( iSearchBinList( rosFileBin ) == wxNOT_FOUND )
    if( ! bAddBinFile( rosFileBin ) ) return( false );

  // The binary should be in the binary file list so search for it again
  i1 = iSearchBinList( rosFileBin );
  if( i1 == wxNOT_FOUND )             return( false );  // This shouldn't happen, but you never know

  return( ! m_oaBinFiles.Item( (size_t) i1 ).GetPath( ).IsEmpty( ) );
}

//**************************************************************************************************
// Get the wxFileName object from the class binary list attribute and return it in the function
// argument.
//
// If the class binary list attribute contains the binary it's is returned, if not the system is
// searched and a new element is added to the list and this is returned.
//
// Argument List :
//   rosFileBin - The binary name to search for and return.
//
// Return Values :
//   true  - Success
//   false - Failure

bool  SysScan::bGetBinFile( wxFileName & rofnFileBin )
{
  int  i1;

  // Check if the binary is installed
  if( ! bIsInstalled( rofnFileBin.GetFullName( ) ) ) return( false );

  // Search the class binary file list for the requested binary file name
  i1 = iSearchBinList( rofnFileBin.GetFullName( ) );
  if( i1 == wxNOT_FOUND )                            return( false );  // This shouldn't happen
  rofnFileBin = m_oaBinFiles.Item( (size_t) i1 );

  return( true );
}

//**************************************************************************************************
// Print the object attributes.
//
// Argument List :
//   rosPrefix - A prefix to every line displayed (usually just spaces)

void  SysScan::Print( const wxString & rosPrefix )
{
  size_t  sz1;

  for( sz1=0; sz1<m_oaBinFiles.GetCount( ); sz1++ )
    std::cout << rosPrefix.mb_str( ) << "m_oaBinFiles[ " << sz1 << " ] : "
              << m_oaBinFiles.Item( sz1 ).GetFullPath( ) << '\n';
}

//**************************************************************************************************
//                                          Test Utility                                           *
//**************************************************************************************************

#ifdef TEST_SYSSCAN

using  namespace  std;

// wxWidgets Includes

#include <wx/cmdline.h>

// Global variable declarations

bool  g_bDebug=false;

// Function prototypes

void  Usage( char * psAppName );

// Declare a structure defining the command line syntax
static  const  wxCmdLineEntryDesc  tCmdLnDesc[] =
{
  { wxCMD_LINE_SWITCH, "h", "", "", wxCMD_LINE_VAL_NONE  , wxCMD_LINE_OPTION_HELP },
  { wxCMD_LINE_SWITCH, "d", "", "", wxCMD_LINE_VAL_NONE                           },
  { wxCMD_LINE_NONE }
};

//**************************************************************************************************

int  main( int argc, char * argv[ ] )
{
  // This function is used in wxBase only and only if an wxApp object isn't created at all. In this
  // case wxInitialize( ) must be called in main( ) before calling any other wxWidgets functions.
  if( ! wxInitialize( argc, argv ) ) exit( EXIT_FAILURE );

  wxCmdLineParser  oCmdLnPsr;
  wxString         os1;

  // Setup the command line parser object and process the command line
  oCmdLnPsr.SetDesc( tCmdLnDesc );
  oCmdLnPsr.SetCmdLine( argc, argv );
  if( oCmdLnPsr.Parse( false ) != 0 ) { Usage( argv[0] ); exit( EXIT_FAILURE ); }
  if( oCmdLnPsr.Found( "h" ) )        { Usage( argv[0] ); exit( EXIT_SUCCESS ); }
  if( oCmdLnPsr.Found( "d" ) )        g_bDebug = true;

  // Create the SysScan object
  SysScan        oSysScan;
  wxArrayString  oas1;
  size_t         sz1;
  bool           bRtn;

  // Display the utility banner
  cout << "\n    SysScan Test Utility"
       << "\n  Version 1.11 (2020-11-01)\n\n";

  // Print the application installation and documentation paths
  cout << "Host application paths :\n"
       << "  Installation  : " << oSysScan.rosGetPathInstall( ) << '\n'
       << "  Documentation : " << oSysScan.rosGetPathDoc( )     << "\n\n";

  // Add some binary names to the SysScan object
  cout << "Add some binary names to the SysScan object :\n";
  oas1.Add( "evince" );
  oas1.Add( "gwave2" );
  oas1.Add( "notapp" );
  for( sz1=0; sz1<oas1.GetCount( ); sz1++ )
  {
    bRtn = oSysScan.bAddBinFile( oas1.Item( sz1 ) );
    cout << "  Add binary \"" << oas1.Item( sz1 ).mb_str( ) << "\" : "
         << (bRtn ? "Success" : "Failure") << '\n';
  }
  std::cout << '\n';

  // Add some extra binary names and see what is and isn't installed
  cout << "Add some extra binary names and see what is and isn't installed :\n";
  oas1.Add( "gnucap" );
  oas1.Add( "nofile" );
  for( sz1=0; sz1<oas1.GetCount( ); sz1++ )
  {
    bRtn = oSysScan.bIsInstalled( oas1.Item( sz1 ) );
    cout << "  Is \"" << oas1.Item( sz1 ).mb_str( ) << "\" installed : "
         << (bRtn ? "Yes" : "No") << '\n';
  }
  std::cout << '\n';

  // Print the object contents
  cout << "Print the SysScan object contents :\n";
  oSysScan.Print( "  oSysScan::" );
  std::cout << '\n';

  // Clean up; the wxWidgets library can't be used any more. This function must be called once for
  // each previous successful call to wxInitialize().
  wxUninitialize( );

  exit( EXIT_SUCCESS );
}

//**************************************************************************************************

void  Usage( char * psAppName )
{
  cout << "\nUsage   : " << psAppName << " [-OPTIONS]"
       << "\nOptions :"
       << "\n  -h : Print usage (this message)"
       << "\n  -d : Enable debug mode\n\n";
}

//**************************************************************************************************

#endif // TEST_SYSSCAN
