/*
 * Decompiled with CFR 0.152.
 */
package org.jabref.logic.citationstyle;

import de.undercouch.citeproc.helper.CSLUtils;
import java.io.IOException;
import java.io.StringReader;
import java.io.UncheckedIOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystem;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import org.jabref.logic.util.StandardFileType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.CharacterData;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class CitationStyle {
    public static final String DEFAULT = "/ieee.csl";
    private static final Logger LOGGER = LoggerFactory.getLogger(CitationStyle.class);
    private static final Pattern SNAPSHOT_NAME = Pattern.compile(".*styles-1\\.0\\.1-SNAPSHOT\\.jar");
    private static final List<CitationStyle> STYLES = new ArrayList<CitationStyle>();
    private final String filePath;
    private final String title;
    private final String source;

    private CitationStyle(String filename, String title, String source) {
        this.filePath = Objects.requireNonNull(filename);
        this.title = Objects.requireNonNull(title);
        this.source = Objects.requireNonNull(source);
    }

    private static Optional<CitationStyle> createCitationStyleFromSource(String source, String filename) {
        if (filename != null && !filename.isEmpty() && source != null && !source.isEmpty()) {
            try {
                DocumentBuilder db = DocumentBuilderFactory.newInstance().newDocumentBuilder();
                InputSource is = new InputSource();
                is.setCharacterStream(new StringReader(CitationStyle.stripInvalidProlog(source)));
                Document doc = db.parse(is);
                NodeList nodes = doc.getElementsByTagName("info");
                NodeList titleNode = ((Element)nodes.item(0)).getElementsByTagName("title");
                String title = ((CharacterData)titleNode.item(0).getFirstChild()).getData();
                return Optional.of(new CitationStyle(filename, title, source));
            }
            catch (IOException | ParserConfigurationException | SAXException e) {
                LOGGER.error("Error while parsing source", e);
            }
        }
        return Optional.empty();
    }

    private static String stripInvalidProlog(String source) {
        int startIndex = source.indexOf("<");
        if (startIndex > 0) {
            return source.substring(startIndex, source.length());
        }
        return source;
    }

    public static Optional<CitationStyle> createCitationStyleFromFile(String styleFile) {
        if (!CitationStyle.isCitationStyleFile(styleFile)) {
            LOGGER.error("Can only load style files: " + styleFile);
            return Optional.empty();
        }
        try {
            String internalFile = (styleFile.startsWith("/") ? "" : "/") + styleFile;
            URL url = CitationStyle.class.getResource(internalFile);
            String text = url != null ? CSLUtils.readURLToString(url, StandardCharsets.UTF_8.toString()) : new String(Files.readAllBytes(Paths.get(styleFile, new String[0])), StandardCharsets.UTF_8);
            return CitationStyle.createCitationStyleFromSource(text, styleFile);
        }
        catch (NoSuchFileException e) {
            LOGGER.error("Could not find file: " + styleFile, e);
        }
        catch (IOException e) {
            LOGGER.error("Error reading source file", e);
        }
        return Optional.empty();
    }

    public static CitationStyle getDefault() {
        return CitationStyle.createCitationStyleFromFile(DEFAULT).orElse(new CitationStyle("", "Empty", ""));
    }

    public static List<CitationStyle> discoverCitationStyles() {
        if (!STYLES.isEmpty()) {
            return STYLES;
        }
        try {
            Path filePath = Paths.get(CitationStyle.class.getProtectionDomain().getCodeSource().getLocation().toURI());
            String path = filePath.toString();
            if (Files.isDirectory(filePath, new LinkOption[0])) {
                String cp = System.getProperty("java.class.path");
                String[] entries = cp.split(System.getProperty("path.separator"));
                Optional<String> foundStyle = Arrays.stream(entries).filter(entry -> SNAPSHOT_NAME.matcher((CharSequence)entry).matches()).findFirst();
                path = foundStyle.orElse(path);
            }
            try (FileSystem jarFs = FileSystems.newFileSystem(Paths.get(path, new String[0]), null);){
                try (Stream<Path> stylefileStream = Files.find(jarFs.getRootDirectories().iterator().next(), 1, (file, attr) -> file.toString().endsWith("csl"), new FileVisitOption[0]);){
                    for (Path style : stylefileStream.collect(Collectors.toList())) {
                        CitationStyle.createCitationStyleFromFile(style.getFileName().toString()).ifPresent(STYLES::add);
                    }
                }
                catch (UncheckedIOException e) {
                    throw new IOException(e);
                }
            }
            return STYLES;
        }
        catch (IOException | UncheckedIOException | URISyntaxException ex) {
            LOGGER.error("something went wrong while searching available CitationStyles. Are you running directly from source code?", ex);
            return Collections.emptyList();
        }
    }

    public static boolean isCitationStyleFile(String styleFile) {
        return StandardFileType.CITATION_STYLE.getExtensions().stream().anyMatch(styleFile::endsWith);
    }

    public String getTitle() {
        return this.title;
    }

    public String getSource() {
        return this.source;
    }

    public String getFilePath() {
        return this.filePath;
    }

    public String toString() {
        return this.title;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        CitationStyle other = (CitationStyle)o;
        return Objects.equals(this.source, other.source);
    }

    public int hashCode() {
        return Objects.hash(this.source);
    }
}

