/* $Id: minipng.hpp 14342 2008-09-16 13:55:51Z truebrain $ */

#ifndef MINIPNG_HPP
#define MINIPNG_HPP

/* define static_assert to Check size of structs and other compile time Checks */
#define assert_compile(_x) typedef int __static_assert[(_x) ? 1 : -1]

static inline unsigned int EndianFix(unsigned int i)
{
#ifdef __BIG_ENDIAN__
	return i;
#else
	return ((i >> 24) & 0xFF) | ((i >> 8) & 0xFF00) | ((i & 0xFF00) << 8) | ((i & 0xFF) << 24);
#endif
}

struct PNGChunk
{
	/**
	 * This struct is never actually newed, but a malloced buffer containing
	 * all the data of one png chunk is cast to it, so that the chunk can be
	 * accessed and modified.
	 */

	/* chunk: these 2 members (BE), then the data (can be 0), then 4 byte CRC (BE) */
	unsigned int DataSizeRaw;
	unsigned int IDRaw;

	/* Accessors */
	unsigned int GetDataLen()        { return EndianFix(DataSizeRaw); }
	void SetDataLen(unsigned int dl) { DataSizeRaw = EndianFix(dl); }
	unsigned int GetID()             { return EndianFix(IDRaw); }
	void SetID(unsigned int id)      { IDRaw = EndianFix(id); }
	unsigned int GetChunkSize()      { return GetDataLen() + 12; }
	bool CheckCRC()                  { return *GetCRCPtr() == CalcCRC(); }

	/* Internal */
	unsigned int *GetCRCPtr()        { return reinterpret_cast<unsigned int*>(reinterpret_cast<char*>(&IDRaw) + sizeof(unsigned int) + GetDataLen()); }
	unsigned int CalcCRC();
	void RecalcCRC()                 { *GetCRCPtr() = CalcCRC(); }

	/* Factory */
	static PNGChunk *Create(unsigned int datalen);

protected:
	/* Hidden constructor */
	PNGChunk() {}
};

struct PNGTextChunk : PNGChunk
{
	/**
	 * This struct is never actually newed, but a malloced buffer containing
	 * all the data of one png tEXt chunk is cast to it, so that the tEXt chunk
	 * data can be accessed and modified. The factory creates an instance
	 * by allocating the amount the chunk will need, and using the functions
	 * available to fill it with the required text data and CRC.
	 */

	/* For Reading */
	const char *GetKey() { return GetKeyPtr(); }
	size_t GetValueLen() { return GetDataLen() - (1 + strlen(GetKey())); }
	void GetValue(char *buf);

	/* Internal */
	char *GetKeyPtr()    { return reinterpret_cast<char*>(this) + sizeof(PNGChunk); }
	char *GetValuePtr()  { return GetKeyPtr() + strlen(GetKeyPtr()) + 1;	}

	/* Factory */
	static PNGTextChunk *Create(const char *key, const char *value);

private:
	/* Hidden constructor */
	PNGTextChunk() {}
};

template<class T>
class DynArrayPtr
{
private:
	T **data;
	unsigned int max;

public:
	DynArrayPtr() : data(NULL), max(0), count(0), delta(16) {}
	/* ~DynArrayPtr will not free the T*'s fed to it by default */
	~DynArrayPtr() { free(data); }

	/* Count may be read only, delta may not be 0 */
	unsigned int count, delta;

	void Add(T *t)
	{
		if (count >= max) {
			max += delta;
			data = reinterpret_cast<T**>(realloc(data, sizeof(T*) * max));
		}
		data[count++] = t;
	}

	void DeleteAll()
	{
		for (unsigned int q = 0; q < count; q++) free(data[q]);
		free(data);
		data = NULL;
		count = 0;
		max = 0;
	}

	void DeleteNo(unsigned int i)
	{
		free(data[i]);
		data[i] = data[--count];
	}

	T* operator [] (unsigned int i) const { return data[i]; }
};

enum PNGRetVal
{
	PNG_OK = 0,
	PNG_FileNotFound,
	PNG_NotPNG,
	PNG_AccessDenied,
	PNG_WriteError,
};

class PNGFile
{
private:
	char png_header[8];
	PNGChunk *header_chunk;
	DynArrayPtr<PNGChunk> data;
	DynArrayPtr<PNGTextChunk> txtdata;
	PNGChunk *footer_chunk;

	void CleanUp();

public:
	/* Filename is read only */
	char filename[1024];

	PNGFile() : header_chunk(NULL), footer_chunk(NULL) { *filename = '\0'; }
	~PNGFile() { CleanUp(); }

	PNGRetVal Read(const char*);
	int GetPropCount() { return txtdata.count; }
	size_t GetPropValueLen(unsigned int i) { return txtdata[i]->GetValueLen(); }
	const char *GetProp(unsigned int i, char *value_buf);

	void ResetProps() { txtdata.DeleteAll(); }
	void SetProp(const char *key, const char *val);
	PNGRetVal Write();
};

#endif /* MINIPNG_HPP */
